/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springunit.framework;

import org.springframework.test.AbstractDependencyInjectionSpringContextTests;

/**
 * Extends Spring's test framework to support data-driven tests.
 * Data-driven tests separate data values from test logic,
 * keeping data values in external files and test logic in Java code.
 * Every descendent of SpringUnitTest is required by convention
 * to have a bean called <code><i>Classname</i></code> of type
 * SpringUnitContext, where <code><i>Classname</i></code> is
 * the simple name of the subclass of SpringUnitTest.
 * Note that the simple names of subclasses must be unique,
 * that is, they must not be distinguished solely by different
 * package qualifiers.
 * 
 * @author <a href="mailto:ted@velkoff.com">Ted Velkoff</a>
 *
 */
public abstract class SpringUnitTest extends AbstractDependencyInjectionSpringContextTests {

	/**
	 * Default constructor.
	 */
	public SpringUnitTest() {
		this(null);
	}

	/**
	 * Constructor with JUnit name.
	 * Sets AutowireMode to "by name" (overriding the default, which is "by type").
	 * Creates a context container that stores the 
	 * hierarchy of test contexts.
	 * @param fName Name of JUnit test
	 */
	public SpringUnitTest(String fName) {
		super(fName);
		setAutowireMode(AUTOWIRE_BY_NAME);
		this.container = new HierarchicalSpringUnitContext<SpringUnitTest>(getClass());
	}

	/**
	 * Return list of file names that the
	 * Spring test framework uses in order to create beans for testing.
	 * @return Array of string filenames
	 */
	protected final String[] getConfigLocations() {
		return this.container.getConfigLocations();
	}
	
	/**
	 * Search for object identified by <code>key</code>
	 * in the hierarchy of contexts.
	 * @param key Identifier of data value to find
	 * @return Object if found or null
	 * @throws Exception if errors occur when using reflection
	 * to access the SpringUnitContext for any
	 * class in the list
	 */
	public final <T> T getObject(String key) throws Exception {
		return (T)this.container.getObject(key, getName(), this);
	}
	
	/**
	 * Container of the hierarchy of contexts for the test.<br/>
	 */
	private HierarchicalSpringUnitContext<SpringUnitTest> container;
	
}
