/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springunit.framework;

import org.springframework.test.AbstractTransactionalSpringContextTests;

/**
 * Extends Spring's test framework to support transactional data-driven tests.
 * Data-driven tests separate data values from test logic,
 * keeping data values in external files and test logic in Java code.
 * Every descendent of SpringUnitTransactionalTest is required by convention
 * to have a bean called <code><i>Classname</i></code> of type
 * SpringUnitContext, where <code><i>Classname</i></code> is
 * the simple name of the subclass of SpringUnitTransactionalTest.
 * Note that the simple names of subclasses must be unique,
 * that is, they must not be distinguished solely by different
 * package qualifiers.
 * 
 * @author <a href="mailto:ted@velkoff.com">Ted Velkoff</a>
 *
 */
public class SpringUnitTransactionalTest extends AbstractTransactionalSpringContextTests {

	/**
	 * Default constructor.
	 */
	public SpringUnitTransactionalTest() {
		this(null);
	}

	/**
	 * Constructor with JUnit name.
	 * Sets AutowireMode to "by name" (overriding the default, which is "by type").
	 * Creates a hierarchy of contexts.
	 * @param fName Name of JUnit test
	 */
	public SpringUnitTransactionalTest(String name) {
		super(name);
		setAutowireMode(AUTOWIRE_BY_NAME);
		this.contexts = new HierarchicalSpringUnitContext<SpringUnitTransactionalTest>(getClass());
	}

	/**
	 * Creates and populates the Spring application context.
	 * @throws Exception
	 */
	protected void populateApplicationContext() throws Exception {
		this.applicationContext = getContext(contextKey());
		if (isPopulateProtectedVariables()) {
			if (this.managedVariableNames == null) {
				initManagedVariableNames();
			}
			populateProtectedVariables();
		}
		else if (getAutowireMode() != AUTOWIRE_NO) {
			this.applicationContext.getBeanFactory().autowireBeanProperties(
			    this, getAutowireMode(), isDependencyCheck());
		}
	}

	/**
	 * Return list of file names that the
	 * Spring test framework uses in order to create beans for testing.
	 * @return Array of string filenames
	 */
	protected final String[] getConfigLocations() {
		return this.contexts.getConfigLocations();
	}
	
	/**
	 * Search for object identified by <code>key</code>
	 * in the hierarchy of contexts.
	 * @param key Identifier of data value to find
	 * @return Object if found or null
	 * @throws Exception if errors occur when using reflection
	 * to access the SpringUnitContext for any
	 * class in the list
	 */
	protected final <T> T getObject(String key) throws Exception {
		return (T)this.contexts.getObject(key, getName(), this);
	}
	
	/**
	 * Subclasses can override this method to perform any setup operations,
	 * such as populating a database table, <i>within</i> the transaction
	 * created by this class.
	 * <p><b>NB:</b> Not called if there is no transaction management, due to no
	 * transaction manager being provided in the context.
	 * @throws Exception simply let any exception propagate
	 */
	protected final void onSetUpInTransaction() throws Exception {
		populateApplicationContext();
		onSetUpInTransactionAtBeginning();
	}

	/**
	 * Subclasses can override this method in order to do any work
	 * after the transaction opens.
	 * @throws Exception
	 */
	protected void onSetUpInTransactionAtBeginning() throws Exception {
	}

	/**
	 * Calls onTearDownAfterTransactionEnds so that subclasses can
	 * insert behavior at conclusion of transaction.
	 * The transaction is <i>not open anymore</i> at this point.
	 * Follows by calling setDirty so that all beans created
	 * from configuration files are refreshed.
	 * @throws Exception simply let any exception propagate
	 */
	protected final void onTearDownAfterTransaction() throws Exception {
		try {
			onTearDownAfterTransactionEnds();
		}
		finally {
			setDirty();
		}
	}

	/**
	 * Subclasses can override this method to perform cleanup here.
	 * The transaction is <i>not open anymore</i> at this point.
	 * @throws Exception simply let any exception propagate
	 */
	protected void onTearDownAfterTransactionEnds() throws Exception {
	}

	/**
	 * Container of the hierarchy of contexts for the test.<br/>
	 */
	private HierarchicalSpringUnitContext<SpringUnitTransactionalTest> contexts;
	
}
