// Styles
import './VRadio.sass'

// Components
import VRadioGroup from './VRadioGroup'
import VLabel from '../VLabel'
import VIcon from '../VIcon'
import VInput from '../VInput'

// Mixins
import Colorable from '../../mixins/colorable'
import { factory as GroupableFactory } from '../../mixins/groupable'
import Rippleable from '../../mixins/rippleable'
import Themeable from '../../mixins/themeable'
import Selectable from '../../mixins/selectable'

// Utilities
import { getSlot } from '../../util/helpers'

// Types
import { VNode, VNodeData } from 'vue'
import mixins from '../../util/mixins'

const baseMixins = mixins(
  Colorable,
  Rippleable,
  GroupableFactory('radioGroup'),
  Themeable
)

interface options extends InstanceType<typeof baseMixins> {
  radioGroup: InstanceType<typeof VRadioGroup>
}

/* @vue/component */
export default baseMixins.extend<options>().extend({
  name: 'v-radio',

  inheritAttrs: false,

  props: {
    color: {
      type: String,
      default: 'accent',
    },
    disabled: Boolean,
    label: String,
    name: String,
    id: String,
    onIcon: {
      type: String,
      default: '$vuetify.icons.radioOn',
    },
    offIcon: {
      type: String,
      default: '$vuetify.icons.radioOff',
    },
    readonly: Boolean,
    value: {
      default: null,
    },
  },

  data: () => ({
    isFocused: false,
  }),

  computed: {
    classes (): object {
      return {
        'v-radio--is-disabled': this.isDisabled,
        'v-radio--is-focused': this.isFocused,
        ...this.themeClasses,
        ...this.groupClasses,
      }
    },
    computedColor (): string | false {
      const color = (this.radioGroup || {}).validationState

      return this.isActive ? this.color : (color || false)
    },
    computedIcon (): string {
      return this.isActive
        ? this.onIcon
        : this.offIcon
    },
    hasLabel: VInput.options.computed.hasLabel,
    hasState (): boolean {
      return (this.radioGroup || {}).hasState
    },
    isDisabled (): boolean {
      return this.disabled || !!(this.radioGroup || {}).disabled
    },
    isReadonly (): boolean {
      return this.readonly || !!(this.radioGroup || {}).readonly
    },
    computedName (): string {
      if (this.name || !this.radioGroup) {
        return this.name
      }

      return this.radioGroup.name || `'v-radio-'${this.radioGroup._uid}`
    },
  },

  methods: {
    genInput (args: any) {
      // We can't actually use the mixin directly because
      // it's made for standalone components, but its
      // genInput method is exactly what we need
      return Selectable.options.methods.genInput.call(this, 'radio', args)
    },
    genLabel () {
      if (!this.hasLabel) return null

      return this.$createElement(VLabel, {
        on: { click: this.onChange },
        attrs: {
          for: this.id,
        },
        props: {
          color: ((this.radioGroup || {}).validationState) || '',
          focused: this.hasState,
        },
      }, getSlot(this, 'label') || this.label)
    },
    genRadio () {
      return this.$createElement('div', {
        staticClass: 'v-input--selection-controls__input',
      }, [
        this.genInput({
          name: this.computedName,
          value: this.value,
          ...this.$attrs,
        }),
        this.genRipple(this.setTextColor(this.computedColor)),
        this.$createElement(VIcon, this.setTextColor(this.computedColor, {}), this.computedIcon),
      ])
    },
    onFocus (e: Event) {
      this.isFocused = true
      this.$emit('focus', e)
    },
    onBlur (e: Event) {
      this.isFocused = false
      this.$emit('blur', e)
    },
    onChange () {
      if (this.isDisabled || this.isReadonly || this.isActive) return

      this.toggle()
    },
    onKeydown: () => {}, // Override default with noop
  },

  render (h): VNode {
    let data = {
      staticClass: 'v-radio',
      class: this.classes,
    } as VNodeData

    if ((this.radioGroup || {}).hasError) {
      data = this.setTextColor(this.color, data)
    }

    return h('div', data, [
      this.genRadio(),
      this.genLabel(),
    ])
  },
})
