// Directives
import { Scroll } from '../../directives';
// Utilities
import { consoleWarn } from '../../util/console';
// Types
import Vue from 'vue';
/**
 * Scrollable
 *
 * Used for monitoring scrolling and
 * invoking functions based upon
 * scrolling thresholds being
 * met.
 */
/* @vue/component */
export default Vue.extend({
    name: 'scrollable',
    directives: { Scroll: Scroll },
    props: {
        scrollTarget: String,
        scrollThreshold: [String, Number]
    },
    data: function data() {
        return {
            currentScroll: 0,
            currentThreshold: 0,
            isActive: false,
            isScrollingUp: false,
            previousScroll: 0,
            savedScroll: 0,
            target: null
        };
    },
    computed: {
        /**
         * A computed property that returns
         * whether scrolling features are
         * enabled or disabled
         */
        canScroll: function canScroll() {
            return typeof window !== 'undefined';
        },

        /**
         * The threshold that must be met before
         * thresholdMet function is invoked
         */
        computedScrollThreshold: function computedScrollThreshold() {
            return this.scrollThreshold ? Number(this.scrollThreshold) : 300;
        }
    },
    watch: {
        isScrollingUp: function isScrollingUp() {
            this.savedScroll = this.savedScroll || this.currentScroll;
        },
        isActive: function isActive() {
            this.savedScroll = 0;
        }
    },
    mounted: function mounted() {
        if (this.scrollTarget) {
            this.target = document.querySelector(this.scrollTarget);
            if (!this.target) {
                consoleWarn('Unable to locate element with identifier ' + this.scrollTarget, this);
            }
        }
    },

    methods: {
        onScroll: function onScroll() {
            var _this = this;

            if (!this.canScroll) return;
            this.previousScroll = this.currentScroll;
            this.currentScroll = this.target ? this.target.scrollTop : window.pageYOffset;
            this.isScrollingUp = this.currentScroll < this.previousScroll;
            this.currentThreshold = Math.abs(this.currentScroll - this.computedScrollThreshold);
            this.$nextTick(function () {
                if (Math.abs(_this.currentScroll - _this.savedScroll) > _this.computedScrollThreshold) _this.thresholdMet();
            });
        },

        /**
         * The method invoked when
         * scrolling in any direction
         * has exceeded the threshold
         */
        thresholdMet: function thresholdMet() {}
    }
});
//# sourceMappingURL=index.js.map