import { createVNode as _createVNode } from "vue";
// Styles
import "./VSlideGroup.css"; // Components

import { VFadeTransition } from "../transitions/index.mjs";
import { VIcon } from "../VIcon/index.mjs"; // Composables

import { makeGroupProps, useGroup } from "../../composables/group.mjs";
import { makeTagProps } from "../../composables/tag.mjs";
import { useDisplay } from "../../composables/index.mjs";
import { useResizeObserver } from "../../composables/resizeObserver.mjs";
import { useRtl } from "../../composables/rtl.mjs";
import { IconValue } from "../../composables/icons.mjs"; // Utilities

import { bias, calculateCenteredOffset, calculateUpdatedOffset } from "./helpers.mjs";
import { clamp, defineComponent, useRender } from "../../util/index.mjs";
import { computed, ref, watch, watchEffect } from 'vue'; // Types

export const VSlideGroupSymbol = Symbol.for('vuetify:v-slide-group');
export const VSlideGroup = defineComponent({
  name: 'VSlideGroup',
  props: {
    centerActive: Boolean,
    direction: {
      type: String,
      default: 'horizontal'
    },
    symbol: {
      type: null,
      default: VSlideGroupSymbol
    },
    nextIcon: {
      type: IconValue,
      default: '$next'
    },
    prevIcon: {
      type: IconValue,
      default: '$prev'
    },
    showArrows: {
      type: [Boolean, String],
      validator: v => typeof v === 'boolean' || ['always', 'desktop', 'mobile'].includes(v)
    },
    ...makeTagProps(),
    ...makeGroupProps({
      selectedClass: 'v-slide-group-item--active'
    })
  },
  emits: {
    'update:modelValue': value => true
  },

  setup(props, _ref) {
    let {
      slots
    } = _ref;
    const {
      isRtl
    } = useRtl();
    const {
      mobile
    } = useDisplay();
    const group = useGroup(props, props.symbol);
    const isOverflowing = ref(false);
    const scrollOffset = ref(0);
    const containerSize = ref(0);
    const contentSize = ref(0);
    const isHorizontal = computed(() => props.direction === 'horizontal');
    const {
      resizeRef: containerRef,
      contentRect: containerRect
    } = useResizeObserver();
    const {
      resizeRef: contentRef,
      contentRect
    } = useResizeObserver();
    watchEffect(() => {
      if (!containerRect.value || !contentRect.value) return;
      const sizeProperty = isHorizontal.value ? 'width' : 'height';
      containerSize.value = containerRect.value[sizeProperty];
      contentSize.value = contentRect.value[sizeProperty];
      isOverflowing.value = containerSize.value + 1 < contentSize.value;
    });
    const firstSelectedIndex = computed(() => {
      if (!group.selected.value.length) return -1;
      return group.items.value.findIndex(item => item.id === group.selected.value[0]);
    });
    const lastSelectedIndex = computed(() => {
      if (!group.selected.value.length) return -1;
      return group.items.value.findIndex(item => item.id === group.selected.value[group.selected.value.length - 1]);
    });
    watch(group.selected, () => {
      if (firstSelectedIndex.value < 0 || !contentRef.value) return; // TODO: Is this too naive? Should we store element references in group composable?

      const selectedElement = contentRef.value.children[lastSelectedIndex.value];

      if (firstSelectedIndex.value === 0 || !isOverflowing.value) {
        scrollOffset.value = 0;
      } else if (props.centerActive) {
        scrollOffset.value = calculateCenteredOffset({
          selectedElement,
          containerSize: containerSize.value,
          contentSize: contentSize.value,
          isRtl: isRtl.value,
          isHorizontal: isHorizontal.value
        });
      } else if (isOverflowing.value) {
        scrollOffset.value = calculateUpdatedOffset({
          selectedElement,
          containerSize: containerSize.value,
          contentSize: contentSize.value,
          isRtl: isRtl.value,
          currentScrollOffset: scrollOffset.value,
          isHorizontal: isHorizontal.value
        });
      }
    });
    let firstOverflow = true;
    watch(isOverflowing, () => {
      if (!firstOverflow || !contentRef.value || firstSelectedIndex.value < 0) return;
      firstOverflow = false; // TODO: Is this too naive? Should we store element references in group composable?

      const selectedElement = contentRef.value.children[firstSelectedIndex.value];
      scrollOffset.value = calculateCenteredOffset({
        selectedElement,
        containerSize: containerSize.value,
        contentSize: contentSize.value,
        isRtl: isRtl.value,
        isHorizontal: isHorizontal.value
      });
    });
    const disableTransition = ref(false);
    let startTouch = 0;
    let startOffset = 0;

    function onTouchstart(e) {
      const sizeProperty = isHorizontal.value ? 'clientX' : 'clientY';
      startOffset = scrollOffset.value;
      startTouch = e.touches[0][sizeProperty];
      disableTransition.value = true;
    }

    function onTouchmove(e) {
      if (!isOverflowing.value) return;
      const sizeProperty = isHorizontal.value ? 'clientX' : 'clientY';
      scrollOffset.value = startOffset + startTouch - e.touches[0][sizeProperty];
    }

    function onTouchend(e) {
      const maxScrollOffset = contentSize.value - containerSize.value;

      if (isRtl.value) {
        if (scrollOffset.value > 0 || !isOverflowing.value) {
          scrollOffset.value = 0;
        } else if (scrollOffset.value <= -maxScrollOffset) {
          scrollOffset.value = -maxScrollOffset;
        }
      } else {
        if (scrollOffset.value < 0 || !isOverflowing.value) {
          scrollOffset.value = 0;
        } else if (scrollOffset.value >= maxScrollOffset) {
          scrollOffset.value = maxScrollOffset;
        }
      }

      disableTransition.value = false;
    }

    function onScroll() {
      containerRef.value && (containerRef.value.scrollLeft = 0);
    }

    const isFocused = ref(false);

    function onFocusin(e) {
      isFocused.value = true;
      if (!isOverflowing.value || !contentRef.value) return; // Focused element is likely to be the root of an item, so a
      // breadth-first search will probably find it in the first iteration

      for (const el of e.composedPath()) {
        for (const item of contentRef.value.children) {
          if (item === el) {
            scrollOffset.value = calculateUpdatedOffset({
              selectedElement: item,
              containerSize: containerSize.value,
              contentSize: contentSize.value,
              isRtl: isRtl.value,
              currentScrollOffset: scrollOffset.value,
              isHorizontal: isHorizontal.value
            });
            return;
          }
        }
      }
    }

    function onFocusout(e) {
      isFocused.value = false;
    }

    function onFocus(e) {
      var _contentRef$value;

      if (!isFocused.value && !(e.relatedTarget && (_contentRef$value = contentRef.value) != null && _contentRef$value.contains(e.relatedTarget))) focus();
    }

    function onKeydown(e) {
      if (!contentRef.value) return;

      if (e.key === (isHorizontal.value ? 'ArrowRight' : 'ArrowDown')) {
        focus('next');
      } else if (e.key === (isHorizontal.value ? 'ArrowLeft' : 'ArrowUp')) {
        focus('prev');
      } else if (e.key === 'Home') {
        focus('first');
      } else if (e.key === 'End') {
        focus('last');
      }
    }

    function focus(location) {
      if (!contentRef.value) return;

      if (!location) {
        var _focusable$;

        contentRef.value.querySelector('[tabindex]');
        const focusable = [...contentRef.value.querySelectorAll('button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])')].filter(el => !el.hasAttribute('disabled'));
        (_focusable$ = focusable[0]) == null ? void 0 : _focusable$.focus();
      } else if (location === 'next') {
        var _contentRef$value$que;

        const el = (_contentRef$value$que = contentRef.value.querySelector(':focus')) == null ? void 0 : _contentRef$value$que.nextElementSibling;
        if (el) el.focus();else focus('first');
      } else if (location === 'prev') {
        var _contentRef$value$que2;

        const el = (_contentRef$value$que2 = contentRef.value.querySelector(':focus')) == null ? void 0 : _contentRef$value$que2.previousElementSibling;
        if (el) el.focus();else focus('last');
      } else if (location === 'first') {
        var _contentRef$value$fir;

        (_contentRef$value$fir = contentRef.value.firstElementChild) == null ? void 0 : _contentRef$value$fir.focus();
      } else if (location === 'last') {
        var _contentRef$value$las;

        (_contentRef$value$las = contentRef.value.lastElementChild) == null ? void 0 : _contentRef$value$las.focus();
      }
    }

    function scrollTo(location) {
      const sign = isRtl.value ? -1 : 1;
      const newAbosluteOffset = sign * scrollOffset.value + (location === 'prev' ? -1 : 1) * containerSize.value;
      scrollOffset.value = sign * clamp(newAbosluteOffset, 0, contentSize.value - containerSize.value);
    }

    const contentStyles = computed(() => {
      const scrollAmount = scrollOffset.value <= 0 ? bias(-scrollOffset.value) : scrollOffset.value > contentSize.value - containerSize.value ? -(contentSize.value - containerSize.value) + bias(contentSize.value - containerSize.value - scrollOffset.value) : -scrollOffset.value;
      return {
        transform: `translate${isHorizontal.value ? 'X' : 'Y'}(${scrollAmount}px)`,
        transition: disableTransition.value ? 'none' : '',
        willChange: disableTransition.value ? 'transform' : ''
      };
    });
    const slotProps = computed(() => ({
      next: group.next,
      prev: group.prev,
      select: group.select,
      isSelected: group.isSelected
    }));
    const hasAffixes = computed(() => {
      switch (props.showArrows) {
        // Always show arrows on desktop & mobile
        case 'always':
          return true;
        // Always show arrows on desktop

        case 'desktop':
          return !mobile.value;
        // Show arrows on mobile when overflowing.
        // This matches the default 2.2 behavior

        case true:
          return isOverflowing.value || Math.abs(scrollOffset.value) > 0;
        // Always show on mobile

        case 'mobile':
          return mobile.value || isOverflowing.value || Math.abs(scrollOffset.value) > 0;
        // https://material.io/components/tabs#scrollable-tabs
        // Always show arrows when
        // overflowed on desktop

        default:
          return !mobile.value && (isOverflowing.value || Math.abs(scrollOffset.value) > 0);
      }
    });
    const hasPrev = computed(() => {
      return hasAffixes.value && scrollOffset.value > 0;
    });
    const hasNext = computed(() => {
      if (!hasAffixes.value) return false; // Check one scroll ahead to know the width of right-most item

      return contentSize.value > Math.abs(scrollOffset.value) + containerSize.value;
    });
    useRender(() => {
      var _slots$prev, _slots$prev2, _slots$default, _slots$next, _slots$next2;

      return _createVNode(props.tag, {
        "class": ['v-slide-group', {
          'v-slide-group--vertical': !isHorizontal.value,
          'v-slide-group--has-affixes': hasAffixes.value,
          'v-slide-group--is-overflowing': isOverflowing.value
        }],
        "tabindex": isFocused.value || group.selected.value.length ? -1 : 0,
        "onFocus": onFocus
      }, {
        default: () => [hasAffixes.value && _createVNode("div", {
          "class": ['v-slide-group__prev', {
            'v-slide-group__prev--disabled': !hasPrev.value
          }],
          "onClick": () => scrollTo('prev')
        }, [(_slots$prev = (_slots$prev2 = slots.prev) == null ? void 0 : _slots$prev2.call(slots, slotProps.value)) != null ? _slots$prev : _createVNode(VFadeTransition, null, {
          default: () => [_createVNode(VIcon, {
            "icon": props.prevIcon
          }, null)]
        })]), _createVNode("div", {
          "ref": containerRef,
          "class": "v-slide-group__container",
          "onScroll": onScroll
        }, [_createVNode("div", {
          "ref": contentRef,
          "class": "v-slide-group__content",
          "style": contentStyles.value,
          "onTouchstartPassive": onTouchstart,
          "onTouchmovePassive": onTouchmove,
          "onTouchendPassive": onTouchend,
          "onFocusin": onFocusin,
          "onFocusout": onFocusout,
          "onKeydown": onKeydown
        }, [(_slots$default = slots.default) == null ? void 0 : _slots$default.call(slots, slotProps.value)])]), hasAffixes.value && _createVNode("div", {
          "class": ['v-slide-group__next', {
            'v-slide-group__next--disabled': !hasNext.value
          }],
          "onClick": () => scrollTo('next')
        }, [(_slots$next = (_slots$next2 = slots.next) == null ? void 0 : _slots$next2.call(slots, slotProps.value)) != null ? _slots$next : _createVNode(VFadeTransition, null, {
          default: () => [_createVNode(VIcon, {
            "icon": props.nextIcon
          }, null)]
        })])]
      });
    });
    return {
      selected: group.selected,
      scrollTo,
      scrollOffset,
      focus
    };
  }

});
//# sourceMappingURL=VSlideGroup.mjs.map