/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.http.impl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import javax.security.auth.DestroyFailedException;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import javax.security.sasl.RealmCallback;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.callback.AuthenticationCompleteCallback;
import org.wildfly.security.auth.callback.AvailableRealmsCallback;
import org.wildfly.security.auth.callback.CredentialCallback;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.http.HttpAuthenticationException;
import org.wildfly.security.http.HttpServerAuthenticationMechanism;
import org.wildfly.security.http.HttpServerRequest;
import org.wildfly.security.http.HttpServerResponse;
import org.wildfly.security.http.impl.NonceManager;
import org.wildfly.security.mechanism.AuthenticationMechanismException;
import org.wildfly.security.mechanism.digest.DigestQuote;
import org.wildfly.security.mechanism.digest.DigestUtil;
import org.wildfly.security.password.TwoWayPassword;
import org.wildfly.security.password.interfaces.DigestPassword;
import org.wildfly.security.util.ByteIterator;

class DigestAuthenticationMechanism
implements HttpServerAuthenticationMechanism {
    private static final String CHALLENGE_PREFIX = "Digest ";
    private static final int PREFIX_LENGTH = "Digest ".length();
    private static final String OPAQUE_VALUE = "00000000000000000000000000000000";
    private static final byte COLON = 58;
    private final CallbackHandler callbackHandler;
    private final NonceManager nonceManager;
    private final String configuredRealm;
    private final String domain;

    DigestAuthenticationMechanism(CallbackHandler callbackHandler, NonceManager nonceManager, String configuredRealm, String domain) {
        this.callbackHandler = callbackHandler;
        this.nonceManager = nonceManager;
        this.configuredRealm = configuredRealm;
        this.domain = domain;
    }

    @Override
    public String getMechanismName() {
        return "DIGEST";
    }

    @Override
    public void evaluateRequest(HttpServerRequest request) throws HttpAuthenticationException {
        String realmName = this.selectRealm(request);
        List<String> authorizationValues = request.getRequestHeaderValues("Authorization");
        if (authorizationValues != null) {
            for (String current : authorizationValues) {
                if (!current.startsWith(CHALLENGE_PREFIX)) continue;
                byte[] rawHeader = current.substring(CHALLENGE_PREFIX.length()).getBytes(StandardCharsets.UTF_8);
                try {
                    HashMap<String, byte[]> responseTokens = DigestUtil.parseResponse(rawHeader, StandardCharsets.UTF_8, false, this.getMechanismName());
                    this.validateResponse(responseTokens, request);
                    return;
                }
                catch (AuthenticationMechanismException e) {
                    request.badRequest(e.toHttpAuthenticationException(), response -> this.prepareResponse(realmName, response, false));
                    return;
                }
            }
        }
        request.noAuthenticationInProgress(response -> this.prepareResponse(realmName, response, false));
    }

    private void validateResponse(HashMap<String, byte[]> responseTokens, HttpServerRequest request) throws AuthenticationMechanismException, HttpAuthenticationException {
        String[] availableRealms;
        MessageDigest messageDigest;
        String nonce = this.convertToken("nonce", responseTokens.get("nonce"));
        String messageRealm = this.convertToken("realm", responseTokens.get("realm"));
        boolean nonceValid = this.nonceManager.useNonce(nonce, messageRealm.getBytes(StandardCharsets.UTF_8));
        String username = this.convertToken("username", responseTokens.get("username"));
        if (!responseTokens.containsKey("uri")) {
            throw ElytronMessages.log.mechMissingDirective(this.getMechanismName(), "uri");
        }
        byte[] digestUri = responseTokens.get("uri");
        if (!responseTokens.containsKey("response")) {
            throw ElytronMessages.log.mechMissingDirective(this.getMechanismName(), "response");
        }
        byte[] response = ByteIterator.ofBytes(responseTokens.get("response")).hexDecode().drain();
        String algorithm = this.convertToken("algorithm", responseTokens.get("algorithm"));
        if (!"MD5".equals(algorithm)) {
            throw ElytronMessages.log.mechUnsupportedAlgorithm(this.getMechanismName(), algorithm);
        }
        try {
            messageDigest = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw ElytronMessages.log.mechMacAlgorithmNotSupported(this.getMechanismName(), e);
        }
        String mechanismRealm = null;
        for (String current : availableRealms = this.getAvailableRealms()) {
            if (!messageRealm.equals(current)) continue;
            mechanismRealm = current;
            break;
        }
        if (mechanismRealm == null && this.getAvailableRealms().length > 0 && (messageRealm.equals(this.configuredRealm) || messageRealm.equals(request.getFirstRequestHeaderValue("Host")))) {
            mechanismRealm = availableRealms[0];
        }
        if (mechanismRealm == null) {
            throw ElytronMessages.log.mechDisallowedClientRealm(this.getMechanismName(), mechanismRealm);
        }
        String selectedRealm = this.selectRealm(request);
        byte[] hA1 = this.getH_A1(messageDigest, username, messageRealm, mechanismRealm);
        if (hA1 == null) {
            this.fail();
            request.authenticationFailed(ElytronMessages.log.authenticationFailed(this.getMechanismName()), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        byte[] calculatedResponse = this.calculateResponseDigest(messageDigest, hA1, nonce, request.getRequestMethod(), digestUri);
        if (!Arrays.equals(response, calculatedResponse)) {
            this.fail();
            request.authenticationFailed(ElytronMessages.log.mechResponseTokenMismatch(this.getMechanismName()), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        if (!nonceValid) {
            request.authenticationInProgress(httpResponse -> this.prepareResponse(selectedRealm, httpResponse, true));
            return;
        }
        if (this.authorize(username)) {
            this.succeed();
            request.authenticationComplete();
        } else {
            this.fail();
            request.authenticationFailed(ElytronMessages.log.authorizationFailed(username, this.getMechanismName()), httpResponse -> httpResponse.setStatusCode(403));
        }
    }

    private byte[] calculateResponseDigest(MessageDigest messageDigest, byte[] hA1, String nonce, String method, byte[] digestUri) {
        messageDigest.update(method.getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        byte[] hA2 = messageDigest.digest(digestUri);
        messageDigest.update(ByteIterator.ofBytes(hA1).hexEncode().drainToString().getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        messageDigest.update(nonce.getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        return messageDigest.digest(ByteIterator.ofBytes(hA2).hexEncode().drainToString().getBytes(StandardCharsets.UTF_8));
    }

    private byte[] getH_A1(MessageDigest messageDigest, String username, String messageRealm, String mechanismRealm) throws AuthenticationMechanismException {
        NameCallback nameCallback = new NameCallback("User name", username);
        RealmCallback realmCallback = new RealmCallback("User realm", messageRealm);
        byte[] response = null;
        if (mechanismRealm.equals(messageRealm) && (response = this.getPredigestedSaltedPassword(realmCallback, nameCallback, "digest-md5", this.getMechanismName())) != null) {
            return response;
        }
        response = this.getSaltedPasswordFromTwoWay(messageDigest, realmCallback, nameCallback);
        if (response != null) {
            return response;
        }
        response = this.getSaltedPasswordFromPasswordCallback(messageDigest, realmCallback, nameCallback);
        return response;
    }

    private String convertToken(String name, byte[] value) throws AuthenticationMechanismException {
        if (value == null) {
            throw ElytronMessages.log.mechMissingDirective(this.getMechanismName(), name);
        }
        return new String(value, StandardCharsets.UTF_8);
    }

    private String selectRealm(HttpServerRequest request) throws HttpAuthenticationException {
        String[] realms;
        if (this.configuredRealm != null) {
            return this.configuredRealm;
        }
        try {
            realms = this.getAvailableRealms();
        }
        catch (AuthenticationMechanismException e) {
            throw e.toHttpAuthenticationException();
        }
        if (realms.length > 0) {
            return realms[0];
        }
        return request.getFirstRequestHeaderValue("Host");
    }

    private String[] getAvailableRealms() throws AuthenticationMechanismException {
        AvailableRealmsCallback availableRealmsCallback = new AvailableRealmsCallback();
        try {
            this.callbackHandler.handle(new Callback[]{availableRealmsCallback});
            return availableRealmsCallback.getRealmNames();
        }
        catch (UnsupportedCallbackException ignored) {
            return new String[0];
        }
        catch (AuthenticationMechanismException e) {
            throw e;
        }
        catch (IOException e) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason("DIGEST", e);
        }
    }

    private void prepareResponse(String realmName, HttpServerResponse response, boolean stale) throws HttpAuthenticationException {
        StringBuilder sb = new StringBuilder(CHALLENGE_PREFIX);
        sb.append("realm").append("=\"").append(DigestQuote.quote(realmName)).append("\"");
        if (this.domain != null) {
            sb.append(", ").append("domain").append("=\"").append(this.domain).append("\"");
        }
        sb.append(", ").append("nonce").append("=\"").append(this.nonceManager.generateNonce(realmName.getBytes(StandardCharsets.UTF_8))).append("\"");
        sb.append(", ").append("opaque").append("=\"").append(OPAQUE_VALUE).append("\"");
        if (stale) {
            sb.append(", ").append("stale").append("=\"true\"");
        }
        sb.append(", ").append("algorithm").append("=\"").append("MD5").append("\"");
        response.addResponseHeader("WWW-Authenticate", sb.toString());
        response.setStatusCode(401);
    }

    public byte[] getPredigestedSaltedPassword(RealmCallback realmCallback, NameCallback nameCallback, String passwordAlgorithm, String mechanismName) throws AuthenticationMechanismException {
        CredentialCallback credentialCallback = new CredentialCallback(PasswordCredential.class, passwordAlgorithm);
        try {
            this.callbackHandler.handle(new Callback[]{realmCallback, nameCallback, credentialCallback});
            return credentialCallback.applyToCredential(PasswordCredential.class, c -> c.getPassword().castAndApply(DigestPassword.class, DigestPassword::getDigest));
        }
        catch (UnsupportedCallbackException e) {
            if (e.getCallback() == credentialCallback) {
                return null;
            }
            if (e.getCallback() == nameCallback) {
                throw ElytronMessages.log.mechCallbackHandlerDoesNotSupportUserName(mechanismName, e);
            }
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(mechanismName, e);
        }
        catch (Throwable t) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(mechanismName, t);
        }
    }

    protected byte[] getSaltedPasswordFromTwoWay(MessageDigest messageDigest, RealmCallback realmCallback, NameCallback nameCallback) throws AuthenticationMechanismException {
        CredentialCallback credentialCallback = new CredentialCallback(PasswordCredential.class, "clear");
        try {
            this.callbackHandler.handle(new Callback[]{realmCallback, nameCallback, credentialCallback});
        }
        catch (UnsupportedCallbackException e) {
            if (e.getCallback() == credentialCallback) {
                return null;
            }
            if (e.getCallback() == nameCallback) {
                throw ElytronMessages.log.mechCallbackHandlerDoesNotSupportUserName(this.getMechanismName(), e);
            }
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), e);
        }
        catch (Throwable t) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), t);
        }
        TwoWayPassword password = credentialCallback.applyToCredential(PasswordCredential.class, c -> c.getPassword().castAs(TwoWayPassword.class));
        char[] passwordChars = DigestUtil.getTwoWayPasswordChars(this.getMechanismName(), password);
        try {
            password.destroy();
        }
        catch (DestroyFailedException e) {
            ElytronMessages.log.credentialDestroyingFailed(e);
        }
        String realm = realmCallback.getDefaultText();
        String username = nameCallback.getDefaultName();
        byte[] digest_urp = DigestUtil.userRealmPasswordDigest(messageDigest, username, realm, passwordChars);
        Arrays.fill(passwordChars, '\u0000');
        return digest_urp;
    }

    protected byte[] getSaltedPasswordFromPasswordCallback(MessageDigest messageDigest, RealmCallback realmCallback, NameCallback nameCallback) throws AuthenticationMechanismException {
        PasswordCallback passwordCallback = new PasswordCallback("User password", false);
        try {
            this.callbackHandler.handle(new Callback[]{realmCallback, nameCallback, passwordCallback});
        }
        catch (UnsupportedCallbackException e) {
            if (e.getCallback() == passwordCallback) {
                return null;
            }
            if (e.getCallback() == nameCallback) {
                throw ElytronMessages.log.mechCallbackHandlerDoesNotSupportUserName(this.getMechanismName(), e);
            }
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), e);
        }
        catch (Throwable t) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), t);
        }
        char[] passwordChars = passwordCallback.getPassword();
        passwordCallback.clearPassword();
        if (passwordChars == null) {
            throw ElytronMessages.log.mechNoPasswordGiven(this.getMechanismName());
        }
        String realm = realmCallback.getDefaultText();
        String username = nameCallback.getDefaultName();
        byte[] digest_urp = DigestUtil.userRealmPasswordDigest(messageDigest, username, realm, passwordChars);
        Arrays.fill(passwordChars, '\u0000');
        return digest_urp;
    }

    protected boolean authorize(String username) throws AuthenticationMechanismException {
        AuthorizeCallback authorizeCallback = new AuthorizeCallback(username, username);
        try {
            this.callbackHandler.handle(new Callback[]{authorizeCallback});
            return authorizeCallback.isAuthorized();
        }
        catch (UnsupportedCallbackException e) {
            return false;
        }
        catch (Throwable t) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), t);
        }
    }

    protected void succeed() throws AuthenticationMechanismException {
        try {
            this.callbackHandler.handle(new Callback[]{AuthenticationCompleteCallback.SUCCEEDED});
        }
        catch (Throwable t) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), t);
        }
    }

    protected void fail() throws AuthenticationMechanismException {
        try {
            this.callbackHandler.handle(new Callback[]{AuthenticationCompleteCallback.FAILED});
        }
        catch (Throwable t) {
            throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason(this.getMechanismName(), t);
        }
    }
}

