/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.http.impl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.GeneralSecurityException;
import java.security.PrivilegedActionException;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import javax.security.auth.Subject;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.auth.kerberos.KerberosTicket;
import javax.security.sasl.AuthorizeCallback;
import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSCredential;
import org.ietf.jgss.GSSException;
import org.ietf.jgss.GSSManager;
import org.ietf.jgss.GSSName;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.auth.callback.AuthenticationCompleteCallback;
import org.wildfly.security.auth.callback.IdentityCredentialCallback;
import org.wildfly.security.auth.callback.ServerCredentialCallback;
import org.wildfly.security.credential.GSSKerberosCredential;
import org.wildfly.security.http.HttpAuthenticationException;
import org.wildfly.security.http.HttpScope;
import org.wildfly.security.http.HttpServerAuthenticationMechanism;
import org.wildfly.security.http.HttpServerRequest;
import org.wildfly.security.http.HttpServerResponse;
import org.wildfly.security.http.Scope;
import org.wildfly.security.mechanism.AuthenticationMechanismException;
import org.wildfly.security.mechanism.MechanismUtil;
import org.wildfly.security.util.ByteIterator;

public class SpnegoAuthenticationMechanism
implements HttpServerAuthenticationMechanism {
    private static final String CHALLENGE_PREFIX = "Negotiate ";
    private static final String GSS_CONTEXT_KEY = SpnegoAuthenticationMechanism.class.getName() + ".GSSContext";
    private static final String KERBEROS_TICKET = SpnegoAuthenticationMechanism.class.getName() + ".KerberosTicket";
    private final CallbackHandler callbackHandler;

    SpnegoAuthenticationMechanism(CallbackHandler callbackHandler) {
        this.callbackHandler = callbackHandler;
    }

    @Override
    public String getMechanismName() {
        return "SPNEGO";
    }

    @Override
    public void evaluateRequest(HttpServerRequest request) throws HttpAuthenticationException {
        List<String> authorizationValues;
        Optional challenge;
        HttpScope connectionScope = request.getScope(Scope.CONNECTION);
        GSSContext gssContext = connectionScope != null ? connectionScope.getAttachment(GSS_CONTEXT_KEY, GSSContext.class) : null;
        KerberosTicket kerberosTicket = connectionScope != null ? connectionScope.getAttachment(KERBEROS_TICKET, KerberosTicket.class) : null;
        ElytronMessages.log.tracef("Evaluating SPNEGO request: cached GSSContext = %s", gssContext);
        if (gssContext != null && gssContext.isEstablished() && this.authorizeCachedGSSContext(gssContext)) {
            ElytronMessages.log.trace("Successfully authorized using cached identity");
            request.authenticationComplete();
            return;
        }
        if (gssContext == null) {
            GSSCredential serviceGssCredential;
            ServerCredentialCallback gssCredentialCallback = new ServerCredentialCallback(GSSKerberosCredential.class);
            try {
                ElytronMessages.log.trace("Obtaining GSSCredential for the service from callback handler...");
                this.callbackHandler.handle(new Callback[]{gssCredentialCallback});
                serviceGssCredential = gssCredentialCallback.applyToCredential(GSSKerberosCredential.class, GSSKerberosCredential::getGssCredential);
                kerberosTicket = gssCredentialCallback.applyToCredential(GSSKerberosCredential.class, GSSKerberosCredential::getKerberosTicket);
            }
            catch (IOException | UnsupportedCallbackException e) {
                throw ElytronMessages.log.mechCallbackHandlerFailedForUnknownReason("SPNEGO", e).toHttpAuthenticationException();
            }
            if (serviceGssCredential == null) {
                ElytronMessages.log.trace("GSSCredential for the service from callback handler is null - cannot perform SPNEGO authentication");
                request.noAuthenticationInProgress();
                return;
            }
            try {
                gssContext = GSSManager.getInstance().createContext(serviceGssCredential);
                ElytronMessages.log.tracef("Using SpnegoAuthenticationMechanism to authenticate %s using the following mechanisms: [%s]", serviceGssCredential.getName(), serviceGssCredential.getMechs());
                if (connectionScope != null) {
                    connectionScope.setAttachment(GSS_CONTEXT_KEY, gssContext);
                    ElytronMessages.log.tracef("Caching GSSContext %s", gssContext);
                    connectionScope.setAttachment(KERBEROS_TICKET, kerberosTicket);
                    ElytronMessages.log.tracef("Caching KerberosTicket %s", kerberosTicket);
                }
            }
            catch (GSSException e) {
                throw ElytronMessages.log.mechUnableToCreateGssContext("SPNEGO", e).toHttpAuthenticationException();
            }
        }
        Optional<Object> optional = challenge = (authorizationValues = request.getRequestHeaderValues("Authorization")) != null ? authorizationValues.stream().filter(s -> s.startsWith(CHALLENGE_PREFIX)).limit(1L).map(s -> s.substring(CHALLENGE_PREFIX.length())).findFirst() : Optional.empty();
        if (ElytronMessages.log.isTraceEnabled()) {
            ElytronMessages.log.tracef("Sent HTTP authorizations: [%s]", authorizationValues == null ? "null" : String.join((CharSequence)", ", authorizationValues));
        }
        if (challenge.isPresent()) {
            ElytronMessages.log.trace("Processing incoming response to a challenge...");
            byte[] decodedValue = ByteIterator.ofBytes(((String)challenge.get()).getBytes(StandardCharsets.UTF_8)).base64Decode().drain();
            try {
                byte[] responseToken;
                Subject subject = new Subject(true, Collections.emptySet(), Collections.emptySet(), kerberosTicket != null ? Collections.singleton(kerberosTicket) : Collections.emptySet());
                try {
                    GSSContext finalGssContext = gssContext;
                    responseToken = Subject.doAs(subject, () -> finalGssContext.acceptSecContext(decodedValue, 0, decodedValue.length));
                }
                catch (PrivilegedActionException e) {
                    if (e.getCause() instanceof GSSException) {
                        throw (GSSException)e.getCause();
                    }
                    throw new GeneralSecurityException(e);
                }
                if (gssContext.isEstablished()) {
                    GSSCredential gssCredential;
                    GSSCredential gSSCredential = gssCredential = gssContext.getCredDelegState() ? gssContext.getDelegCred() : null;
                    if (gssCredential != null) {
                        ElytronMessages.log.trace("Associating delegated GSSCredential with identity.");
                        this.handleCallback(new IdentityCredentialCallback(new GSSKerberosCredential(gssCredential), true));
                    } else {
                        ElytronMessages.log.trace("No GSSCredential delegated from client.");
                    }
                    ElytronMessages.log.trace("GSSContext established, authorizing...");
                    GSSName srcName = gssContext.getSrcName();
                    if (srcName == null) {
                        ElytronMessages.log.trace("Authorization failed - srcName of GSSContext (name of initiator) is null - wrong realm or kdc?");
                        if (connectionScope != null) {
                            connectionScope.setAttachment(GSS_CONTEXT_KEY, null);
                        }
                        request.noAuthenticationInProgress(response -> this.sendChallenge(responseToken, response, 401));
                        return;
                    }
                    if (this.authorizeSrcName(srcName, gssContext)) {
                        ElytronMessages.log.trace("GSSContext established and authorized - authentication complete");
                        request.authenticationComplete(response -> this.sendChallenge(responseToken, response, 0));
                        return;
                    }
                    ElytronMessages.log.trace("Authorization of established GSSContext failed");
                    this.handleCallback(AuthenticationCompleteCallback.FAILED);
                    request.authenticationFailed(ElytronMessages.log.authenticationFailed("SPNEGO"), response -> this.sendChallenge(responseToken, response, 403));
                    return;
                }
                if (responseToken != null) {
                    ElytronMessages.log.trace("GSSContext establishing - sending negotiation token to the peer");
                    request.authenticationInProgress(response -> this.sendChallenge(responseToken, response, 401));
                    return;
                }
            }
            catch (GeneralSecurityException | GSSException e) {
                ElytronMessages.log.trace("GSSContext message exchange failed", e);
                this.handleCallback(AuthenticationCompleteCallback.FAILED);
                request.authenticationFailed(ElytronMessages.log.authenticationFailed("SPNEGO"), this::sendBareChallenge);
                return;
            }
        }
        ElytronMessages.log.trace("Request lacks valid authentication credentials");
        if (connectionScope != null) {
            connectionScope.setAttachment(GSS_CONTEXT_KEY, null);
            connectionScope.setAttachment(KERBEROS_TICKET, null);
        }
        request.noAuthenticationInProgress(this::sendBareChallenge);
    }

    private void sendBareChallenge(HttpServerResponse response) {
        response.addResponseHeader("WWW-Authenticate", "Negotiate");
        response.setStatusCode(401);
    }

    private void sendChallenge(byte[] responseToken, HttpServerResponse response, int statusCode) {
        ElytronMessages.log.tracef("Sending intermediate challenge: %s", responseToken);
        if (responseToken == null) {
            response.addResponseHeader("WWW-Authenticate", "Negotiate");
        } else {
            String responseConverted = ByteIterator.ofBytes(responseToken).base64Encode().drainToString();
            response.addResponseHeader("WWW-Authenticate", CHALLENGE_PREFIX + responseConverted);
        }
        if (statusCode != 0) {
            response.setStatusCode(statusCode);
        }
    }

    private boolean authorizeCachedGSSContext(GSSContext gssContext) throws HttpAuthenticationException {
        try {
            boolean authorized;
            GSSName srcName = gssContext.getSrcName();
            boolean bl = authorized = srcName != null && this.authorizeSrcName(srcName, gssContext);
            if (authorized && gssContext.getCredDelegState()) {
                GSSCredential gssCredential = gssContext.getDelegCred();
                if (gssCredential != null) {
                    ElytronMessages.log.trace("Associating delegated GSSCredential with identity.");
                    this.handleCallback(new IdentityCredentialCallback(new GSSKerberosCredential(gssCredential), true));
                } else {
                    ElytronMessages.log.trace("No GSSCredential delegated from client.");
                }
            }
            return authorized;
        }
        catch (GSSException e) {
            ElytronMessages.log.trace("GSSException while obtaining srcName of GSSContext (name of initiator)");
            this.handleCallback(AuthenticationCompleteCallback.FAILED);
            throw ElytronMessages.log.mechServerSideAuthenticationFailed("SPNEGO", e).toHttpAuthenticationException();
        }
    }

    private boolean authorizeSrcName(GSSName srcName, GSSContext gssContext) throws HttpAuthenticationException {
        boolean authorized = false;
        try {
            String clientName = srcName.toString();
            AuthorizeCallback authorize = new AuthorizeCallback(clientName, clientName);
            this.callbackHandler.handle(new Callback[]{authorize});
            authorized = authorize.isAuthorized();
            ElytronMessages.log.tracef("Authorized by callback handler = %b  clientName = [%s]", authorized, clientName);
        }
        catch (IOException e) {
            ElytronMessages.log.trace("IOException during AuthorizeCallback handling", e);
            throw ElytronMessages.log.mechServerSideAuthenticationFailed("SPNEGO", e).toHttpAuthenticationException();
        }
        catch (UnsupportedCallbackException e) {
            // empty catch block
        }
        if (authorized) {
            if (gssContext.getCredDelegState()) {
                try {
                    GSSCredential credential = gssContext.getDelegCred();
                    ElytronMessages.log.tracef("Credential delegation enabled, delegated credential = %s", credential);
                    MechanismUtil.handleCallbacks("SPNEGO", this.callbackHandler, new IdentityCredentialCallback(new GSSKerberosCredential(credential), true));
                }
                catch (UnsupportedCallbackException credential) {
                }
                catch (AuthenticationMechanismException e) {
                    throw e.toHttpAuthenticationException();
                }
                catch (GSSException e) {
                    throw new HttpAuthenticationException(e);
                }
            } else {
                ElytronMessages.log.trace("Credential delegation not enabled");
            }
        }
        this.handleCallback(authorized ? AuthenticationCompleteCallback.SUCCEEDED : AuthenticationCompleteCallback.FAILED);
        return authorized;
    }

    private void handleCallback(Callback callback) throws HttpAuthenticationException {
        try {
            MechanismUtil.handleCallbacks("SPNEGO", this.callbackHandler, callback);
        }
        catch (AuthenticationMechanismException e) {
            throw e.toHttpAuthenticationException();
        }
        catch (UnsupportedCallbackException unsupportedCallbackException) {
            // empty catch block
        }
    }
}

