/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.http.digest;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.function.Supplier;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.UnsupportedCallbackException;
import javax.security.sasl.AuthorizeCallback;
import org.wildfly.common.iteration.ByteIterator;
import org.wildfly.security.auth.callback.AuthenticationCompleteCallback;
import org.wildfly.security.auth.callback.AvailableRealmsCallback;
import org.wildfly.security.http.HttpAuthenticationException;
import org.wildfly.security.http.HttpServerAuthenticationMechanism;
import org.wildfly.security.http.HttpServerMechanismsResponder;
import org.wildfly.security.http.HttpServerRequest;
import org.wildfly.security.http.HttpServerResponse;
import org.wildfly.security.http.digest.NonceManager;
import org.wildfly.security.mechanism.AuthenticationMechanismException;
import org.wildfly.security.mechanism._private.ElytronMessages;
import org.wildfly.security.mechanism.digest.DigestQuote;
import org.wildfly.security.mechanism.digest.DigestUtil;
import org.wildfly.security.mechanism.digest.PasswordDigestObtainer;

final class DigestAuthenticationMechanism
implements HttpServerAuthenticationMechanism {
    private static final String CHALLENGE_PREFIX = "Digest ";
    private static final String OPAQUE_VALUE = "00000000000000000000000000000000";
    private static final byte COLON = 58;
    private final Supplier<Provider[]> providers;
    private final CallbackHandler callbackHandler;
    private final NonceManager nonceManager;
    private final String configuredRealm;
    private final String domain;
    private final String mechanismName;
    private final String algorithm;
    private final boolean validateUri;

    DigestAuthenticationMechanism(CallbackHandler callbackHandler, NonceManager nonceManager, String configuredRealm, String domain, String mechanismName, String algorithm, Supplier<Provider[]> providers, String validateUri) {
        this.callbackHandler = callbackHandler;
        this.nonceManager = nonceManager;
        this.configuredRealm = configuredRealm;
        this.domain = domain;
        this.mechanismName = mechanismName;
        this.algorithm = algorithm;
        this.providers = providers;
        this.validateUri = validateUri == null ? true : Boolean.parseBoolean(validateUri);
    }

    @Override
    public String getMechanismName() {
        return this.mechanismName;
    }

    @Override
    public void evaluateRequest(HttpServerRequest request) throws HttpAuthenticationException {
        List<String> authorizationValues = request.getRequestHeaderValues("Authorization");
        if (authorizationValues != null) {
            for (String current : authorizationValues) {
                if (!current.startsWith(CHALLENGE_PREFIX)) continue;
                byte[] rawHeader = current.substring(CHALLENGE_PREFIX.length()).getBytes(StandardCharsets.UTF_8);
                try {
                    HashMap<String, byte[]> responseTokens = DigestUtil.parseResponse(rawHeader, StandardCharsets.UTF_8, false, ElytronMessages.httpDigest);
                    this.validateResponse(responseTokens, request);
                    return;
                }
                catch (AuthenticationMechanismException e) {
                    ElytronMessages.httpDigest.trace("Failed to parse or validate the response", e);
                    request.badRequest(e.toHttpAuthenticationException(), response -> this.prepareResponse(this.selectRealm(), response, false));
                    return;
                }
            }
        }
        request.noAuthenticationInProgress(response -> this.prepareResponse(this.selectRealm(), response, false));
    }

    private void validateResponse(HashMap<String, byte[]> responseTokens, HttpServerRequest request) throws AuthenticationMechanismException, HttpAuthenticationException {
        MessageDigest messageDigest;
        String algorithm;
        String username;
        int nonceCount;
        String nonce = this.convertToken("nonce", responseTokens.get("nonce"));
        String messageRealm = this.convertToken("realm", responseTokens.get("realm"));
        if (!responseTokens.containsKey("nc")) {
            nonceCount = -1;
        } else {
            String nonceCountHex = this.convertToken("realm", responseTokens.get("nc"));
            nonceCount = Integer.parseInt(nonceCountHex, 16);
            if (nonceCount < 0) {
                throw ElytronMessages.httpDigest.invalidNonceCount(nonceCount);
            }
        }
        final byte[] salt = messageRealm.getBytes(StandardCharsets.UTF_8);
        boolean nonceValid = this.nonceManager.useNonce(nonce, salt, nonceCount);
        if (responseTokens.containsKey("username") && !responseTokens.containsKey("username*")) {
            username = this.convertToken("username", responseTokens.get("username"));
        } else if (responseTokens.containsKey("username*") && !responseTokens.containsKey("username")) {
            try {
                username = DigestAuthenticationMechanism.decodeRfc2231(this.convertToken("username*", responseTokens.get("username*")));
            }
            catch (UnsupportedEncodingException e) {
                throw ElytronMessages.httpDigest.mechInvalidClientMessageWithCause(e);
            }
        } else {
            throw ElytronMessages.httpDigest.mechOneOfDirectivesHasToBeDefined("username", "username*");
        }
        if (!responseTokens.containsKey("uri")) {
            throw ElytronMessages.httpDigest.mechMissingDirective("uri");
        }
        byte[] digestUri = responseTokens.get("uri");
        if (!this.digestUriMatchesRequestUri(request, digestUri)) {
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.mechResponseTokenMismatch(this.getMechanismName()), httpResponse -> httpResponse.setStatusCode(400));
            return;
        }
        if (!responseTokens.containsKey("response")) {
            throw ElytronMessages.httpDigest.mechMissingDirective("response");
        }
        byte[] response = ByteIterator.ofBytes((byte[])responseTokens.get("response")).asUtf8String().hexDecode().drain();
        String string = algorithm = responseTokens.containsKey("algorithm") ? this.convertToken("algorithm", responseTokens.get("algorithm")) : "MD5";
        if (!this.algorithm.equals(algorithm)) {
            throw ElytronMessages.httpDigest.mechUnsupportedAlgorithm(algorithm);
        }
        try {
            messageDigest = MessageDigest.getInstance(algorithm);
        }
        catch (NoSuchAlgorithmException e) {
            throw ElytronMessages.httpDigest.mechMacAlgorithmNotSupported(e);
        }
        if (!this.checkRealm(messageRealm)) {
            throw ElytronMessages.httpDigest.mechDisallowedClientRealm(messageRealm);
        }
        String selectedRealm = this.selectRealm();
        if (username.length() == 0) {
            ElytronMessages.httpDigest.trace("Failed: no username");
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.authenticationFailed(), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        byte[] hA1 = this.getH_A1(messageDigest, username, messageRealm);
        if (hA1 == null) {
            ElytronMessages.httpDigest.trace("Failed: unable to get expected proof");
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.authenticationFailed(), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        byte[] calculatedResponse = this.calculateResponseDigest(messageDigest, hA1, nonce, request.getRequestMethod(), digestUri, responseTokens.get("qop"), responseTokens.get("cnonce"), responseTokens.get("nc"));
        if (!Arrays.equals(response, calculatedResponse)) {
            ElytronMessages.httpDigest.trace("Failed: invalid proof");
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.mechResponseTokenMismatch(this.getMechanismName()), httpResponse -> this.prepareResponse(selectedRealm, httpResponse, false));
            return;
        }
        if (!nonceValid) {
            ElytronMessages.httpDigest.trace("Failed: invalid nonce");
            request.authenticationInProgress(httpResponse -> this.prepareResponse(selectedRealm, httpResponse, true));
            return;
        }
        if (this.authorize(username)) {
            ElytronMessages.httpDigest.trace("Succeed");
            this.succeed();
            if (nonceCount < 0) {
                request.authenticationComplete(new HttpServerMechanismsResponder(){

                    @Override
                    public void sendResponse(HttpServerResponse response) throws HttpAuthenticationException {
                        DigestAuthenticationMechanism.this.sendAuthenticationInfoHeader(response, salt);
                    }
                });
            } else {
                request.authenticationComplete();
            }
        } else {
            ElytronMessages.httpDigest.trace("Failed: not authorized");
            this.fail();
            request.authenticationFailed(ElytronMessages.httpDigest.authorizationFailed(username), httpResponse -> httpResponse.setStatusCode(403));
        }
    }

    private void sendAuthenticationInfoHeader(HttpServerResponse response, byte[] salt) {
        String nextNonce = this.nonceManager.generateNonce(salt);
        response.addResponseHeader("Authentication-Info", "nextnonce=\"" + nextNonce + "\"");
    }

    private boolean digestUriMatchesRequestUri(HttpServerRequest request, byte[] digestUri) {
        if (!this.validateUri) {
            return true;
        }
        URI requestURI = request.getRequestURI();
        String digestUriStr = new String(digestUri, StandardCharsets.UTF_8);
        if (requestURI.toString().equals(digestUriStr)) {
            return true;
        }
        String query = requestURI.getQuery();
        String relativeRequestUri = query == null || query.isEmpty() ? requestURI.getPath() : requestURI.getPath() + "?" + requestURI.getQuery();
        return relativeRequestUri.equals(digestUriStr);
    }

    private boolean checkRealm(String realm) throws AuthenticationMechanismException {
        String[] realms = this.getAvailableRealms();
        if (realms != null) {
            for (String current : realms) {
                if (!realm.equals(current)) continue;
                return true;
            }
        }
        return false;
    }

    private byte[] calculateResponseDigest(MessageDigest messageDigest, byte[] hA1, String nonce, String method, byte[] digestUri, byte[] qop, byte[] cnonce, byte[] nc) {
        messageDigest.update(method.getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        byte[] hA2 = messageDigest.digest(digestUri);
        messageDigest.update(ByteIterator.ofBytes((byte[])hA1).hexEncode().drainToString().getBytes(StandardCharsets.UTF_8));
        messageDigest.update((byte)58);
        messageDigest.update(nonce.getBytes(StandardCharsets.UTF_8));
        if (qop != null) {
            messageDigest.update((byte)58);
            messageDigest.update(nc);
            messageDigest.update((byte)58);
            messageDigest.update(cnonce);
            messageDigest.update((byte)58);
            messageDigest.update(qop);
        }
        messageDigest.update((byte)58);
        return messageDigest.digest(ByteIterator.ofBytes((byte[])hA2).hexEncode().drainToString().getBytes(StandardCharsets.UTF_8));
    }

    private byte[] getH_A1(MessageDigest messageDigest, String username, String messageRealm) throws AuthenticationMechanismException {
        PasswordDigestObtainer obtainer = new PasswordDigestObtainer(this.callbackHandler, username, messageRealm, ElytronMessages.httpDigest, "digest-md5", messageDigest, this.providers, null, true, false);
        return obtainer.handleUserRealmPasswordCallbacks();
    }

    private String convertToken(String name, byte[] value) throws AuthenticationMechanismException {
        if (value == null) {
            throw ElytronMessages.httpDigest.mechMissingDirective(name);
        }
        return new String(value, StandardCharsets.UTF_8);
    }

    private String selectRealm() throws HttpAuthenticationException {
        try {
            if (this.configuredRealm != null) {
                if (!this.checkRealm(this.configuredRealm)) {
                    throw ElytronMessages.httpDigest.digestMechanismInvalidRealm(this.configuredRealm);
                }
                return this.configuredRealm;
            }
            String[] realms = this.getAvailableRealms();
            if (realms != null && realms.length > 0) {
                return realms[0];
            }
            throw ElytronMessages.httpDigest.digestMechanismRequireRealm();
        }
        catch (AuthenticationMechanismException e) {
            throw e.toHttpAuthenticationException();
        }
    }

    private String[] getAvailableRealms() throws AuthenticationMechanismException {
        AvailableRealmsCallback availableRealmsCallback = new AvailableRealmsCallback();
        try {
            this.callbackHandler.handle(new Callback[]{availableRealmsCallback});
            return availableRealmsCallback.getRealmNames();
        }
        catch (UnsupportedCallbackException ignored) {
            return new String[0];
        }
        catch (AuthenticationMechanismException e) {
            throw e;
        }
        catch (IOException e) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(e);
        }
    }

    private void prepareResponse(String realmName, HttpServerResponse response, boolean stale) throws HttpAuthenticationException {
        StringBuilder sb = new StringBuilder(CHALLENGE_PREFIX);
        sb.append("realm").append("=\"").append(DigestQuote.quote(realmName)).append("\"");
        if (this.domain != null) {
            sb.append(", ").append("domain").append("=\"").append(this.domain).append("\"");
        }
        sb.append(", ").append("nonce").append("=\"").append(this.nonceManager.generateNonce(realmName.getBytes(StandardCharsets.UTF_8))).append("\"");
        sb.append(", ").append("opaque").append("=\"").append(OPAQUE_VALUE).append("\"");
        if (stale) {
            sb.append(", ").append("stale").append("=true");
        }
        sb.append(", ").append("algorithm").append("=").append(this.algorithm);
        sb.append(", ").append("qop").append("=").append("auth");
        response.addResponseHeader("WWW-Authenticate", sb.toString());
        response.setStatusCode(401);
    }

    private boolean authorize(String username) throws AuthenticationMechanismException {
        AuthorizeCallback authorizeCallback = new AuthorizeCallback(username, username);
        try {
            this.callbackHandler.handle(new Callback[]{authorizeCallback});
            return authorizeCallback.isAuthorized();
        }
        catch (UnsupportedCallbackException e) {
            return false;
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }

    private void succeed() throws AuthenticationMechanismException {
        try {
            this.callbackHandler.handle(new Callback[]{AuthenticationCompleteCallback.SUCCEEDED});
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }

    private void fail() throws AuthenticationMechanismException {
        try {
            this.callbackHandler.handle(new Callback[]{AuthenticationCompleteCallback.FAILED});
        }
        catch (Throwable t) {
            throw ElytronMessages.httpDigest.mechCallbackHandlerFailedForUnknownReason(t);
        }
    }

    private static String decodeRfc2231(String encoded) throws UnsupportedEncodingException {
        int charsetEnd = encoded.indexOf(39);
        int languageEnd = encoded.indexOf(39, charsetEnd + 1);
        String charset = encoded.substring(0, charsetEnd);
        return URLDecoder.decode(encoded.substring(languageEnd + 1), charset);
    }
}

