/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.http.oidc;

import java.nio.charset.StandardCharsets;
import java.security.AccessController;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.PrivilegedAction;
import java.util.Arrays;
import javax.crypto.SecretKey;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jwt.JwtClaims;
import org.jose4j.jwt.MalformedClaimException;
import org.jose4j.jwt.consumer.ErrorCodeValidator;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwt.consumer.JwtContext;
import org.jose4j.jwx.JsonWebStructure;
import org.wildfly.common.Assert;
import org.wildfly.common.iteration.ByteIterator;
import org.wildfly.security.http.oidc.AccessToken;
import org.wildfly.security.http.oidc.ClientSecretCredentialsProvider;
import org.wildfly.security.http.oidc.ElytronMessages;
import org.wildfly.security.http.oidc.IDToken;
import org.wildfly.security.http.oidc.Oidc;
import org.wildfly.security.http.oidc.OidcClientConfiguration;
import org.wildfly.security.http.oidc.OidcException;
import org.wildfly.security.http.oidc.PublicKeyLocator;
import org.wildfly.security.jose.jwk.JWKUtil;

public class TokenValidator {
    static final boolean DISABLE_TYP_CLAIM_VALIDATION_PROPERTY = AccessController.doPrivileged(new PrivilegedAction<Boolean>(){

        @Override
        public Boolean run() {
            return Boolean.parseBoolean(System.getProperty("wildfly.elytron.oidc.disable.typ.claim.validation", "false"));
        }
    });
    private static final int HEADER_INDEX = 0;
    private JwtConsumerBuilder jwtConsumerBuilder;
    private OidcClientConfiguration clientConfiguration;

    private TokenValidator(Builder builder) {
        this.jwtConsumerBuilder = builder.jwtConsumerBuilder;
        this.clientConfiguration = builder.clientConfiguration;
    }

    public VerifiedTokens parseAndVerifyToken(String idToken, String accessToken) throws OidcException {
        try {
            JwtContext idJwtContext = this.setVerificationKey(idToken, this.jwtConsumerBuilder);
            this.jwtConsumerBuilder.setExpectedAudience(new String[]{this.clientConfiguration.getResourceName()});
            this.jwtConsumerBuilder.registerValidator((ErrorCodeValidator)new AzpValidator(this.clientConfiguration.getResourceName()));
            this.jwtConsumerBuilder.registerValidator((ErrorCodeValidator)new AtHashValidator(accessToken, this.clientConfiguration.getTokenSignatureAlgorithm()));
            this.jwtConsumerBuilder.build().processContext(idJwtContext);
            JwtClaims idJwtClaims = idJwtContext.getJwtClaims();
            if (idJwtClaims == null) {
                throw ElytronMessages.log.invalidIDTokenClaims();
            }
            JwtClaims jwtClaims = new JwtConsumerBuilder().setSkipSignatureVerification().setSkipAllValidators().build().processToClaims(accessToken);
            return new VerifiedTokens(new IDToken(idJwtClaims), new AccessToken(jwtClaims));
        }
        catch (InvalidJwtException e) {
            ElytronMessages.log.tracef("Problem parsing ID token: " + idToken, (Object)e);
            throw ElytronMessages.log.invalidIDToken(e);
        }
    }

    public AccessToken parseAndVerifyToken(String bearerToken) throws OidcException {
        try {
            JwtContext jwtContext = this.setVerificationKey(bearerToken, this.jwtConsumerBuilder);
            this.jwtConsumerBuilder.setRequireSubject();
            if (!DISABLE_TYP_CLAIM_VALIDATION_PROPERTY) {
                this.jwtConsumerBuilder.registerValidator((ErrorCodeValidator)new TypeValidator("Bearer"));
            }
            if (this.clientConfiguration.isVerifyTokenAudience()) {
                this.jwtConsumerBuilder.setExpectedAudience(new String[]{this.clientConfiguration.getResourceName()});
            } else {
                this.jwtConsumerBuilder.setSkipDefaultAudienceValidation();
            }
            this.jwtConsumerBuilder.build().processContext(jwtContext);
            JwtClaims jwtClaims = jwtContext.getJwtClaims();
            if (jwtClaims == null) {
                throw ElytronMessages.log.invalidBearerTokenClaims();
            }
            return new AccessToken(jwtClaims);
        }
        catch (InvalidJwtException e) {
            ElytronMessages.log.tracef("Problem parsing bearer token: " + bearerToken, (Object)e);
            throw ElytronMessages.log.invalidBearerToken(e);
        }
    }

    private JwtContext setVerificationKey(String token, JwtConsumerBuilder jwtConsumerBuilder) throws InvalidJwtException {
        JwtConsumer firstPassJwtConsumer = new JwtConsumerBuilder().setSkipAllValidators().setDisableRequireSignature().setSkipSignatureVerification().build();
        JwtContext jwtContext = firstPassJwtConsumer.process(token);
        String kid = ((JsonWebStructure)jwtContext.getJoseObjects().get(0)).getKeyIdHeaderValue();
        if (kid != null && this.clientConfiguration.getPublicKeyLocator() != null) {
            jwtConsumerBuilder.setVerificationKey((Key)this.clientConfiguration.getPublicKeyLocator().getPublicKey(kid, this.clientConfiguration));
        } else {
            ClientSecretCredentialsProvider clientSecretCredentialsProvider = (ClientSecretCredentialsProvider)this.clientConfiguration.getClientAuthenticator();
            jwtConsumerBuilder.setVerificationKey((Key)clientSecretCredentialsProvider.getClientSecret());
        }
        return jwtContext;
    }

    public static Builder builder(OidcClientConfiguration clientConfiguration) {
        return new Builder(clientConfiguration);
    }

    public static class VerifiedTokens {
        private final AccessToken accessToken;
        private final IDToken idToken;

        public VerifiedTokens(IDToken idToken, AccessToken accessToken) {
            this.idToken = idToken;
            this.accessToken = accessToken;
        }

        public AccessToken getAccessToken() {
            return this.accessToken;
        }

        public IDToken getIdToken() {
            return this.idToken;
        }
    }

    private static class TypeValidator
    implements ErrorCodeValidator {
        public static final String TYPE = "typ";
        private final String expectedType;

        public TypeValidator(String expectedType) {
            this.expectedType = expectedType;
        }

        public ErrorCodeValidator.Error validate(JwtContext jwtContext) throws MalformedClaimException {
            JwtClaims jwtClaims = jwtContext.getJwtClaims();
            boolean valid = false;
            if (jwtClaims.hasClaim(TYPE)) {
                valid = jwtClaims.getStringClaimValue(TYPE).equals(this.expectedType);
            }
            if (!valid) {
                return new ErrorCodeValidator.Error(-3, ElytronMessages.log.unexpectedValueForTypeClaim());
            }
            return null;
        }
    }

    private static class AtHashValidator
    implements ErrorCodeValidator {
        private final String accessTokenString;
        private final String jwsAlgorithm;

        public AtHashValidator(String accessTokenString, String jwsAlgorithm) {
            this.accessTokenString = accessTokenString;
            this.jwsAlgorithm = jwsAlgorithm;
        }

        public ErrorCodeValidator.Error validate(JwtContext jwtContext) throws MalformedClaimException {
            JwtClaims jwtClaims = jwtContext.getJwtClaims();
            boolean valid = true;
            if (jwtClaims.hasClaim("at_hash")) {
                String atHash = jwtClaims.getStringClaimValue("at_hash");
                try {
                    String actualHash = AtHashValidator.getAccessTokenHash(this.accessTokenString, this.jwsAlgorithm);
                    valid = atHash.equals(actualHash);
                }
                catch (Exception e) {
                    valid = false;
                }
            }
            if (!valid) {
                return new ErrorCodeValidator.Error(-2, ElytronMessages.log.unexpectedValueForAtHashClaim());
            }
            return null;
        }

        private static String getAccessTokenHash(String accessTokenString, String jwsAlgorithm) throws NoSuchAlgorithmException {
            byte[] inputBytes = accessTokenString.getBytes(StandardCharsets.UTF_8);
            String javaAlgName = Oidc.getJavaAlgorithmForHash(jwsAlgorithm);
            MessageDigest md = MessageDigest.getInstance(javaAlgName);
            md.update(inputBytes);
            byte[] hash = md.digest();
            int hashLength = hash.length / 2;
            byte[] hashInput = Arrays.copyOf(hash, hashLength);
            return ByteIterator.ofBytes((byte[])hashInput).base64Encode(JWKUtil.BASE64_URL, false).drainToString();
        }
    }

    private static class AzpValidator
    implements ErrorCodeValidator {
        public static final String AZP = "azp";
        private final String issuedFor;

        public AzpValidator(String issuedFor) {
            this.issuedFor = issuedFor;
        }

        public ErrorCodeValidator.Error validate(JwtContext jwtContext) throws MalformedClaimException {
            JwtClaims jwtClaims = jwtContext.getJwtClaims();
            boolean valid = false;
            if (jwtClaims.getAudience().size() > 1) {
                if (jwtClaims.hasClaim(AZP)) {
                    String azpValue = jwtClaims.getStringClaimValue(AZP);
                    valid = azpValue != null && jwtClaims.getClaimValueAsString(AZP).equals(this.issuedFor);
                }
            } else {
                valid = true;
            }
            if (!valid) {
                return new ErrorCodeValidator.Error(-1, ElytronMessages.log.unexpectedValueForIssuedForClaim());
            }
            return null;
        }
    }

    public static class Builder {
        private OidcClientConfiguration clientConfiguration;
        private String expectedIssuer;
        private String clientId;
        private String expectedJwsAlgorithm;
        private PublicKeyLocator publicKeyLocator;
        private SecretKey clientSecretKey;
        private JwtConsumerBuilder jwtConsumerBuilder;

        Builder(OidcClientConfiguration clientConfiguration) {
            Assert.checkNotNullParam((String)"clientConfiguration", (Object)clientConfiguration);
            this.clientConfiguration = clientConfiguration;
        }

        public TokenValidator build() throws IllegalArgumentException {
            this.expectedIssuer = this.clientConfiguration.getIssuerUrl();
            if (this.expectedIssuer == null || this.expectedIssuer.length() == 0) {
                throw ElytronMessages.log.noExpectedIssuerGiven();
            }
            this.clientId = this.clientConfiguration.getResourceName();
            if (this.clientId == null || this.clientId.length() == 0) {
                throw ElytronMessages.log.noClientIDGiven();
            }
            this.expectedJwsAlgorithm = this.clientConfiguration.getTokenSignatureAlgorithm();
            if (this.expectedJwsAlgorithm == null || this.expectedJwsAlgorithm.length() == 0) {
                throw ElytronMessages.log.noExpectedJwsAlgorithmGiven();
            }
            this.publicKeyLocator = this.clientConfiguration.getPublicKeyLocator();
            if (this.clientConfiguration.getClientAuthenticator() instanceof ClientSecretCredentialsProvider) {
                ClientSecretCredentialsProvider clientSecretCredentialsProvider = (ClientSecretCredentialsProvider)this.clientConfiguration.getClientAuthenticator();
                this.clientSecretKey = clientSecretCredentialsProvider.getClientSecret();
            }
            if (this.publicKeyLocator == null && this.clientSecretKey == null) {
                throw ElytronMessages.log.noJwksPublicKeyOrClientSecretKeyGiven();
            }
            this.jwtConsumerBuilder = new JwtConsumerBuilder().setExpectedIssuer(this.expectedIssuer).setJwsAlgorithmConstraints(new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, new String[]{this.expectedJwsAlgorithm})).setRequireExpirationTime();
            return new TokenValidator(this);
        }
    }
}

