/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.util.Collections;
import java.util.Locale;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final Set<String> allowedQueryParameterNames;
    private final boolean prettyPrintBody;
    public static final String RETRY_COUNT_CONTEXT = "requestRetryCount";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.NONE;
            this.allowedHeaderNames = Collections.emptySet();
            this.allowedQueryParameterNames = Collections.emptySet();
            this.prettyPrintBody = false;
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHeaderNames().stream().map(headerName -> headerName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.allowedQueryParameterNames = httpLogOptions.getAllowedQueryParamNames().stream().map(queryParamName -> queryParamName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.prettyPrintBody = httpLogOptions.isPrettyPrintBody();
        }
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        if (this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
            return next.process();
        }
        ClientLogger logger = new ClientLogger((String)context.getData("caller-method").orElse(""));
        long startNs = System.nanoTime();
        return this.logRequest(logger, context.getHttpRequest(), context.getData(RETRY_COUNT_CONTEXT)).then(next.process()).flatMap(response -> this.logResponse(logger, (HttpResponse)response, startNs)).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
    }

    private Mono<Void> logRequest(ClientLogger logger, HttpRequest request, Optional<Object> optionalRetryCount) {
        long contentLength;
        if (!logger.canLogAtLevel(LogLevel.INFORMATIONAL)) {
            return Mono.empty();
        }
        StringBuilder requestLogMessage = new StringBuilder();
        if (this.httpLogDetailLevel.shouldLogUrl()) {
            requestLogMessage.append("--> ").append((Object)request.getHttpMethod()).append(" ").append(this.getRedactedUrl(request.getUrl())).append(System.lineSeparator());
            optionalRetryCount.ifPresent(o -> requestLogMessage.append("Try count: ").append(o).append(System.lineSeparator()));
        }
        this.addHeadersToLogMessage(logger, request.getHeaders(), requestLogMessage);
        if (!this.httpLogDetailLevel.shouldLogBody()) {
            return this.logAndReturn(logger, requestLogMessage, null);
        }
        if (request.getBody() == null) {
            requestLogMessage.append("(empty body)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
            return this.logAndReturn(logger, requestLogMessage, null);
        }
        String contentType = request.getHeaders().getValue("Content-Type");
        if (this.shouldBodyBeLogged(contentType, contentLength = this.getContentLength(logger, request.getHeaders()))) {
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
            request.setBody(request.getBody().flatMap(byteBuffer -> HttpLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, byteBuffer)).doFinally(ignored -> {
                requestLogMessage.append(contentLength).append("-byte body:").append(System.lineSeparator()).append(this.prettyPrintIfNeeded(logger, contentType, HttpLoggingPolicy.convertStreamToString(outputStream, logger))).append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                logger.info(requestLogMessage.toString());
            }));
            return Mono.empty();
        }
        requestLogMessage.append(contentLength).append("-byte body: (content not logged)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
        return this.logAndReturn(logger, requestLogMessage, null);
    }

    private Mono<HttpResponse> logResponse(ClientLogger logger, HttpResponse response, long startNs) {
        long contentLength;
        if (!logger.canLogAtLevel(LogLevel.INFORMATIONAL)) {
            return Mono.just(response);
        }
        long tookMs = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNs);
        String contentLengthString = response.getHeaderValue("Content-Length");
        String bodySize = CoreUtils.isNullOrEmpty(contentLengthString) ? "unknown-length body" : contentLengthString + "-byte body";
        StringBuilder responseLogMessage = new StringBuilder();
        if (this.httpLogDetailLevel.shouldLogUrl()) {
            responseLogMessage.append("<-- ").append(response.getStatusCode()).append(" ").append(this.getRedactedUrl(response.getRequest().getUrl())).append(" (").append(tookMs).append(" ms, ").append(bodySize).append(")").append(System.lineSeparator());
        }
        this.addHeadersToLogMessage(logger, response.getHeaders(), responseLogMessage);
        if (!this.httpLogDetailLevel.shouldLogBody()) {
            responseLogMessage.append("<-- END HTTP");
            return this.logAndReturn(logger, responseLogMessage, response);
        }
        String contentTypeHeader = response.getHeaderValue("Content-Type");
        if (this.shouldBodyBeLogged(contentTypeHeader, contentLength = this.getContentLength(logger, response.getHeaders()))) {
            HttpResponse bufferedResponse = response.buffer();
            ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
            WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
            return bufferedResponse.getBody().flatMap(byteBuffer -> HttpLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, byteBuffer)).doFinally(ignored -> {
                responseLogMessage.append("Response body:").append(System.lineSeparator()).append(this.prettyPrintIfNeeded(logger, contentTypeHeader, HttpLoggingPolicy.convertStreamToString(outputStream, logger))).append(System.lineSeparator()).append("<-- END HTTP");
                logger.info(responseLogMessage.toString());
            }).then(Mono.just(bufferedResponse));
        }
        responseLogMessage.append("(body content not logged)").append(System.lineSeparator()).append("<-- END HTTP");
        return this.logAndReturn(logger, responseLogMessage, response);
    }

    private <T> Mono<T> logAndReturn(ClientLogger logger, StringBuilder logMessageBuilder, T data) {
        logger.info(logMessageBuilder.toString());
        return Mono.justOrEmpty(data);
    }

    private String getRedactedUrl(URL url) {
        return UrlBuilder.parse(url).setQuery(this.getAllowedQueryString(url.getQuery())).toString();
    }

    private String getAllowedQueryString(String queryString) {
        String[] queryParams;
        if (CoreUtils.isNullOrEmpty(queryString)) {
            return "";
        }
        StringBuilder queryStringBuilder = new StringBuilder();
        for (String queryParam : queryParams = queryString.split("&")) {
            String[] queryPair;
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }
            if ((queryPair = queryParam.split("=", 2)).length == 2) {
                String queryName = queryPair[0];
                if (this.allowedQueryParameterNames.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                    continue;
                }
                queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                continue;
            }
            queryStringBuilder.append(queryParam);
        }
        return queryStringBuilder.toString();
    }

    private void addHeadersToLogMessage(ClientLogger logger, HttpHeaders headers, StringBuilder sb) {
        if (!this.httpLogDetailLevel.shouldLogHeaders() || !logger.canLogAtLevel(LogLevel.VERBOSE)) {
            return;
        }
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (this.allowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private String prettyPrintIfNeeded(ClientLogger logger, String contentType, String body) {
        String result = body;
        if (this.prettyPrintBody && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString(deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0L;
        String contentLengthString = headers.getValue("Content-Length");
        if (CoreUtils.isNullOrEmpty(contentLengthString)) {
            return contentLength;
        }
        try {
            contentLength = Long.parseLong(contentLengthString);
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.", headers.getValue("content-length"), e);
        }
        return contentLength;
    }

    private boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L;
    }

    private static String convertStreamToString(ByteArrayOutputStream stream, ClientLogger logger) {
        try {
            return stream.toString("UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            throw logger.logExceptionAsError(new RuntimeException(ex));
        }
    }

    private static Mono<ByteBuffer> writeBufferToBodyStream(WritableByteChannel channel, ByteBuffer byteBuffer) {
        try {
            channel.write(byteBuffer.duplicate());
            return Mono.just(byteBuffer);
        }
        catch (IOException ex) {
            return Mono.error(ex);
        }
    }
}

