/*
 * Decompiled with CFR 0.152.
 */
package com.azure.messaging.eventhubs.implementation;

import com.azure.core.credential.AccessToken;
import com.azure.core.credential.TokenCredential;
import com.azure.core.credential.TokenRequestContext;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.logging.ClientLogger;
import com.azure.messaging.eventhubs.implementation.ClientConstants;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.time.Instant;
import java.time.OffsetDateTime;
import java.time.ZoneOffset;
import java.util.Arrays;
import java.util.Base64;
import java.util.Locale;
import java.util.Objects;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import reactor.core.publisher.Mono;

public class EventHubSharedKeyCredential
implements TokenCredential {
    private static final String SHARED_ACCESS_SIGNATURE_FORMAT = "SharedAccessSignature sr=%s&sig=%s&se=%s&skn=%s";
    private static final String HASH_ALGORITHM = "HMACSHA256";
    private final ClientLogger logger = new ClientLogger(EventHubSharedKeyCredential.class);
    private final String policyName;
    private final Duration tokenValidity;
    private final SecretKeySpec secretKeySpec;
    private final String sharedAccessSignature;

    public EventHubSharedKeyCredential(String policyName, String sharedAccessKey) {
        this(policyName, sharedAccessKey, ClientConstants.TOKEN_VALIDITY);
    }

    public EventHubSharedKeyCredential(String policyName, String sharedAccessKey, Duration tokenValidity) {
        Objects.requireNonNull(sharedAccessKey, "'sharedAccessKey' cannot be null.");
        this.policyName = Objects.requireNonNull(policyName, "'sharedAccessKey' cannot be null.");
        this.tokenValidity = Objects.requireNonNull(tokenValidity, "'tokenValidity' cannot be null.");
        if (policyName.isEmpty()) {
            throw new IllegalArgumentException("'policyName' cannot be an empty string.");
        }
        if (sharedAccessKey.isEmpty()) {
            throw new IllegalArgumentException("'sharedAccessKey' cannot be an empty string.");
        }
        if (tokenValidity.isZero() || tokenValidity.isNegative()) {
            throw new IllegalArgumentException("'tokenTimeToLive' has to positive and in the order-of seconds");
        }
        byte[] sasKeyBytes = sharedAccessKey.getBytes(StandardCharsets.UTF_8);
        this.secretKeySpec = new SecretKeySpec(sasKeyBytes, HASH_ALGORITHM);
        this.sharedAccessSignature = null;
    }

    public EventHubSharedKeyCredential(String sharedAccessSignature) {
        this.sharedAccessSignature = Objects.requireNonNull(sharedAccessSignature, "'sharedAccessSignature' cannot be null");
        this.policyName = null;
        this.secretKeySpec = null;
        this.tokenValidity = null;
    }

    @Override
    public Mono<AccessToken> getToken(TokenRequestContext request) {
        if (request.getScopes().size() != 1) {
            throw this.logger.logExceptionAsError(new IllegalArgumentException("'scopes' should only contain a single argument that is the token audience or resource name."));
        }
        return Mono.fromCallable(() -> this.generateSharedAccessSignature(request.getScopes().get(0)));
    }

    private AccessToken generateSharedAccessSignature(String resource) throws UnsupportedEncodingException {
        Mac hmac;
        if (CoreUtils.isNullOrEmpty(resource)) {
            throw this.logger.logExceptionAsError(new IllegalArgumentException("resource cannot be empty"));
        }
        if (this.sharedAccessSignature != null) {
            return new AccessToken(this.sharedAccessSignature, this.getExpirationTime(this.sharedAccessSignature));
        }
        try {
            hmac = Mac.getInstance(HASH_ALGORITHM);
            hmac.init(this.secretKeySpec);
        }
        catch (NoSuchAlgorithmException e) {
            throw this.logger.logExceptionAsError(new UnsupportedOperationException(String.format("Unable to create hashing algorithm '%s'", HASH_ALGORITHM), e));
        }
        catch (InvalidKeyException e) {
            throw this.logger.logExceptionAsError(new IllegalArgumentException("'sharedAccessKey' is an invalid value for the hashing algorithm.", e));
        }
        String utf8Encoding = StandardCharsets.UTF_8.name();
        OffsetDateTime expiresOn = OffsetDateTime.now(ZoneOffset.UTC).plus(this.tokenValidity);
        String expiresOnEpochSeconds = Long.toString(expiresOn.toEpochSecond());
        String audienceUri = URLEncoder.encode(resource, utf8Encoding);
        String secretToSign = audienceUri + "\n" + expiresOnEpochSeconds;
        byte[] signatureBytes = hmac.doFinal(secretToSign.getBytes(utf8Encoding));
        String signature = Base64.getEncoder().encodeToString(signatureBytes);
        String token = String.format(Locale.US, SHARED_ACCESS_SIGNATURE_FORMAT, audienceUri, URLEncoder.encode(signature, utf8Encoding), URLEncoder.encode(expiresOnEpochSeconds, utf8Encoding), URLEncoder.encode(this.policyName, utf8Encoding));
        return new AccessToken(token, expiresOn);
    }

    private OffsetDateTime getExpirationTime(String sharedAccessSignature) {
        String[] parts = sharedAccessSignature.split("&");
        return Arrays.stream(parts).map(part -> part.split("=")).filter(pair -> ((String[])pair).length == 2 && pair[0].equalsIgnoreCase("se")).findFirst().map(pair -> pair[1]).map(expirationTimeStr -> {
            try {
                long epochSeconds = Long.parseLong(expirationTimeStr);
                return Instant.ofEpochSecond(epochSeconds).atOffset(ZoneOffset.UTC);
            }
            catch (NumberFormatException exception) {
                this.logger.verbose("Invalid expiration time format in the SAS token: {}. Falling back to max expiration time.", expirationTimeStr);
                return OffsetDateTime.MAX;
            }
        }).orElse(OffsetDateTime.MAX);
    }
}

