/*
 * Decompiled with CFR 0.152.
 */
package org.apache.lucene.util.fst;

import java.io.IOException;
import org.apache.lucene.store.ByteArrayDataOutput;
import org.apache.lucene.store.DataOutput;
import org.apache.lucene.util.ArrayUtil;
import org.apache.lucene.util.IntsRef;
import org.apache.lucene.util.IntsRefBuilder;
import org.apache.lucene.util.fst.BytesStore;
import org.apache.lucene.util.fst.FST;
import org.apache.lucene.util.fst.NodeHash;
import org.apache.lucene.util.fst.Outputs;

public class FSTCompiler<T> {
    static final float DIRECT_ADDRESSING_MAX_OVERSIZING_FACTOR = 1.0f;
    static final int FIXED_LENGTH_ARC_SHALLOW_DEPTH = 3;
    static final int FIXED_LENGTH_ARC_SHALLOW_NUM_ARCS = 5;
    static final int FIXED_LENGTH_ARC_DEEP_NUM_ARCS = 10;
    private static final float DIRECT_ADDRESSING_MAX_OVERSIZE_WITH_CREDIT_FACTOR = 1.66f;
    private final NodeHash<T> dedupHash;
    final FST<T> fst;
    private final T NO_OUTPUT;
    private final IntsRefBuilder lastInput = new IntsRefBuilder();
    private UnCompiledNode<T>[] frontier;
    long lastFrozenNode;
    int[] numBytesPerArc = new int[4];
    int[] numLabelBytesPerArc = new int[this.numBytesPerArc.length];
    final FixedLengthArcsBuffer fixedLengthArcsBuffer = new FixedLengthArcsBuffer();
    long arcCount;
    long nodeCount;
    long binarySearchNodeCount;
    long directAddressingNodeCount;
    long continuousNodeCount;
    final boolean allowFixedLengthArcs;
    final float directAddressingMaxOversizingFactor;
    final int version;
    long directAddressingExpansionCredit;
    final BytesStore bytes;

    private FSTCompiler(FST.INPUT_TYPE inputType, double suffixRAMLimitMB, Outputs<T> outputs, boolean allowFixedLengthArcs, int bytesPageBits, float directAddressingMaxOversizingFactor, int version) {
        this.allowFixedLengthArcs = allowFixedLengthArcs;
        this.directAddressingMaxOversizingFactor = directAddressingMaxOversizingFactor;
        this.version = version;
        this.bytes = new BytesStore(bytesPageBits);
        this.bytes.writeByte((byte)0);
        this.fst = new FST<Object>(new FST.FSTMetadata<Object>(inputType, outputs, null, -1L, version, 0L), this.bytes);
        if (suffixRAMLimitMB < 0.0) {
            throw new IllegalArgumentException("ramLimitMB must be >= 0; got: " + suffixRAMLimitMB);
        }
        this.dedupHash = suffixRAMLimitMB > 0.0 ? new NodeHash(this, suffixRAMLimitMB) : null;
        this.NO_OUTPUT = outputs.getNoOutput();
        UnCompiledNode[] f = new UnCompiledNode[10];
        this.frontier = f;
        for (int idx = 0; idx < this.frontier.length; ++idx) {
            this.frontier[idx] = new UnCompiledNode(this, idx);
        }
    }

    public float getDirectAddressingMaxOversizingFactor() {
        return this.directAddressingMaxOversizingFactor;
    }

    public long getNodeCount() {
        return 1L + this.nodeCount;
    }

    public long getArcCount() {
        return this.arcCount;
    }

    public long getMappedStateCount() {
        return this.dedupHash == null ? 0L : this.nodeCount;
    }

    private CompiledNode compileNode(UnCompiledNode<T> nodeIn, int tailLength) throws IOException {
        long node;
        long bytesPosStart = this.bytes.getPosition();
        if (this.dedupHash != null) {
            if (nodeIn.numArcs == 0) {
                this.lastFrozenNode = node = this.addNode(nodeIn);
            } else {
                node = this.dedupHash.add(nodeIn);
            }
        } else {
            node = this.addNode(nodeIn);
        }
        assert (node != -2L);
        long bytesPosEnd = this.bytes.getPosition();
        if (bytesPosEnd != bytesPosStart) {
            assert (bytesPosEnd > bytesPosStart);
            this.lastFrozenNode = node;
        }
        nodeIn.clear();
        CompiledNode fn = new CompiledNode();
        fn.node = node;
        return fn;
    }

    long addNode(UnCompiledNode<T> nodeIn) throws IOException {
        if (nodeIn.numArcs == 0) {
            if (nodeIn.isFinal) {
                return -1L;
            }
            return 0L;
        }
        long startAddress = this.bytes.getPosition();
        boolean doFixedLengthArcs = this.shouldExpandNodeWithFixedLengthArcs(nodeIn);
        if (doFixedLengthArcs && this.numBytesPerArc.length < nodeIn.numArcs) {
            this.numBytesPerArc = new int[ArrayUtil.oversize(nodeIn.numArcs, 4)];
            this.numLabelBytesPerArc = new int[this.numBytesPerArc.length];
        }
        this.arcCount += (long)nodeIn.numArcs;
        int lastArc = nodeIn.numArcs - 1;
        long lastArcStart = this.bytes.getPosition();
        int maxBytesPerArc = 0;
        int maxBytesPerArcWithoutLabel = 0;
        for (int arcIdx = 0; arcIdx < nodeIn.numArcs; ++arcIdx) {
            int numArcBytes;
            boolean targetHasArcs;
            Arc arc = nodeIn.arcs[arcIdx];
            CompiledNode target = (CompiledNode)arc.target;
            int flags = 0;
            if (arcIdx == lastArc) {
                flags += 2;
            }
            if (this.lastFrozenNode == target.node && !doFixedLengthArcs) {
                flags += 4;
            }
            if (arc.isFinal) {
                ++flags;
                if (arc.nextFinalOutput != this.NO_OUTPUT) {
                    flags += 32;
                }
            } else assert (arc.nextFinalOutput == this.NO_OUTPUT);
            boolean bl = targetHasArcs = target.node > 0L;
            if (!targetHasArcs) {
                flags += 8;
            }
            if (arc.output != this.NO_OUTPUT) {
                flags += 16;
            }
            this.bytes.writeByte((byte)flags);
            long labelStart = this.bytes.getPosition();
            this.writeLabel(this.bytes, arc.label);
            int numLabelBytes = (int)(this.bytes.getPosition() - labelStart);
            if (arc.output != this.NO_OUTPUT) {
                this.fst.outputs.write(arc.output, this.bytes);
            }
            if (arc.nextFinalOutput != this.NO_OUTPUT) {
                this.fst.outputs.writeFinalOutput(arc.nextFinalOutput, this.bytes);
            }
            if (targetHasArcs && (flags & 4) == 0) {
                assert (target.node > 0L);
                this.bytes.writeVLong(target.node);
            }
            if (!doFixedLengthArcs) continue;
            this.numBytesPerArc[arcIdx] = numArcBytes = (int)(this.bytes.getPosition() - lastArcStart);
            this.numLabelBytesPerArc[arcIdx] = numLabelBytes;
            lastArcStart = this.bytes.getPosition();
            maxBytesPerArc = Math.max(maxBytesPerArc, numArcBytes);
            maxBytesPerArcWithoutLabel = Math.max(maxBytesPerArcWithoutLabel, numArcBytes - numLabelBytes);
        }
        if (doFixedLengthArcs) {
            boolean continuousLabel;
            assert (maxBytesPerArc > 0);
            int labelRange = nodeIn.arcs[nodeIn.numArcs - 1].label - nodeIn.arcs[0].label + 1;
            assert (labelRange > 0);
            boolean bl = continuousLabel = labelRange == nodeIn.numArcs;
            if (continuousLabel && this.version >= 9) {
                this.writeNodeForDirectAddressingOrContinuous(nodeIn, startAddress, maxBytesPerArcWithoutLabel, labelRange, true);
                ++this.continuousNodeCount;
            } else if (this.shouldExpandNodeWithDirectAddressing(nodeIn, maxBytesPerArc, maxBytesPerArcWithoutLabel, labelRange)) {
                this.writeNodeForDirectAddressingOrContinuous(nodeIn, startAddress, maxBytesPerArcWithoutLabel, labelRange, false);
                ++this.directAddressingNodeCount;
            } else {
                this.writeNodeForBinarySearch(nodeIn, startAddress, maxBytesPerArc);
                ++this.binarySearchNodeCount;
            }
        }
        long thisNodeAddress = this.bytes.getPosition() - 1L;
        this.bytes.reverse(startAddress, thisNodeAddress);
        ++this.nodeCount;
        return thisNodeAddress;
    }

    private void writeLabel(DataOutput out, int v) throws IOException {
        assert (v >= 0) : "v=" + v;
        if (this.fst.metadata.inputType == FST.INPUT_TYPE.BYTE1) {
            assert (v <= 255) : "v=" + v;
            out.writeByte((byte)v);
        } else if (this.fst.metadata.inputType == FST.INPUT_TYPE.BYTE2) {
            assert (v <= 65535) : "v=" + v;
            out.writeShort((short)v);
        } else {
            out.writeVInt(v);
        }
    }

    private boolean shouldExpandNodeWithFixedLengthArcs(UnCompiledNode<T> node) {
        return this.allowFixedLengthArcs && (node.depth <= 3 && node.numArcs >= 5 || node.numArcs >= 10);
    }

    private boolean shouldExpandNodeWithDirectAddressing(UnCompiledNode<T> nodeIn, int numBytesPerArc, int maxBytesPerArcWithoutLabel, int labelRange) {
        int allowedOversize;
        int sizeForBinarySearch = numBytesPerArc * nodeIn.numArcs;
        int sizeForDirectAddressing = FST.getNumPresenceBytes(labelRange) + this.numLabelBytesPerArc[0] + maxBytesPerArcWithoutLabel * nodeIn.numArcs;
        int expansionCost = sizeForDirectAddressing - (allowedOversize = (int)((float)sizeForBinarySearch * this.getDirectAddressingMaxOversizingFactor()));
        if (expansionCost <= 0 || this.directAddressingExpansionCredit >= (long)expansionCost && (float)sizeForDirectAddressing <= (float)allowedOversize * 1.66f) {
            this.directAddressingExpansionCredit -= (long)expansionCost;
            return true;
        }
        return false;
    }

    private void writeNodeForBinarySearch(UnCompiledNode<T> nodeIn, long startAddress, int maxBytesPerArc) {
        this.fixedLengthArcsBuffer.resetPosition().writeByte((byte)32).writeVInt(nodeIn.numArcs).writeVInt(maxBytesPerArc);
        int headerLen = this.fixedLengthArcsBuffer.getPosition();
        long srcPos = this.bytes.getPosition();
        long destPos = startAddress + (long)headerLen + (long)nodeIn.numArcs * (long)maxBytesPerArc;
        assert (destPos >= srcPos);
        if (destPos > srcPos) {
            this.bytes.setPosition(destPos);
            for (int arcIdx = nodeIn.numArcs - 1; arcIdx >= 0; --arcIdx) {
                int arcLen = this.numBytesPerArc[arcIdx];
                if ((srcPos -= (long)arcLen) == (destPos -= (long)maxBytesPerArc)) continue;
                assert (destPos > srcPos) : "destPos=" + destPos + " srcPos=" + srcPos + " arcIdx=" + arcIdx + " maxBytesPerArc=" + maxBytesPerArc + " arcLen=" + arcLen + " nodeIn.numArcs=" + nodeIn.numArcs;
                this.bytes.copyBytes(srcPos, destPos, arcLen);
            }
        }
        this.bytes.writeBytes(startAddress, this.fixedLengthArcsBuffer.getBytes(), 0, headerLen);
    }

    private void writeNodeForDirectAddressingOrContinuous(UnCompiledNode<T> nodeIn, long startAddress, int maxBytesPerArcWithoutLabel, int labelRange, boolean continuous) {
        int headerMaxLen = 11;
        int numPresenceBytes = continuous ? 0 : FST.getNumPresenceBytes(labelRange);
        long srcPos = this.bytes.getPosition();
        int totalArcBytes = this.numLabelBytesPerArc[0] + nodeIn.numArcs * maxBytesPerArcWithoutLabel;
        int bufferOffset = headerMaxLen + numPresenceBytes + totalArcBytes;
        byte[] buffer = this.fixedLengthArcsBuffer.ensureCapacity(bufferOffset).getBytes();
        for (int arcIdx = nodeIn.numArcs - 1; arcIdx >= 0; --arcIdx) {
            int srcArcLen = this.numBytesPerArc[arcIdx];
            int labelLen = this.numLabelBytesPerArc[arcIdx];
            this.bytes.writeTo(srcPos -= (long)srcArcLen, buffer, bufferOffset -= maxBytesPerArcWithoutLabel, 1);
            int remainingArcLen = srcArcLen - 1 - labelLen;
            if (remainingArcLen != 0) {
                this.bytes.writeTo(srcPos + 1L + (long)labelLen, buffer, bufferOffset + 1, remainingArcLen);
            }
            if (arcIdx != 0) continue;
            this.bytes.writeTo(srcPos + 1L, buffer, bufferOffset -= labelLen, labelLen);
        }
        assert (bufferOffset == headerMaxLen + numPresenceBytes);
        this.fixedLengthArcsBuffer.resetPosition().writeByte(continuous ? (byte)96 : 64).writeVInt(labelRange).writeVInt(maxBytesPerArcWithoutLabel);
        int headerLen = this.fixedLengthArcsBuffer.getPosition();
        this.bytes.setPosition(startAddress);
        this.bytes.writeBytes(this.fixedLengthArcsBuffer.getBytes(), 0, headerLen);
        if (!continuous) {
            this.writePresenceBits(nodeIn);
            assert (this.bytes.getPosition() == startAddress + (long)headerLen + (long)numPresenceBytes);
        }
        this.bytes.writeBytes(this.fixedLengthArcsBuffer.getBytes(), bufferOffset, totalArcBytes);
        assert (this.bytes.getPosition() == startAddress + (long)headerLen + (long)numPresenceBytes + (long)totalArcBytes);
    }

    private void writePresenceBits(UnCompiledNode<T> nodeIn) {
        byte presenceBits = 1;
        int presenceIndex = 0;
        int previousLabel = nodeIn.arcs[0].label;
        for (int arcIdx = 1; arcIdx < nodeIn.numArcs; ++arcIdx) {
            int label = nodeIn.arcs[arcIdx].label;
            assert (label > previousLabel);
            presenceIndex += label - previousLabel;
            while (presenceIndex >= 8) {
                this.bytes.writeByte(presenceBits);
                presenceBits = 0;
                presenceIndex -= 8;
            }
            presenceBits = (byte)(presenceBits | 1 << presenceIndex);
            previousLabel = label;
        }
        assert (presenceIndex == (nodeIn.arcs[nodeIn.numArcs - 1].label - nodeIn.arcs[0].label) % 8);
        assert (presenceBits != 0);
        assert ((presenceBits & 1 << presenceIndex) != 0);
        this.bytes.writeByte(presenceBits);
    }

    private void freezeTail(int prefixLenPlus1) throws IOException {
        int downTo = Math.max(1, prefixLenPlus1);
        for (int idx = this.lastInput.length(); idx >= downTo; --idx) {
            UnCompiledNode<T> node = this.frontier[idx];
            UnCompiledNode parent = this.frontier[idx - 1];
            Object nextFinalOutput = node.output;
            boolean isFinal = node.isFinal || node.numArcs == 0;
            parent.replaceLast(this.lastInput.intAt(idx - 1), this.compileNode(node, 1 + this.lastInput.length() - idx), nextFinalOutput, isFinal);
        }
    }

    public void add(IntsRef input, T output) throws IOException {
        int idx;
        if (output.equals(this.NO_OUTPUT)) {
            output = this.NO_OUTPUT;
        }
        assert (this.lastInput.length() == 0 || input.compareTo(this.lastInput.get()) >= 0) : "inputs are added out of order lastInput=" + this.lastInput.get() + " vs input=" + input;
        assert (this.validOutput(output));
        if (input.length == 0) {
            this.frontier[0].isFinal = true;
            this.setEmptyOutput(output);
            return;
        }
        int pos1 = 0;
        int pos2 = input.offset;
        int pos1Stop = Math.min(this.lastInput.length(), input.length);
        while (pos1 < pos1Stop && this.lastInput.intAt(pos1) == input.ints[pos2]) {
            ++pos1;
            ++pos2;
        }
        int prefixLenPlus1 = pos1 + 1;
        if (this.frontier.length < input.length + 1) {
            UnCompiledNode<T>[] next = ArrayUtil.grow(this.frontier, input.length + 1);
            for (idx = this.frontier.length; idx < next.length; ++idx) {
                next[idx] = new UnCompiledNode(this, idx);
            }
            this.frontier = next;
        }
        this.freezeTail(prefixLenPlus1);
        for (int idx2 = prefixLenPlus1; idx2 <= input.length; ++idx2) {
            this.frontier[idx2 - 1].addArc(input.ints[input.offset + idx2 - 1], this.frontier[idx2]);
        }
        UnCompiledNode<T> lastNode = this.frontier[input.length];
        if (this.lastInput.length() != input.length || prefixLenPlus1 != input.length + 1) {
            lastNode.isFinal = true;
            lastNode.output = this.NO_OUTPUT;
        }
        for (idx = 1; idx < prefixLenPlus1; ++idx) {
            T wordSuffix;
            T commonOutputPrefix;
            UnCompiledNode<T> node = this.frontier[idx];
            UnCompiledNode<T> parentNode = this.frontier[idx - 1];
            T lastOutput = parentNode.getLastOutput(input.ints[input.offset + idx - 1]);
            assert (this.validOutput(lastOutput));
            if (lastOutput != this.NO_OUTPUT) {
                commonOutputPrefix = this.fst.outputs.common(output, lastOutput);
                assert (this.validOutput(commonOutputPrefix));
                wordSuffix = this.fst.outputs.subtract(lastOutput, commonOutputPrefix);
                assert (this.validOutput(wordSuffix));
                parentNode.setLastOutput(input.ints[input.offset + idx - 1], commonOutputPrefix);
                node.prependOutput(wordSuffix);
            } else {
                commonOutputPrefix = wordSuffix = this.NO_OUTPUT;
            }
            output = this.fst.outputs.subtract(output, commonOutputPrefix);
            assert (this.validOutput(output));
        }
        if (this.lastInput.length() == input.length && prefixLenPlus1 == 1 + input.length) {
            lastNode.output = this.fst.outputs.merge(lastNode.output, output);
        } else {
            this.frontier[prefixLenPlus1 - 1].setLastOutput(input.ints[input.offset + prefixLenPlus1 - 1], output);
        }
        this.lastInput.copyInts(input);
    }

    void setEmptyOutput(T v) {
        this.fst.metadata.emptyOutput = this.fst.metadata.emptyOutput != null ? this.fst.outputs.merge(this.fst.metadata.emptyOutput, v) : v;
    }

    void finish(long newStartNode) {
        assert (newStartNode <= this.bytes.getPosition());
        if (this.fst.metadata.startNode != -1L) {
            throw new IllegalStateException("already finished");
        }
        if (newStartNode == -1L && this.fst.metadata.emptyOutput != null) {
            newStartNode = 0L;
        }
        this.fst.metadata.startNode = newStartNode;
        this.fst.metadata.numBytes = this.bytes.getPosition();
    }

    private boolean validOutput(T output) {
        return output == this.NO_OUTPUT || !output.equals(this.NO_OUTPUT);
    }

    public FST<T> compile() throws IOException {
        UnCompiledNode<T> root = this.frontier[0];
        this.freezeTail(0);
        if (root.numArcs == 0 && this.fst.metadata.emptyOutput == null) {
            return null;
        }
        this.finish(this.compileNode(root, (int)this.lastInput.length()).node);
        this.bytes.finish();
        return this.fst;
    }

    public long fstRamBytesUsed() {
        return this.fst.ramBytesUsed();
    }

    public long fstSizeInBytes() {
        return this.bytes.getPosition();
    }

    static class FixedLengthArcsBuffer {
        private byte[] bytes = new byte[11];
        private final ByteArrayDataOutput bado = new ByteArrayDataOutput(this.bytes);

        FixedLengthArcsBuffer() {
        }

        FixedLengthArcsBuffer ensureCapacity(int capacity) {
            if (this.bytes.length < capacity) {
                this.bytes = new byte[ArrayUtil.oversize(capacity, 1)];
                this.bado.reset(this.bytes);
            }
            return this;
        }

        FixedLengthArcsBuffer resetPosition() {
            this.bado.reset(this.bytes);
            return this;
        }

        FixedLengthArcsBuffer writeByte(byte b) {
            this.bado.writeByte(b);
            return this;
        }

        FixedLengthArcsBuffer writeVInt(int i) {
            try {
                this.bado.writeVInt(i);
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
            return this;
        }

        int getPosition() {
            return this.bado.getPosition();
        }

        byte[] getBytes() {
            return this.bytes;
        }
    }

    static final class UnCompiledNode<T>
    implements Node {
        final FSTCompiler<T> owner;
        int numArcs;
        Arc<T>[] arcs;
        T output;
        boolean isFinal;
        final int depth;

        UnCompiledNode(FSTCompiler<T> owner, int depth) {
            this.owner = owner;
            this.arcs = new Arc[1];
            this.arcs[0] = new Arc();
            this.output = owner.NO_OUTPUT;
            this.depth = depth;
        }

        @Override
        public boolean isCompiled() {
            return false;
        }

        void clear() {
            this.numArcs = 0;
            this.isFinal = false;
            this.output = this.owner.NO_OUTPUT;
        }

        T getLastOutput(int labelToMatch) {
            assert (this.numArcs > 0);
            assert (this.arcs[this.numArcs - 1].label == labelToMatch);
            return this.arcs[this.numArcs - 1].output;
        }

        void addArc(int label, Node target) {
            assert (label >= 0);
            assert (this.numArcs == 0 || label > this.arcs[this.numArcs - 1].label) : "arc[numArcs-1].label=" + this.arcs[this.numArcs - 1].label + " new label=" + label + " numArcs=" + this.numArcs;
            if (this.numArcs == this.arcs.length) {
                Arc<T>[] newArcs = ArrayUtil.grow(this.arcs);
                for (int arcIdx = this.numArcs; arcIdx < newArcs.length; ++arcIdx) {
                    newArcs[arcIdx] = new Arc();
                }
                this.arcs = newArcs;
            }
            Arc<T> arc = this.arcs[this.numArcs++];
            arc.label = label;
            arc.target = target;
            arc.nextFinalOutput = this.owner.NO_OUTPUT;
            arc.output = arc.nextFinalOutput;
            arc.isFinal = false;
        }

        void replaceLast(int labelToMatch, Node target, T nextFinalOutput, boolean isFinal) {
            assert (this.numArcs > 0);
            Arc<T> arc = this.arcs[this.numArcs - 1];
            assert (arc.label == labelToMatch) : "arc.label=" + arc.label + " vs " + labelToMatch;
            arc.target = target;
            arc.nextFinalOutput = nextFinalOutput;
            arc.isFinal = isFinal;
        }

        void deleteLast(int label, Node target) {
            assert (this.numArcs > 0);
            assert (label == this.arcs[this.numArcs - 1].label);
            assert (target == this.arcs[this.numArcs - 1].target);
            --this.numArcs;
        }

        void setLastOutput(int labelToMatch, T newOutput) {
            assert (this.owner.validOutput(newOutput));
            assert (this.numArcs > 0);
            Arc<T> arc = this.arcs[this.numArcs - 1];
            assert (arc.label == labelToMatch);
            arc.output = newOutput;
        }

        void prependOutput(T outputPrefix) {
            assert (this.owner.validOutput(outputPrefix));
            for (int arcIdx = 0; arcIdx < this.numArcs; ++arcIdx) {
                this.arcs[arcIdx].output = this.owner.fst.outputs.add(outputPrefix, this.arcs[arcIdx].output);
                assert (this.owner.validOutput(this.arcs[arcIdx].output));
            }
            if (this.isFinal) {
                this.output = this.owner.fst.outputs.add(outputPrefix, this.output);
                assert (this.owner.validOutput(this.output));
            }
        }
    }

    static final class CompiledNode
    implements Node {
        long node;

        CompiledNode() {
        }

        @Override
        public boolean isCompiled() {
            return true;
        }
    }

    static interface Node {
        public boolean isCompiled();
    }

    static class Arc<T> {
        int label;
        Node target;
        boolean isFinal;
        T output;
        T nextFinalOutput;

        Arc() {
        }
    }

    public static class Builder<T> {
        private final FST.INPUT_TYPE inputType;
        private final Outputs<T> outputs;
        private double suffixRAMLimitMB = 32.0;
        private boolean allowFixedLengthArcs = true;
        private int bytesPageBits = 15;
        private float directAddressingMaxOversizingFactor = 1.0f;
        private int version = 9;

        public Builder(FST.INPUT_TYPE inputType, Outputs<T> outputs) {
            this.inputType = inputType;
            this.outputs = outputs;
        }

        public Builder<T> suffixRAMLimitMB(double mb) {
            if (mb < 0.0) {
                throw new IllegalArgumentException("suffixRAMLimitMB must be >= 0; got: " + mb);
            }
            this.suffixRAMLimitMB = mb;
            return this;
        }

        public Builder<T> allowFixedLengthArcs(boolean allowFixedLengthArcs) {
            this.allowFixedLengthArcs = allowFixedLengthArcs;
            return this;
        }

        public Builder<T> bytesPageBits(int bytesPageBits) {
            this.bytesPageBits = bytesPageBits;
            return this;
        }

        public Builder<T> directAddressingMaxOversizingFactor(float factor) {
            this.directAddressingMaxOversizingFactor = factor;
            return this;
        }

        public Builder<T> setVersion(int version) {
            if (version < 8 || version > 9) {
                throw new IllegalArgumentException("Expected version in range [8, 9], got " + version);
            }
            this.version = version;
            return this;
        }

        public FSTCompiler<T> build() throws IOException {
            return new FSTCompiler<T>(this.inputType, this.suffixRAMLimitMB, this.outputs, this.allowFixedLengthArcs, this.bytesPageBits, this.directAddressingMaxOversizingFactor, this.version);
        }
    }
}

