/*
 * Decompiled with CFR 0.152.
 */
package org.apache.solr.core.backup;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.lang.invoke.MethodHandles;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import org.apache.lucene.store.IOContext;
import org.apache.lucene.store.IndexInput;
import org.apache.solr.common.SolrException;
import org.apache.solr.common.cloud.ClusterState;
import org.apache.solr.common.cloud.DocCollection;
import org.apache.solr.common.cloud.SolrZkClient;
import org.apache.solr.common.cloud.ZkMaintenanceUtils;
import org.apache.solr.common.cloud.ZkStateReader;
import org.apache.solr.common.util.Utils;
import org.apache.solr.core.ConfigSetService;
import org.apache.solr.core.backup.BackupFilePaths;
import org.apache.solr.core.backup.BackupId;
import org.apache.solr.core.backup.BackupProperties;
import org.apache.solr.core.backup.repository.BackupRepository;
import org.apache.solr.util.FileTypeMagicUtil;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.KeeperException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BackupManager {
    private static final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    public static final String COLLECTION_PROPS_FILE = "collection_state.json";
    public static final String TRADITIONAL_BACKUP_PROPS_FILE = "backup.properties";
    public static final String ZK_STATE_DIR = "zk_backup";
    public static final String CONFIG_STATE_DIR = "configs";
    public static final String COLLECTION_NAME_PROP = "collection";
    public static final String COLLECTION_ALIAS_PROP = "collectionAlias";
    public static final String BACKUP_NAME_PROP = "backupName";
    public static final String INDEX_VERSION_PROP = "indexVersion";
    public static final String START_TIME_PROP = "startTime";
    public static final String END_TIME_PROP = "endTime";
    protected final ZkStateReader zkStateReader;
    protected final BackupRepository repository;
    protected final BackupId backupId;
    protected final URI backupPath;
    protected final String existingPropsFile;

    private BackupManager(BackupRepository repository, URI backupPath, ZkStateReader zkStateReader, String existingPropsFile, BackupId backupId) {
        this.repository = Objects.requireNonNull(repository);
        this.backupPath = backupPath;
        this.zkStateReader = Objects.requireNonNull(zkStateReader);
        this.existingPropsFile = existingPropsFile;
        this.backupId = backupId;
    }

    public static BackupManager forIncrementalBackup(BackupRepository repository, ZkStateReader stateReader, URI backupPath) {
        Objects.requireNonNull(repository);
        Objects.requireNonNull(stateReader);
        Optional<BackupId> lastBackupId = BackupFilePaths.findMostRecentBackupIdFromFileListing(repository.listAllOrEmpty(backupPath));
        return new BackupManager(repository, backupPath, stateReader, lastBackupId.map(id -> BackupFilePaths.getBackupPropsName(id)).orElse(null), lastBackupId.map(BackupId::nextBackupId).orElse(BackupId.zero()));
    }

    public static BackupManager forBackup(BackupRepository repository, ZkStateReader stateReader, URI backupPath) {
        Objects.requireNonNull(repository);
        Objects.requireNonNull(stateReader);
        return new BackupManager(repository, backupPath, stateReader, null, BackupId.traditionalBackup());
    }

    public static BackupManager forRestore(BackupRepository repository, ZkStateReader stateReader, URI backupPath, int bid) throws IOException {
        Objects.requireNonNull(repository);
        Objects.requireNonNull(stateReader);
        BackupId backupId = new BackupId(bid);
        String backupPropsName = BackupFilePaths.getBackupPropsName(backupId);
        if (!repository.exists(repository.resolve(backupPath, backupPropsName))) {
            throw new IllegalStateException("Backup id " + bid + " was not found");
        }
        return new BackupManager(repository, backupPath, stateReader, backupPropsName, backupId);
    }

    public static BackupManager forRestore(BackupRepository repository, ZkStateReader stateReader, URI backupPath) throws IOException {
        Objects.requireNonNull(repository);
        Objects.requireNonNull(stateReader);
        if (!repository.exists(backupPath)) {
            throw new SolrException(SolrException.ErrorCode.SERVER_ERROR, "Couldn't restore since doesn't exist: " + backupPath);
        }
        Optional<BackupId> opFileGen = BackupFilePaths.findMostRecentBackupIdFromFileListing(repository.listAll(backupPath));
        if (opFileGen.isPresent()) {
            BackupId backupPropFile = opFileGen.get();
            return new BackupManager(repository, backupPath, stateReader, BackupFilePaths.getBackupPropsName(backupPropFile), backupPropFile);
        }
        if (repository.exists(repository.resolve(backupPath, TRADITIONAL_BACKUP_PROPS_FILE))) {
            return new BackupManager(repository, backupPath, stateReader, TRADITIONAL_BACKUP_PROPS_FILE, null);
        }
        throw new IllegalStateException("No backup.properties was found, the backup does not exist or not complete");
    }

    public final BackupId getBackupId() {
        return this.backupId;
    }

    public final String getVersion() {
        return "1.0";
    }

    public BackupProperties readBackupProperties() throws IOException {
        if (this.existingPropsFile == null) {
            throw new IllegalStateException("No backup.properties was found, the backup does not exist or not complete");
        }
        return BackupProperties.readFrom(this.repository, this.backupPath, this.existingPropsFile);
    }

    public Optional<BackupProperties> tryReadBackupProperties() throws IOException {
        if (this.existingPropsFile != null) {
            return Optional.of(BackupProperties.readFrom(this.repository, this.backupPath, this.existingPropsFile));
        }
        return Optional.empty();
    }

    public void writeBackupProperties(BackupProperties props) throws IOException {
        URI dest = this.repository.resolve(this.backupPath, BackupFilePaths.getBackupPropsName(this.backupId));
        try (OutputStreamWriter propsWriter = new OutputStreamWriter(this.repository.createOutput(dest), StandardCharsets.UTF_8);){
            props.store(propsWriter);
        }
    }

    public DocCollection readCollectionState(String collectionName) throws IOException {
        Objects.requireNonNull(collectionName);
        URI zkStateDir = this.getZkStateDir();
        try (IndexInput is = this.repository.openInput(zkStateDir, COLLECTION_PROPS_FILE, IOContext.DEFAULT);){
            byte[] arr = new byte[(int)is.length()];
            is.readBytes(arr, 0, (int)is.length());
            ClusterState c_state = ClusterState.createFromJson(-1, arr, Collections.emptySet(), null);
            DocCollection docCollection = c_state.getCollection(collectionName);
            return docCollection;
        }
    }

    public void writeCollectionState(String collectionName, DocCollection collectionState) throws IOException {
        URI dest = this.repository.resolve(this.getZkStateDir(), COLLECTION_PROPS_FILE);
        try (OutputStream collectionStateOs = this.repository.createOutput(dest);){
            collectionStateOs.write(Utils.toJSON(Collections.singletonMap(collectionName, collectionState)));
        }
    }

    public void uploadConfigDir(String sourceConfigName, String targetConfigName, ConfigSetService configSetService) throws IOException {
        URI source = this.repository.resolveDirectory(this.getZkStateDir(), CONFIG_STATE_DIR, sourceConfigName);
        if (!this.repository.exists(source)) {
            throw new IllegalArgumentException("Configset expected at " + source + " does not exist");
        }
        this.uploadConfigToSolrCloud(configSetService, source, targetConfigName, "");
    }

    public void downloadConfigDir(String configName, ConfigSetService configSetService) throws IOException {
        URI dest = this.repository.resolveDirectory(this.getZkStateDir(), CONFIG_STATE_DIR, configName);
        this.repository.createDirectory(this.repository.resolveDirectory(this.getZkStateDir(), CONFIG_STATE_DIR));
        this.repository.createDirectory(dest);
        this.downloadConfigToRepo(configSetService, configName, dest);
    }

    public void uploadCollectionProperties(String collectionName) throws IOException {
        URI sourceDir = this.getZkStateDir();
        URI source = this.repository.resolve(sourceDir, "collectionprops.json");
        if (!this.repository.exists(source)) {
            return;
        }
        String zkPath = "/collections/" + collectionName + "/collectionprops.json";
        try (IndexInput is = this.repository.openInput(sourceDir, "collectionprops.json", IOContext.DEFAULT);){
            byte[] arr = new byte[(int)is.length()];
            is.readBytes(arr, 0, (int)is.length());
            this.zkStateReader.getZkClient().create(zkPath, arr, CreateMode.PERSISTENT, true);
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error uploading file to zookeeper path " + source.toString() + " to " + zkPath, SolrZkClient.checkInterrupted(e));
        }
    }

    public void downloadCollectionProperties(String collectionName) throws IOException {
        URI dest = this.repository.resolve(this.getZkStateDir(), "collectionprops.json");
        String zkPath = "/collections/" + collectionName + "/collectionprops.json";
        try {
            if (!this.zkStateReader.getZkClient().exists(zkPath, true).booleanValue()) {
                return;
            }
            try (OutputStream os = this.repository.createOutput(dest);){
                byte[] data = this.zkStateReader.getZkClient().getData(zkPath, null, null, true);
                os.write(data);
            }
        }
        catch (InterruptedException | KeeperException e) {
            throw new IOException("Error downloading file from zookeeper path " + zkPath + " to " + dest.toString(), SolrZkClient.checkInterrupted(e));
        }
    }

    private void downloadConfigToRepo(ConfigSetService configSetService, String configName, URI dir) throws IOException {
        List<String> filePaths = configSetService.getAllConfigFiles(configName);
        for (String filePath : filePaths) {
            URI uri = this.repository.resolve(dir, filePath.replace('/', File.separatorChar));
            if (!filePath.endsWith("/")) {
                if (ZkMaintenanceUtils.isFileForbiddenInConfigSets(filePath)) {
                    log.warn("Not including zookeeper file in backup, as it is a forbidden type: {}", (Object)filePath);
                    continue;
                }
                log.debug("Writing file {}", (Object)filePath);
                byte[] data = configSetService.downloadFileFromConfig(configName, filePath);
                if (data == null) {
                    data = new byte[]{};
                }
                if (!FileTypeMagicUtil.isFileForbiddenInConfigset(data)) {
                    OutputStream os = this.repository.createOutput(uri);
                    try {
                        os.write(data);
                        continue;
                    }
                    finally {
                        if (os != null) {
                            os.close();
                        }
                        continue;
                    }
                }
                String mimeType = FileTypeMagicUtil.INSTANCE.guessMimeType(data);
                log.warn("Not including zookeeper file {} in backup, as it matched the MAGIC signature of a forbidden mime type {}", (Object)filePath, (Object)mimeType);
                continue;
            }
            if (this.repository.exists(uri)) continue;
            this.repository.createDirectory(uri);
        }
    }

    private void uploadConfigToSolrCloud(ConfigSetService configSetService, URI sourceDir, String configName, String filePrefix) throws IOException {
        block9: for (String file : this.repository.listAll(sourceDir)) {
            String filePath = filePrefix + file;
            URI path = this.repository.resolve(sourceDir, file);
            BackupRepository.PathType t = this.repository.getPathType(path);
            switch (t) {
                case FILE: {
                    if (ZkMaintenanceUtils.isFileForbiddenInConfigSets(filePath)) {
                        log.warn("Not including zookeeper file in restore, as it is a forbidden type: {}", (Object)file);
                        continue block9;
                    }
                    try (IndexInput is = this.repository.openInput(sourceDir, file, IOContext.DEFAULT);){
                        byte[] arr = new byte[(int)is.length()];
                        is.readBytes(arr, 0, (int)is.length());
                        if (!FileTypeMagicUtil.isFileForbiddenInConfigset(arr)) {
                            configSetService.uploadFileToConfig(configName, filePath, arr, false);
                            continue block9;
                        }
                        String mimeType = FileTypeMagicUtil.INSTANCE.guessMimeType(arr);
                        log.warn("Not including zookeeper file {} in restore, as it matched the MAGIC signature of a forbidden mime type {}", (Object)filePath, (Object)mimeType);
                        continue block9;
                    }
                }
                case DIRECTORY: {
                    if (file.startsWith(".")) continue block9;
                    this.uploadConfigToSolrCloud(configSetService, path, configName, filePath + "/");
                    continue block9;
                }
                default: {
                    throw new IllegalStateException("Unknown path type " + t);
                }
            }
        }
    }

    private URI getZkStateDir() {
        URI zkStateDir;
        if (this.backupId != null) {
            String zkBackupFolder = BackupFilePaths.getZkStateDir(this.backupId);
            zkStateDir = this.repository.resolveDirectory(this.backupPath, zkBackupFolder);
        } else {
            zkStateDir = this.repository.resolveDirectory(this.backupPath, ZK_STATE_DIR);
        }
        return zkStateDir;
    }

    public void createZkStateDir() throws IOException {
        this.repository.createDirectory(this.getZkStateDir());
    }
}

