package org.xmlresolver;

import org.w3c.dom.ls.LSInput;
import org.w3c.dom.ls.LSResourceResolver;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.ext.EntityResolver2;
import org.xmlresolver.logging.AbstractLogger;
import org.xmlresolver.logging.ResolverLogger;
import org.xmlresolver.sources.ResolverInputSource;
import org.xmlresolver.sources.ResolverLSInput;
import org.xmlresolver.sources.ResolverSAXSource;

import javax.xml.transform.Source;
import javax.xml.transform.TransformerException;
import javax.xml.transform.URIResolver;
import java.io.IOException;

/** An implementation of many resolver interfaces.
 *
 * <p>This class is probably the most common entry point to the XML Catalog resolver. It has a zero
 * argument constructor so it can be instantiated directly from its class name (for example, passed to
 * an application as a commend line argument or stored in a configuration file). When instantiated
 * this way, it will automatically be configured by system properties and an <code>xmlresolver.properties</code>
 * configuration file, if one exists.</p>
 *
 * <p>This class implements the {@link org.xml.sax.EntityResolver}, {@link org.xml.sax.ext.EntityResolver2},
 * {@link LSResourceResolver}
 * and {@link org.xmlresolver.NamespaceResolver}, and {@link javax.xml.transform.URIResolver} interfaces.</p>
 *
 * <p>The StAX {@link javax.xml.stream.XMLResolver} interface is implemented by the
 * {@link org.xmlresolver.StAXResolver} class because the <code>resolveEntity</code> method
 * of the <code>XMLResolver</code> interface isn't compatible with the <code>EntityResolver2</code>
 * method of the same name.</p>
 *
 * @see org.xmlresolver.StAXResolver
 */

public class Resolver implements URIResolver, EntityResolver, EntityResolver2, NamespaceResolver, LSResourceResolver {
    private final ResolverLogger logger;
    protected final XMLResolverConfiguration config;
    protected final CatalogResolver resolver;

    /** Creates a new instance of Resolver.
     *
     * The default resolver is a new ResourceResolver that uses a static catalog shared by all threads.
     */
    public Resolver() {
        config = new XMLResolverConfiguration();
        resolver = new CatalogResolver(config);
        logger = config.getFeature(ResolverFeature.RESOLVER_LOGGER);
    }

    /** Creates a new instance of a Resolver.
     *
     * Creates a resolver using a specific Catalog.
     *
     * @param config The configuration to use.
     */
    public Resolver(XMLResolverConfiguration config) {
        this.config = config;
        resolver = new CatalogResolver(config);
        logger = config.getFeature(ResolverFeature.RESOLVER_LOGGER);
    }

    /** Creates a new instance of a Resolver.
     *
     * Creates a resolver using a specific underlying ResourceResolver.
     *
     * @param resolver The resource resolver to use.
     */
    public Resolver(CatalogResolver resolver) {
        config = resolver.getConfiguration();
        this.resolver = resolver;
        logger = config.getFeature(ResolverFeature.RESOLVER_LOGGER);
    }

    /** What version is this?
     *
     * Returns the version number of this resolver instance.
     *
     * @return The version number
     */
    public static String version() {
        return BuildConfig.VERSION;
    }

    /** Get the Catalog used by this resolver.
     *
     * @return The underlying catalog.
     */
    public XMLResolverConfiguration getConfiguration() {
        return resolver.getConfiguration();
    }

    /** Get the underlying {@link CatalogResolver} used by this resolver.
     * @return The catalog resolver.
     */
    public CatalogResolver getCatalogResolver() {
        return resolver;
    }

    /** Implements the {@link javax.xml.transform.URIResolver} interface. */
    @Override
    public Source resolve(String href, String base) throws TransformerException {
        ResolvedResource rsrc = resolver.resolveURI(href, base);
        if (rsrc == null) {
            return null;
        }

        ResolverSAXSource source = new ResolverSAXSource(rsrc.getLocalURI(), new InputSource(rsrc.getInputStream()));
        source.setSystemId(rsrc.getResolvedURI().toString());
        return source;
    }

    /** Implements the {@link org.w3c.dom.ls.LSResourceResolver} interface. */
    @Override
    public LSInput resolveResource(String type, String namespaceURI, String publicId, String systemId, String baseURI) {
        ResolvedResource rsrc = null;
        if (type == null || "http://www.w3.org/TR/REC-xml".equals(type)) {
            logger.log(AbstractLogger.REQUEST, "resolveResource: XML: %s (baseURI: %s, publicId: %s)",
                    systemId, baseURI, publicId);
            rsrc = resolver.resolveEntity(null, publicId, systemId, baseURI);
        } else {
            logger.log(AbstractLogger.REQUEST, "resolveResource: %s, %s (namespace: %s, baseURI: %s, publicId: %s)",
                    type, systemId, namespaceURI, baseURI, publicId);

            String purpose = null;
            // If it looks like it's going to be used for validation, ...
            if ("http://www.w3.org/2001/XMLSchema".equals(type)
                    || "http://www.w3.org/XML/XMLSchema/v1.1".equals(type)
                    || "http://relaxng.org/ns/structure/1.0".equals(type)) {
                purpose = "http://www.rddl.org/purposes#schema-validation";
            }

            rsrc = resolver.resolveNamespace(systemId, baseURI, type, purpose);
        }

        if (rsrc == null) {
            return null;
        }

        return new ResolverLSInput(rsrc, publicId);
    }

    /** Implements the {@link org.xml.sax.ext.EntityResolver2} interface. */
    @Override
    public InputSource getExternalSubset(String name, String baseURI) throws SAXException, IOException {
        ResolvedResource rsrc = resolver.resolveEntity(name, null, null, baseURI);
        if (rsrc == null) {
            return null;
        }

        ResolverInputSource source = new ResolverInputSource(rsrc.getLocalURI(), rsrc.getInputStream());
        source.setSystemId(rsrc.getResolvedURI().toString());
        return source;
    }

    /** Implements the {@link org.xml.sax.ext.EntityResolver2} interface. */
    @Override
    public InputSource resolveEntity(String name, String publicId, String baseURI, String systemId) throws SAXException, IOException {
        ResolvedResource rsrc = resolver.resolveEntity(name, publicId, systemId, baseURI);
        if (rsrc == null) {
            return null;
        }

        ResolverInputSource source = new ResolverInputSource(rsrc.getLocalURI(), rsrc.getInputStream());
        source.setSystemId(rsrc.getResolvedURI().toString());
        return source;
    }

    /** Implements the {@link org.xml.sax.EntityResolver} interface. */
    @Override
    public InputSource resolveEntity(String publicId, String systemId) throws SAXException, IOException {
        ResolvedResource rsrc = resolver.resolveEntity(null, publicId, systemId, null);
        if (rsrc == null) {
            return null;
        }

        ResolverInputSource source = new ResolverInputSource(rsrc.getLocalURI(), rsrc.getInputStream());
        source.setSystemId(rsrc.getResolvedURI().toString());
        return source;
    }

    /** Implements the {@link org.xmlresolver.NamespaceResolver} interface. */
    @Override
    public Source resolveNamespace(String uri, String nature, String purpose) throws TransformerException {
        ResolvedResource rsrc = resolver.resolveNamespace(uri, null, nature, purpose);
        if (rsrc == null) {
            return null;
        }

        ResolverSAXSource source = new ResolverSAXSource(rsrc.getLocalURI(), new InputSource(rsrc.getInputStream()));
        source.setSystemId(rsrc.getResolvedURI().toString());
        return source;
    }
}
