package cat.inspiracio.orange;

import java.io.IOException;

import javax.servlet.ServletConfig;
import javax.servlet.ServletContext;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import javax.servlet.jsp.JspWriter;
import javax.servlet.jsp.PageContext;

public abstract class Template {
	private static final String NL="\n";
	
	// state -----------------------------------------------
	//All these variables are visible in the write() method.

	protected JspWriter out;	
	protected HttpServletRequest request;
	protected HttpServletResponse response;
	protected HttpSession session;
	protected PageContext pageContext;
	protected ServletContext application;
	protected ServletConfig config;
	protected Template page;
	protected Throwable exception;

	// construction --------------------------------
	
	public Template(){}

	/** The only setter you need to call.
	 * It sets all fields. */
	public final void setPageContext(PageContext pc){
		out=pc.getOut();
		pageContext=pc;
		request=(HttpServletRequest)pc.getRequest();
		response=(HttpServletResponse)pc.getResponse();
		session=pc.getSession();
		application=pc.getServletContext();
		config=pc.getServletConfig();
		page=this;//Or better?
		exception=pc.getException();
	}
	
	// abstract method -----------------------------
	
	/** This is the method that template must implement. 
	 * @throws Exception Java-islands threw an exception. */
	public abstract void write()throws Exception;

	// orange logic ------------------------------
	
	/** Called for an element with attribute data-substitute=v.
	 * 
	 * The attribute value is a path. The servlet reads a file at the path. 
	 * The file contains an HTML fragment. The fragment is inserted in the 
	 * document, substituting the original element.
	 * 
	 * Example:
	 * <div data-substitute="header.html"/>
	 * 
	 * The attribute value is literally a path. It is not a javascript expression 
	 * which when evaluated results in a path.
	 * 
	 * The path is relative to the location of the document in which it occurs, 
	 * or absolute if it starts with /.
	 * 
	 * @throws IOException */
	protected final void substitute(String v) throws Exception{
		String n=resolve(v);
		
		//instantiate the class and call it
		@SuppressWarnings("unchecked")
		Class<Template>c=(Class<Template>)Class.forName(n);
		Template t=c.newInstance();
		t.setPageContext(pageContext);
		t.write();
	}

	/** Gets the class name of this template.
	 * 
	 * Package-visible so that tests can fake it. */
	String getClassName(){return getClass().getName();}
	
	/** Resolves path v relative to the path of this template.
	 * 
	 * Package-visible for tests.
	 * 
	 * @param v relative or absolute path, like "hr.html" or "/cacti/cactus.html".
	 * 	Absolute path must not contain "." or ".." directories.
	 * 
	 * @return fully qualified class name of v */
	final String resolve(String v){
		//XXX Unite with OrangeServlet.packageName(path) and className(path).
		
		String n="cat.inspiracio.orange.webapp.hr";//not the real class name yet

		//absolute path, May it contain ".." or "."?
		if(v.startsWith("/")){
			//absolute, not relative to current template
			v=v.substring(1);//drop initial "/"
			v=v.substring(0, v.length()-5);//drop final ".html"
			v=v.replace('-', '_');
			v=v.replace('/', '.');//not correct if there is ".." or "."
			n="cat.inspiracio.orange.webapp." + v;
			return n;
		}
		
		//resolve v with respect to the current template
		else{
			//Improve this
			String current=getClassName();
			//treat ".." and "."
			String[]parts=current.split("\\.");
			
			v=v.substring(0, v.length()-5);//drop final ".html"
			v=v.replace('-', '_');
			String[]vs=v.split("/");//first split
			
			String[] cs=new String[parts.length-1 + vs.length];
			for(int i=0; i<parts.length-1; i++)
				cs[i]=parts[i];
			int i=parts.length-1;
			for(int j=0; j<vs.length; j++){
				String s=vs[j];
				if(".".equals(s)){}
				else if("..".equals(s)){i--;}
				else cs[i++]=s;
			}
			n="";
			for(int j=0; j<i; j++)
				n = n + "." + cs[j];
			n=n.substring(1);
		}
		return n;
	}
	
    /** Quotes a value so that it can be an attribute's value.
     * Escapes " by &quot; and encloses in ". 
     * Called from generated write(). */
    protected final String quote(Object o){return quote(o.toString());}

    /** Quotes a value so that it can be an attribute's value.
     * Escapes " by &quot; and encloses in ". 
     * Called from generated write(). */
    protected final String quote(String value){
        //opt: In html5, quotes are often not needed.
        if(contains(value, '"'))
            value=value.replaceAll("\"", "&quot;");
        return '"' + value + '"';
    }

	/** escape for html: < & 
	 * Called from generated write(). */
	protected final String escape(Object o){return escape(o.toString());}
	
	/** escape for html: < & 
	 * Called from generated write(). */
	protected final String escape(String s){
		if(s==null)
			return null;
		//Need to optimise?
		return s.replace("&", "&amp;").replace("<", "&lt;");
	}
	
	/** Escapes an int for html: just convert it to String.
	 * Called from generated write(). */
	protected final String escape(int i){return Integer.toString(i);}
	
	/** Boolean attribute: write the key if the value is true.
	 * Called from generated write(). */
    protected final void attribute(String key, boolean b) throws IOException{
    	if(b)write(" " + key);
    }
    
    /** Attribute: 
     * If the value is empty, writes just the key,
     * otherwise key and value, quoted. */
    protected final void attribute(String key, Object v) throws IOException{
    	if(v==null){
    		write(" " + key);
    		return;
    	}
    	String s=v.toString();
    	if(empty(s)){
    		write(" " + key);
    		return;
    	}
    	write(" " + key + "=" + quote(s));
    }
    
    private final boolean empty(String s){return s==null || 0==s.length();}
    
    /** Does the string contain this character? */
    private final boolean contains(String value, char c){return 0<=value.indexOf(c);}
    
    // writing -------------------------------------
	
    /** Writes an int to out. 
     * 
     * Called from generated write(). 
     * Public and returns this for fluent style. */
	public final Template write(int i)throws IOException{
		out.write(Integer.toString(i));
		return this;
	}
	
    /** Writes an object to out.
     * Does nothing if the object is empty (null or its toString() is null or ""). 
     * 
     * Called from generated write(). 
     * Public and returns this for fluent style. */
	public final Template write(Object o)throws IOException{
		if(o==null)
			return this;
		String s=o.toString();
		return write(s);
	}
	
    /** Writes a String to out.
     * Does nothing if the String is null or "". 
     * 
     * Called from generated write(). 
     * Public and returns this for fluent style. */
	public final Template write(String s)throws IOException{
		if(!empty(s))
			out.write(s);
		return this;
	}
	
	/** Write a string to output.
	 * 
	 *  Called from generated write().
	 * Public and returns this for fluid style. */
	public final Template writeln()throws IOException{return write(NL);}
	
	/** Write a string to output.
	 * 
	 * Called from generated write().
	 * Public and returns this for fluid style. */
	public final Template writeln(String s)throws IOException{
		write(s);
		return writeln();
	}
		
}
