/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.http.policy;

import com.azure.core.http.HttpHeader;
import com.azure.core.http.HttpHeaders;
import com.azure.core.http.HttpPipelineCallContext;
import com.azure.core.http.HttpPipelineNextPolicy;
import com.azure.core.http.HttpRequest;
import com.azure.core.http.HttpResponse;
import com.azure.core.http.policy.HttpLogDetailLevel;
import com.azure.core.http.policy.HttpLogOptions;
import com.azure.core.http.policy.HttpPipelinePolicy;
import com.azure.core.http.policy.HttpRequestLogger;
import com.azure.core.http.policy.HttpRequestLoggingContext;
import com.azure.core.http.policy.HttpResponseLogger;
import com.azure.core.http.policy.HttpResponseLoggingContext;
import com.azure.core.implementation.http.HttpPipelineCallContextHelper;
import com.azure.core.util.Context;
import com.azure.core.util.CoreUtils;
import com.azure.core.util.UrlBuilder;
import com.azure.core.util.logging.ClientLogger;
import com.azure.core.util.logging.LogLevel;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.WritableByteChannel;
import java.time.Duration;
import java.util.Collections;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Collectors;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class HttpLoggingPolicy
implements HttpPipelinePolicy {
    private static final ObjectMapper PRETTY_PRINTER = new ObjectMapper().enable(SerializationFeature.INDENT_OUTPUT);
    private static final int MAX_BODY_LOG_SIZE = 16384;
    private static final String REDACTED_PLACEHOLDER = "REDACTED";
    private final ClientLogger logger = new ClientLogger(HttpLoggingPolicy.class);
    private final HttpLogDetailLevel httpLogDetailLevel;
    private final Set<String> allowedHeaderNames;
    private final Set<String> allowedQueryParameterNames;
    private final boolean prettyPrintBody;
    private final HttpRequestLogger requestLogger;
    private final HttpResponseLogger responseLogger;
    public static final String RETRY_COUNT_CONTEXT = "requestRetryCount";

    public HttpLoggingPolicy(HttpLogOptions httpLogOptions) {
        if (httpLogOptions == null) {
            this.httpLogDetailLevel = HttpLogDetailLevel.NONE;
            this.allowedHeaderNames = Collections.emptySet();
            this.allowedQueryParameterNames = Collections.emptySet();
            this.prettyPrintBody = false;
            this.requestLogger = new DefaultHttpRequestLogger();
            this.responseLogger = new DefaultHttpResponseLogger();
        } else {
            this.httpLogDetailLevel = httpLogOptions.getLogLevel();
            this.allowedHeaderNames = httpLogOptions.getAllowedHeaderNames().stream().map(headerName -> headerName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.allowedQueryParameterNames = httpLogOptions.getAllowedQueryParamNames().stream().map(queryParamName -> queryParamName.toLowerCase(Locale.ROOT)).collect(Collectors.toSet());
            this.prettyPrintBody = httpLogOptions.isPrettyPrintBody();
            this.requestLogger = httpLogOptions.getRequestLogger() == null ? new DefaultHttpRequestLogger() : httpLogOptions.getRequestLogger();
            this.responseLogger = httpLogOptions.getResponseLogger() == null ? new DefaultHttpResponseLogger() : httpLogOptions.getResponseLogger();
        }
    }

    @Override
    public Mono<HttpResponse> process(HttpPipelineCallContext context, HttpPipelineNextPolicy next) {
        if (this.httpLogDetailLevel == HttpLogDetailLevel.NONE) {
            return next.process();
        }
        ClientLogger logger = new ClientLogger((String)context.getData("caller-method").orElse(""));
        long startNs = System.nanoTime();
        return this.requestLogger.logRequest(logger, this.getRequestLoggingOptions(context)).then(next.process()).flatMap(response -> this.responseLogger.logResponse(logger, this.getResponseLoggingOptions((HttpResponse)response, startNs, context))).doOnError(throwable -> logger.warning("<-- HTTP FAILED: ", throwable));
    }

    private HttpRequestLoggingContext getRequestLoggingOptions(HttpPipelineCallContext callContext) {
        return new HttpRequestLoggingContext(callContext.getHttpRequest(), HttpPipelineCallContextHelper.getContext(callContext), HttpLoggingPolicy.getRequestRetryCount(HttpPipelineCallContextHelper.getContext(callContext), this.getHttpLoggingPolicyLogger()));
    }

    private HttpResponseLoggingContext getResponseLoggingOptions(HttpResponse httpResponse, long startNs, HttpPipelineCallContext callContext) {
        return new HttpResponseLoggingContext(httpResponse, Duration.ofNanos(System.nanoTime() - startNs), HttpPipelineCallContextHelper.getContext(callContext), HttpLoggingPolicy.getRequestRetryCount(HttpPipelineCallContextHelper.getContext(callContext), this.getHttpLoggingPolicyLogger()));
    }

    private ClientLogger getHttpLoggingPolicyLogger() {
        return this.logger;
    }

    private static <T> Mono<T> logAndReturn(ClientLogger logger, LogLevel logLevel, StringBuilder logMessageBuilder, T data) {
        switch (logLevel) {
            case VERBOSE: {
                logger.verbose(logMessageBuilder.toString());
                break;
            }
            case INFORMATIONAL: {
                logger.info(logMessageBuilder.toString());
                break;
            }
            case WARNING: {
                logger.warning(logMessageBuilder.toString());
                break;
            }
            case ERROR: {
                logger.error(logMessageBuilder.toString());
                break;
            }
        }
        return Mono.justOrEmpty(data);
    }

    private static String getRedactedUrl(URL url, Set<String> allowedQueryParameterNames) {
        UrlBuilder builder = UrlBuilder.parse(url);
        String allowedQueryString = HttpLoggingPolicy.getAllowedQueryString(url.getQuery(), allowedQueryParameterNames);
        return builder.clearQuery().setQuery(allowedQueryString).toString();
    }

    private static String getAllowedQueryString(String queryString, Set<String> allowedQueryParameterNames) {
        String[] queryParams;
        if (CoreUtils.isNullOrEmpty(queryString)) {
            return "";
        }
        StringBuilder queryStringBuilder = new StringBuilder();
        for (String queryParam : queryParams = queryString.split("&")) {
            String[] queryPair;
            if (queryStringBuilder.length() > 0) {
                queryStringBuilder.append("&");
            }
            if ((queryPair = queryParam.split("=", 2)).length == 2) {
                String queryName = queryPair[0];
                if (allowedQueryParameterNames.contains(queryName.toLowerCase(Locale.ROOT))) {
                    queryStringBuilder.append(queryParam);
                    continue;
                }
                queryStringBuilder.append(queryPair[0]).append("=").append(REDACTED_PLACEHOLDER);
                continue;
            }
            queryStringBuilder.append(queryParam);
        }
        return queryStringBuilder.toString();
    }

    private static void addHeadersToLogMessage(Set<String> allowedHeaderNames, HttpHeaders headers, StringBuilder sb) {
        for (HttpHeader header : headers) {
            String headerName = header.getName();
            sb.append(headerName).append(":");
            if (allowedHeaderNames.contains(headerName.toLowerCase(Locale.ROOT))) {
                sb.append(header.getValue());
            } else {
                sb.append(REDACTED_PLACEHOLDER);
            }
            sb.append(System.lineSeparator());
        }
    }

    private static String prettyPrintIfNeeded(ClientLogger logger, boolean prettyPrintBody, String contentType, String body) {
        String result = body;
        if (prettyPrintBody && contentType != null && (contentType.startsWith("application/json") || contentType.startsWith("text/json"))) {
            try {
                JsonNode deserialized = PRETTY_PRINTER.readTree(body);
                result = PRETTY_PRINTER.writeValueAsString((Object)deserialized);
            }
            catch (Exception e) {
                logger.warning("Failed to pretty print JSON: {}", e.getMessage());
            }
        }
        return result;
    }

    private static long getContentLength(ClientLogger logger, HttpHeaders headers) {
        long contentLength = 0L;
        String contentLengthString = headers.getValue("Content-Length");
        if (CoreUtils.isNullOrEmpty(contentLengthString)) {
            return contentLength;
        }
        try {
            contentLength = Long.parseLong(contentLengthString);
        }
        catch (NullPointerException | NumberFormatException e) {
            logger.warning("Could not parse the HTTP header content-length: '{}'.", headers.getValue("content-length"), e);
        }
        return contentLength;
    }

    private static boolean shouldBodyBeLogged(String contentTypeHeader, long contentLength) {
        return !"application/octet-stream".equalsIgnoreCase(contentTypeHeader) && contentLength != 0L && contentLength < 16384L;
    }

    private static String convertStreamToString(ByteArrayOutputStream stream, ClientLogger logger) {
        try {
            return stream.toString("UTF-8");
        }
        catch (UnsupportedEncodingException ex) {
            throw logger.logExceptionAsError(new RuntimeException(ex));
        }
    }

    private static Mono<ByteBuffer> writeBufferToBodyStream(WritableByteChannel channel, ByteBuffer byteBuffer) {
        try {
            channel.write(byteBuffer.duplicate());
            return Mono.just((Object)byteBuffer);
        }
        catch (IOException ex) {
            return Mono.error((Throwable)ex);
        }
    }

    private static Integer getRequestRetryCount(Context context, ClientLogger logger) {
        Object rawRetryCount = context.getData(RETRY_COUNT_CONTEXT).orElse(null);
        if (rawRetryCount == null) {
            return null;
        }
        try {
            return Integer.valueOf(rawRetryCount.toString());
        }
        catch (NumberFormatException ex) {
            logger.warning("Could not parse the request retry count: '{}'.", rawRetryCount);
            return null;
        }
    }

    private final class DefaultHttpResponseLogger
    implements HttpResponseLogger {
        private DefaultHttpResponseLogger() {
        }

        @Override
        public Mono<HttpResponse> logResponse(ClientLogger logger, HttpResponseLoggingContext loggingOptions) {
            long contentLength;
            LogLevel logLevel = this.getLogLevel(loggingOptions);
            HttpResponse response = loggingOptions.getHttpResponse();
            if (!logger.canLogAtLevel(logLevel)) {
                return Mono.just((Object)response);
            }
            String contentLengthString = response.getHeaderValue("Content-Length");
            String bodySize = CoreUtils.isNullOrEmpty(contentLengthString) ? "unknown-length body" : contentLengthString + "-byte body";
            StringBuilder responseLogMessage = new StringBuilder();
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogUrl()) {
                responseLogMessage.append("<-- ").append(response.getStatusCode()).append(" ").append(HttpLoggingPolicy.getRedactedUrl(response.getRequest().getUrl(), HttpLoggingPolicy.this.allowedQueryParameterNames)).append(" (").append(loggingOptions.getResponseDuration().toMillis()).append(" ms, ").append(bodySize).append(")").append(System.lineSeparator());
            }
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogHeaders() && logger.canLogAtLevel(LogLevel.VERBOSE)) {
                HttpLoggingPolicy.addHeadersToLogMessage(HttpLoggingPolicy.this.allowedHeaderNames, response.getHeaders(), responseLogMessage);
            }
            if (!HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody()) {
                responseLogMessage.append("<-- END HTTP");
                return HttpLoggingPolicy.logAndReturn(logger, logLevel, responseLogMessage, response);
            }
            String contentTypeHeader = response.getHeaderValue("Content-Type");
            if (HttpLoggingPolicy.shouldBodyBeLogged(contentTypeHeader, contentLength = HttpLoggingPolicy.getContentLength(logger, response.getHeaders()))) {
                HttpResponse bufferedResponse = response.buffer();
                ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
                WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
                return bufferedResponse.getBody().flatMap(byteBuffer -> HttpLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, byteBuffer)).doFinally(ignored -> {
                    responseLogMessage.append("Response body:").append(System.lineSeparator()).append(HttpLoggingPolicy.prettyPrintIfNeeded(logger, HttpLoggingPolicy.this.prettyPrintBody, contentTypeHeader, HttpLoggingPolicy.convertStreamToString(outputStream, logger))).append(System.lineSeparator()).append("<-- END HTTP");
                    HttpLoggingPolicy.logAndReturn(logger, logLevel, responseLogMessage, response);
                }).then(Mono.just((Object)bufferedResponse));
            }
            responseLogMessage.append("(body content not logged)").append(System.lineSeparator()).append("<-- END HTTP");
            return HttpLoggingPolicy.logAndReturn(logger, logLevel, responseLogMessage, response);
        }
    }

    private final class DefaultHttpRequestLogger
    implements HttpRequestLogger {
        private DefaultHttpRequestLogger() {
        }

        @Override
        public Mono<Void> logRequest(ClientLogger logger, HttpRequestLoggingContext loggingOptions) {
            long contentLength;
            LogLevel logLevel = this.getLogLevel(loggingOptions);
            if (!logger.canLogAtLevel(logLevel)) {
                return Mono.empty();
            }
            HttpRequest request = loggingOptions.getHttpRequest();
            StringBuilder requestLogMessage = new StringBuilder();
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogUrl()) {
                requestLogMessage.append("--> ").append((Object)request.getHttpMethod()).append(" ").append(HttpLoggingPolicy.getRedactedUrl(request.getUrl(), HttpLoggingPolicy.this.allowedQueryParameterNames)).append(System.lineSeparator());
                Integer retryCount = loggingOptions.getTryCount();
                if (retryCount != null) {
                    requestLogMessage.append("Try count: ").append(retryCount).append(System.lineSeparator());
                }
            }
            if (HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogHeaders() && logger.canLogAtLevel(LogLevel.VERBOSE)) {
                HttpLoggingPolicy.addHeadersToLogMessage(HttpLoggingPolicy.this.allowedHeaderNames, request.getHeaders(), requestLogMessage);
            }
            if (!HttpLoggingPolicy.this.httpLogDetailLevel.shouldLogBody()) {
                return HttpLoggingPolicy.logAndReturn(logger, logLevel, requestLogMessage, null);
            }
            if (request.getBody() == null) {
                requestLogMessage.append("(empty body)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                return HttpLoggingPolicy.logAndReturn(logger, logLevel, requestLogMessage, null);
            }
            String contentType = request.getHeaders().getValue("Content-Type");
            if (HttpLoggingPolicy.shouldBodyBeLogged(contentType, contentLength = HttpLoggingPolicy.getContentLength(logger, request.getHeaders()))) {
                ByteArrayOutputStream outputStream = new ByteArrayOutputStream((int)contentLength);
                WritableByteChannel bodyContentChannel = Channels.newChannel(outputStream);
                request.setBody((Flux<ByteBuffer>)request.getBody().flatMap(byteBuffer -> HttpLoggingPolicy.writeBufferToBodyStream(bodyContentChannel, byteBuffer)).doFinally(ignored -> {
                    requestLogMessage.append(contentLength).append("-byte body:").append(System.lineSeparator()).append(HttpLoggingPolicy.prettyPrintIfNeeded(logger, HttpLoggingPolicy.this.prettyPrintBody, contentType, HttpLoggingPolicy.convertStreamToString(outputStream, logger))).append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
                    HttpLoggingPolicy.logAndReturn(logger, logLevel, requestLogMessage, null);
                }));
                return Mono.empty();
            }
            requestLogMessage.append(contentLength).append("-byte body: (content not logged)").append(System.lineSeparator()).append("--> END ").append((Object)request.getHttpMethod()).append(System.lineSeparator());
            return HttpLoggingPolicy.logAndReturn(logger, logLevel, requestLogMessage, null);
        }
    }
}

