/*
 * Decompiled with CFR 0.152.
 */
package com.azure.core.implementation;

import com.azure.core.credential.AccessToken;
import com.azure.core.credential.TokenCredential;
import com.azure.core.credential.TokenRequestContext;
import com.azure.core.util.logging.ClientLogger;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Signal;
import reactor.core.publisher.Sinks;

public final class AccessTokenCache {
    private static final Duration REFRESH_DELAY = Duration.ofSeconds(30L);
    private static final Duration REFRESH_OFFSET = Duration.ofMinutes(5L);
    private static final ClientLogger LOGGER = new ClientLogger(AccessTokenCache.class);
    private final AtomicReference<Sinks.One<AccessToken>> wip;
    private volatile AccessToken cache;
    private volatile OffsetDateTime nextTokenRefresh = OffsetDateTime.now();
    private final TokenCredential tokenCredential;
    private TokenRequestContext tokenRequestContext;
    private final Predicate<AccessToken> shouldRefresh;

    public AccessTokenCache(TokenCredential tokenCredential) {
        Objects.requireNonNull(tokenCredential, "The token credential cannot be null");
        this.wip = new AtomicReference();
        this.tokenCredential = tokenCredential;
        this.shouldRefresh = accessToken -> OffsetDateTime.now().isAfter(accessToken.getExpiresAt().minus(REFRESH_OFFSET));
    }

    public Mono<AccessToken> getToken(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        return Mono.defer(this.retrieveToken(tokenRequestContext, checkToForceFetchToken)).repeatWhenEmpty(longFlux -> longFlux.concatMap(ignored -> Flux.just((Object)true)));
    }

    public AccessToken getTokenSync(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        return (AccessToken)this.getToken(tokenRequestContext, checkToForceFetchToken).block();
    }

    private Supplier<Mono<? extends AccessToken>> retrieveToken(TokenRequestContext tokenRequestContext, boolean checkToForceFetchToken) {
        return () -> {
            try {
                if (tokenRequestContext == null) {
                    return Mono.error((Throwable)LOGGER.logExceptionAsError(new IllegalArgumentException("The token request context input cannot be null.")));
                }
                if (this.wip.compareAndSet(null, (Sinks.One<AccessToken>)Sinks.one())) {
                    Mono fallback;
                    Mono tokenRefresh;
                    Sinks.One<AccessToken> sinksOne = this.wip.get();
                    OffsetDateTime now = OffsetDateTime.now();
                    boolean forceRefresh = checkToForceFetchToken && this.checkIfForceRefreshRequired(tokenRequestContext) || this.tokenRequestContext == null;
                    Supplier<Mono> tokenSupplier = () -> this.tokenCredential.getToken(this.tokenRequestContext);
                    if (forceRefresh) {
                        this.tokenRequestContext = tokenRequestContext;
                        tokenRefresh = Mono.defer(() -> this.tokenCredential.getToken(this.tokenRequestContext));
                        fallback = Mono.empty();
                    } else if (this.cache != null && !this.shouldRefresh.test(this.cache)) {
                        tokenRefresh = Mono.empty();
                        fallback = Mono.just((Object)this.cache);
                    } else if (this.cache == null || this.cache.isExpired()) {
                        tokenRefresh = now.isAfter(this.nextTokenRefresh) ? Mono.defer(tokenSupplier) : Mono.defer(tokenSupplier).delaySubscription(Duration.between(now, this.nextTokenRefresh));
                        fallback = Mono.empty();
                    } else {
                        tokenRefresh = now.isAfter(this.nextTokenRefresh) ? Mono.defer(tokenSupplier) : Mono.empty();
                        fallback = Mono.just((Object)this.cache);
                    }
                    return tokenRefresh.materialize().flatMap(this.processTokenRefreshResult(sinksOne, now, (Mono<AccessToken>)fallback)).doOnError(arg_0 -> sinksOne.tryEmitError(arg_0)).doFinally(ignored -> this.wip.set(null));
                }
                if (this.cache != null && !this.cache.isExpired() && !checkToForceFetchToken) {
                    return Mono.just((Object)this.cache);
                }
                if (checkToForceFetchToken) {
                    return Mono.empty();
                }
                Sinks.One<AccessToken> sinksOne = this.wip.get();
                if (sinksOne == null) {
                    return Mono.just((Object)this.cache);
                }
                return sinksOne.asMono().switchIfEmpty(Mono.fromSupplier(() -> this.cache));
            }
            catch (Exception ex) {
                return Mono.error((Throwable)ex);
            }
        };
    }

    private boolean checkIfForceRefreshRequired(TokenRequestContext tokenRequestContext) {
        return this.tokenRequestContext == null || !(this.tokenRequestContext.getClaims() == null ? tokenRequestContext.getClaims() == null : tokenRequestContext.getClaims() != null && tokenRequestContext.getClaims().equals(this.tokenRequestContext.getClaims())) || !this.tokenRequestContext.getScopes().equals(tokenRequestContext.getScopes());
    }

    private Function<Signal<AccessToken>, Mono<? extends AccessToken>> processTokenRefreshResult(Sinks.One<AccessToken> sinksOne, OffsetDateTime now, Mono<AccessToken> fallback) {
        return signal -> {
            AccessToken accessToken = (AccessToken)signal.get();
            Throwable error = signal.getThrowable();
            if (signal.isOnNext() && accessToken != null) {
                LOGGER.info(AccessTokenCache.refreshLog(this.cache, now, "Acquired a new access token"));
                this.cache = accessToken;
                sinksOne.tryEmitValue((Object)accessToken);
                this.nextTokenRefresh = OffsetDateTime.now().plus(REFRESH_DELAY);
                return Mono.just((Object)accessToken);
            }
            if (signal.isOnError() && error != null) {
                LOGGER.error(AccessTokenCache.refreshLog(this.cache, now, "Failed to acquire a new access token"));
                this.nextTokenRefresh = OffsetDateTime.now().plus(REFRESH_DELAY);
                return fallback.switchIfEmpty(Mono.error((Throwable)error));
            }
            sinksOne.tryEmitEmpty();
            return fallback;
        };
    }

    private static String refreshLog(AccessToken cache, OffsetDateTime now, String log) {
        StringBuilder info = new StringBuilder(log);
        if (cache == null) {
            info.append(".");
        } else {
            Duration tte = Duration.between(now, cache.getExpiresAt());
            info.append(" at ").append(tte.abs().getSeconds()).append(" seconds ").append(tte.isNegative() ? "after" : "before").append(" expiry. ").append("Retry may be attempted after ").append(REFRESH_DELAY.getSeconds()).append(" seconds.");
            if (!tte.isNegative()) {
                info.append(" The token currently cached will be used.");
            }
        }
        return info.toString();
    }
}

