/*
 * Copyright 2006-2010 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.consol.citrus.ws.interceptor;

import com.consol.citrus.message.RawMessage;
import com.consol.citrus.report.MessageListeners;
import com.consol.citrus.util.XMLUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.ws.WebServiceMessage;
import org.springframework.ws.context.MessageContext;
import org.springframework.ws.soap.SoapMessage;
import org.springframework.xml.transform.TransformerObjectSupport;

import javax.xml.transform.*;
import javax.xml.transform.stream.StreamResult;
import java.io.*;

/**
 * Abstract logging support class offers basic log methods for SOAP messages.
 * 
 * @author Christoph Deppisch
 */
public abstract class LoggingInterceptorSupport extends TransformerObjectSupport {

    /** Logger */
    protected final Logger log = LoggerFactory.getLogger(getClass());
    
    @Autowired(required = false)
    private MessageListeners messageListener;
    
    /**
     * Prevent instantiation. 
     */
    protected LoggingInterceptorSupport() {
    }

    /**
     * Logs request message from message context. SOAP messages get logged with envelope transformation
     * other messages with serialization.
     * 
     * @param logMessage
     * @param messageContext
     * @param incoming
     * @throws TransformerException
     */
    protected void logRequest(String logMessage, MessageContext messageContext, boolean incoming) throws TransformerException {
        if (messageContext.getRequest() instanceof SoapMessage) {
            logSoapMessage(logMessage, (SoapMessage) messageContext.getRequest(), incoming);
        } else {
            logWebServiceMessage(logMessage, messageContext.getRequest(), incoming);
        }
    }
    
    /**
     * Logs response message from message context if any. SOAP messages get logged with envelope transformation
     * other messages with serialization.
     * 
     * @param logMessage
     * @param messageContext
     * @param incoming
     * @throws TransformerException
     */
    protected void logResponse(String logMessage, MessageContext messageContext, boolean incoming) throws TransformerException {
        if (messageContext.hasResponse()) {
            if (messageContext.getResponse() instanceof SoapMessage) {
                logSoapMessage(logMessage, (SoapMessage) messageContext.getResponse(), incoming);
            } else {
                logWebServiceMessage(logMessage, messageContext.getResponse(), incoming);
            }
        }
    }
    
    /**
     * Log SOAP message with transformer instance.
     * 
     * @param logMessage the customized log message.
     * @param soapMessage the message content as SOAP envelope source.
     * @param incoming
     * @throws TransformerException
     */
    protected void logSoapMessage(String logMessage, SoapMessage soapMessage, boolean incoming) throws TransformerException {
        Transformer transformer = createIndentingTransformer();
        StringWriter writer = new StringWriter();
        
        transformer.transform(soapMessage.getEnvelope().getSource(), new StreamResult(writer));
        logMessage(logMessage, XMLUtils.prettyPrint(writer.toString()), incoming);
    }
    
    /**
     * Log WebService message (other than SOAP) with in memory
     * {@link ByteArrayOutputStream}
     * 
     * @param logMessage the customized log message.
     * @param message the message to log.
     * @param incoming
     */
    protected void logWebServiceMessage(String logMessage, WebServiceMessage message, boolean incoming) {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        
        try {
            message.writeTo(os);
            logMessage(logMessage, os.toString(), incoming);
        } catch (IOException e) {
            log.warn("Unable to log WebService message", e);
        }
    }
    
    /**
     * Performs the final logger call with dynamic message.
     *
     * @param logMessage a custom log message entry.
     * @param message the message content.
     * @param incoming
     */
    protected void logMessage(String logMessage, String message, boolean incoming) {
        if (messageListener != null) {
            log.info(logMessage);

            if (incoming) {
                messageListener.onInboundMessage(new RawMessage(message), null);
            } else {
                messageListener.onOutboundMessage(new RawMessage(message), null);
            }
        } else {
            log.info(logMessage + ":" + System.getProperty("line.separator") + message);
        }
    }
    
    /**
     * Get transformer implementation with output properties set.
     * 
     * @return the transformer instance.
     * @throws TransformerConfigurationException
     */
    private Transformer createIndentingTransformer() throws TransformerConfigurationException {
        Transformer transformer = createTransformer();
        transformer.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
        transformer.setOutputProperty(OutputKeys.INDENT, "yes");
        return transformer;
    }

    /**
     * Sets the message listener.
     * @param messageListener
     */
    public void setMessageListener(MessageListeners messageListener) {
        this.messageListener = messageListener;
    }
}
