/*
 * $Id: Properties.java 109 2007-03-24 14:55:03Z max $
 * 
 * Copyright (c) 2006-2007 Maximilian Antoni. All rights reserved.
 * 
 * This software is licensed as described in the file LICENSE.txt, which you
 * should have received as part of this distribution. The terms are also
 * available at http://www.maxantoni.de/projects/eva-properties/license.txt.
 */
package com.eva.properties;

/**
 * base class for properties.
 * 
 * @author Max Antoni
 * @version $Revision: 109 $
 */
public abstract class Properties {
    /**
     * is the name of the property that references the base path of the
     * properties file. This may be a file or a URL, depending on the mechanism
     * used to load the properties file.
     */
    public static final String DATASOURCE_BASE = "datasource-base";
    
    /**
     * flags debug mode. In debug mode recursive references are detected and log
     * messages may be more detailed. Debug mode is also active when the log
     * level is at least <code>FINE</code>.
     */
    public static boolean DEBUG = false;
    
    /**
     * the parent properties for this properties object.
     */
    private Properties parent;
    
    /**
     * Default constructor.
     */
    protected Properties() {
        // Empty default constructor.
    }
    
    /**
     * creates new properties with parent properties.
     * 
     * @param inParent the parent properties.
     */
    protected Properties(Properties inParent) {
        parent = inParent;
    }
    
    /* 
     * @see java.lang.Object#toString()
     */
    public String toString() {
        Writer writer = new Writer();
        write(writer);
        return writer.toString();
    }
    
    /**
     * creates a deep copy of this properties object.
     * 
     * @param inParent the parent to use for the copy.
     * @return the new properties.
     */
    abstract Properties copy(Properties inParent);
    
    /**
     * returns the parent properties for this properties object.
     * 
     * @return the parent properties.
     */
    public Properties getParent() {
        return parent;
    }
    
    /**
     * sets the parent of this properties object.
     * 
     * @param inParent the parent.
     */
    public void setParent(Properties inParent) {
        if(parent != null) {
            throw new PropertiesException(
                    "Properties object has already a parent.");
        }
        parent = inParent;
    }
    
    /**
     * reads a property spcified by the given key from this properties object.
     * 
     * @param inContext the context.
     * @param inKey the key.
     * @return the value.
     * @throws PropertiesException
     */
    abstract Object getProperty(Context inContext, String inKey)
            throws PropertiesException;
    
    /**
     * associates an object with a key in this properties object.
     * 
     * @param inContext the context.
     * @param inKey the key.
     * @param inValue the value.
     * @return the previous object stored under the given key, or
     *         <code>null</code> if no such object exists.
     * @throws PropertiesException
     */
    abstract Object putProperty(Context inContext, String inKey, Object inValue)
            throws PropertiesException;

    /**
     * writes this properties object to the given writer.
     * 
     * @param inoutWriter the writer.
     */
    abstract void write(Writer inoutWriter);

}
