/*
 * Decompiled with CFR 0.152.
 */
package com.schibsted.security.strongbox.sdk.internal.encryption;

import com.amazonaws.auth.AWSCredentialsProvider;
import com.amazonaws.services.kms.AWSKMS;
import com.amazonaws.services.kms.AWSKMSClientBuilder;
import com.amazonaws.services.kms.model.CancelKeyDeletionRequest;
import com.amazonaws.services.kms.model.CreateAliasRequest;
import com.amazonaws.services.kms.model.CreateKeyRequest;
import com.amazonaws.services.kms.model.CreateKeyResult;
import com.amazonaws.services.kms.model.DescribeKeyRequest;
import com.amazonaws.services.kms.model.DescribeKeyResult;
import com.amazonaws.services.kms.model.EnableKeyRequest;
import com.amazonaws.services.kms.model.EnableKeyRotationRequest;
import com.amazonaws.services.kms.model.GenerateRandomRequest;
import com.amazonaws.services.kms.model.GenerateRandomResult;
import com.amazonaws.services.kms.model.KMSInvalidStateException;
import com.amazonaws.services.kms.model.KeyMetadata;
import com.amazonaws.services.kms.model.NotFoundException;
import com.amazonaws.services.kms.model.ScheduleKeyDeletionRequest;
import com.amazonaws.services.kms.model.ScheduleKeyDeletionResult;
import com.schibsted.security.strongbox.sdk.exceptions.AlreadyExistsException;
import com.schibsted.security.strongbox.sdk.exceptions.DoesNotExistException;
import com.schibsted.security.strongbox.sdk.exceptions.UnexpectedStateException;
import com.schibsted.security.strongbox.sdk.internal.ClientConfigurationHelper;
import com.schibsted.security.strongbox.sdk.internal.RegionLocalResourceName;
import com.schibsted.security.strongbox.sdk.internal.access.IAMPolicyManager;
import com.schibsted.security.strongbox.sdk.internal.encryption.KMSKeyState;
import com.schibsted.security.strongbox.sdk.internal.interfaces.ManagedResource;
import com.schibsted.security.strongbox.sdk.types.ClientConfiguration;
import com.schibsted.security.strongbox.sdk.types.SecretsGroupIdentifier;
import java.util.Date;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KMSManager
implements ManagedResource {
    private static final Logger log = LoggerFactory.getLogger(KMSManager.class);
    private static final int SLEEP_TIME = 100;
    private static final int MAX_RETRIES = 30;
    private final AWSKMS kms;
    private static final String ALIAS_PREFIX = "alias/";
    private final String aliasKeyName;
    private final SecretsGroupIdentifier group;
    private final AWSCredentialsProvider awsCredentials;
    private final ClientConfiguration clientConfiguration;

    public KMSManager(AWSKMS client, AWSCredentialsProvider awsCredentials, ClientConfiguration clientConfiguration, SecretsGroupIdentifier groupIdentifier) {
        this.kms = client;
        this.awsCredentials = awsCredentials;
        this.clientConfiguration = clientConfiguration;
        this.group = groupIdentifier;
        RegionLocalResourceName resourceName = new RegionLocalResourceName(groupIdentifier);
        this.aliasKeyName = ALIAS_PREFIX + resourceName.toString();
    }

    public static KMSManager fromCredentials(AWSCredentialsProvider awsCredentials, ClientConfiguration clientConfiguration, SecretsGroupIdentifier groupIdentifier) {
        AWSKMS client = (AWSKMS)((AWSKMSClientBuilder)((AWSKMSClientBuilder)((AWSKMSClientBuilder)AWSKMSClientBuilder.standard().withCredentials(awsCredentials)).withClientConfiguration(ClientConfigurationHelper.transformAndVerifyOrThrow(clientConfiguration))).withRegion(groupIdentifier.region.getName())).build();
        return new KMSManager(client, awsCredentials, clientConfiguration, groupIdentifier);
    }

    public String create(boolean allowExistingPendingDeletedOrDisabledKey) {
        String arn;
        Optional<KeyMetadata> current = this.describeKey();
        if (current.isPresent()) {
            if (!allowExistingPendingDeletedOrDisabledKey) {
                throw new AlreadyExistsException(String.format("KMS key already exists for group=%s,region=%s, and override not set", this.group.name, this.group.region.getName()));
            }
            arn = current.get().getArn();
            KMSKeyState state = KMSKeyState.fromString(current.get().getKeyState());
            switch (state) {
                case PENDING_DELETION: {
                    CancelKeyDeletionRequest request = new CancelKeyDeletionRequest();
                    request.withKeyId(arn);
                    this.kms.cancelKeyDeletion(request);
                }
                case DISABLED: {
                    EnableKeyRequest enableKeyRequest = new EnableKeyRequest();
                    enableKeyRequest.withKeyId(arn);
                    this.kms.enableKey(enableKeyRequest);
                    break;
                }
                default: {
                    throw new AlreadyExistsException(String.format("KMS key already exists for group=%s,region=%s", this.group.name, this.group.region.getName()));
                }
            }
        } else {
            CreateKeyRequest keyRequest = new CreateKeyRequest();
            keyRequest.setDescription("This key is automatically managed by Strongbox");
            CreateKeyResult result = this.kms.createKey(keyRequest);
            arn = result.getKeyMetadata().getArn();
            CreateAliasRequest createAliasRequest = new CreateAliasRequest();
            createAliasRequest.setAliasName(this.aliasKeyName);
            createAliasRequest.setTargetKeyId(arn);
            this.kms.createAlias(createAliasRequest);
            EnableKeyRotationRequest enableKeyRotationRequest = new EnableKeyRotationRequest();
            enableKeyRotationRequest.setKeyId(arn);
            this.kms.enableKeyRotation(enableKeyRotationRequest);
        }
        this.waitForKeyState(KMSKeyState.ENABLED);
        return arn;
    }

    @Override
    public String create() {
        return this.create(false);
    }

    private void waitForKeyState(KMSKeyState requiredState) {
        String currentState = "Unknown";
        try {
            for (int retries = 0; retries < 30; ++retries) {
                Optional<KeyMetadata> keyData = this.describeKey();
                if (keyData.isPresent() && KMSKeyState.fromString(currentState = keyData.get().getKeyState()) == requiredState) {
                    return;
                }
                log.info("Waiting for key to reach state:'{}'", (Object)requiredState.name());
                Thread.sleep(100L);
            }
        }
        catch (InterruptedException e) {
            throw new UnexpectedStateException(this.aliasKeyName, currentState, requiredState.name(), "Error occurred while waiting for KMS key to update", e);
        }
        throw new UnexpectedStateException(this.aliasKeyName, currentState, requiredState.name(), "KMS key did not reach expected state before timeout");
    }

    public byte[] generateRandom(Integer numberOfBytes) {
        GenerateRandomRequest request = new GenerateRandomRequest();
        request.withNumberOfBytes(numberOfBytes);
        GenerateRandomResult result = this.kms.generateRandom(request);
        return result.getPlaintext().array();
    }

    @Override
    public void delete() {
        this.deleteAndGetSchedule();
        this.waitForKeyState(KMSKeyState.PENDING_DELETION);
    }

    @Override
    public Optional<String> awsAdminPolicy() {
        return Optional.of("    {\n        \"Sid\": \"KMS\",\n        \"Effect\": \"Allow\",\n        \"Action\": [\n            \"kms:*\"\n        ],\n        \"Resource\": \"" + this.getArn() + "\"\n    }");
    }

    @Override
    public Optional<String> awsReadOnlyPolicy() {
        return Optional.of("    {\n        \"Sid\": \"KMS\",\n        \"Effect\": \"Allow\",\n        \"Action\": [\n            \"kms:Decrypt\",\n            \"kms:DescribeKey\"\n        ],\n        \"Resource\": \"" + this.getArn() + "\"\n    }");
    }

    @Override
    public String getArn() {
        Optional<KeyMetadata> metadata = this.describeKey();
        if (metadata.isPresent()) {
            return metadata.get().getArn();
        }
        throw new DoesNotExistException(String.format("Failed to find KMS key with alias '%s'", this.getAliasArn()));
    }

    @Override
    public boolean exists() {
        return this.exists(false);
    }

    public boolean exists(boolean allowExistingPendingDeletedOrDisabledKey) {
        Optional<KeyMetadata> current = this.describeKey();
        if (current.isPresent()) {
            if (!allowExistingPendingDeletedOrDisabledKey) {
                return true;
            }
            KMSKeyState state = KMSKeyState.fromString(current.get().getKeyState());
            switch (state) {
                case PENDING_DELETION: 
                case DISABLED: {
                    return false;
                }
            }
            return true;
        }
        return false;
    }

    public String getAliasArn() {
        return String.format("arn:aws:kms:%s:%s:%s", this.group.region.getName(), IAMPolicyManager.getAccount(this.awsCredentials, this.clientConfiguration), this.aliasKeyName);
    }

    private Optional<KeyMetadata> describeKey() {
        try {
            DescribeKeyRequest request = new DescribeKeyRequest();
            request.withKeyId(this.getAliasArn());
            DescribeKeyResult result = this.kms.describeKey(request);
            return Optional.of(result.getKeyMetadata());
        }
        catch (NotFoundException e) {
            return Optional.empty();
        }
    }

    public int pendingDeletionWindowInDays() {
        return 7;
    }

    private Date deleteAndGetSchedule() {
        String arn = this.getArn();
        try {
            int daysUntilDeletion = this.pendingDeletionWindowInDays();
            ScheduleKeyDeletionRequest request = new ScheduleKeyDeletionRequest();
            request.withKeyId(arn).withPendingWindowInDays(Integer.valueOf(daysUntilDeletion));
            ScheduleKeyDeletionResult result = this.kms.scheduleKeyDeletion(request);
            return result.getDeletionDate();
        }
        catch (KMSInvalidStateException e) {
            throw new UnexpectedStateException(arn, KMSKeyState.ENABLED.toString(), KMSKeyState.PENDING_DELETION.toString(), "Unable to delete KMS keys", e);
        }
    }
}

