package arrow.mtl.extensions.kleisli.functor

import arrow.Kind
import arrow.core.Tuple2
import arrow.mtl.ForKleisli
import arrow.mtl.Kleisli
import arrow.mtl.Kleisli.Companion
import arrow.mtl.extensions.KleisliFunctor
import arrow.typeclasses.Functor
import kotlin.Function1
import kotlin.Suppress
import kotlin.Unit
import kotlin.jvm.JvmName

/**
 * Transform the [F] wrapped value [A] into [B] preserving the [F] structure
 * Kind<F, A> -> Kind<F, B>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just<ForId, String, CharSequence>(Id.monad()).map<ForId, String, CharSequence,
 * String>(Id.monad(), { "$it World" })
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("map")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.map(FF: Functor<F>, arg1: Function1<A, B>):
    Kleisli<F, D, B> = arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@map.map<A, B>(arg1) as arrow.mtl.Kleisli<F, D, B>
}

@JvmName("imap")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.imap(
  FF: Functor<F>,
  arg1: Function1<A, B>,
  arg2: Function1<B, A>
): Kleisli<F, D, B> = arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@imap.imap<A, B>(arg1, arg2) as arrow.mtl.Kleisli<F, D, B>
}

/**
 * Lifts a function `A -> B` to the [F] structure returning a polymorphic function
 * that can be applied over all [F] values in the shape of Kind<F, A>
 *
 * `A -> B -> Kind<F, A> -> Kind<F, B>`
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   lift<ForId, String, CharSequence, String>(Id.monad(),
 * { s: CharSequence -> "$s World" })("Hello".just<ForId, String, CharSequence>(Id.monad()))
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("lift")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> lift(FF: Functor<F>, arg0: Function1<A, B>): Function1<Kind<Kind<Kind<ForKleisli,
    F>, D>, A>, Kind<Kind<Kind<ForKleisli, F>, D>, B>> = arrow.mtl.Kleisli
   .functor<F, D>(FF)
   .lift<A, B>(arg0) as kotlin.Function1<arrow.Kind<arrow.Kind<arrow.Kind<arrow.mtl.ForKleisli, F>,
    D>, A>, arrow.Kind<arrow.Kind<arrow.Kind<arrow.mtl.ForKleisli, F>, D>, B>>

/**
 * Discards the [A] value inside [F] signaling this container may be pointing to a noop
 * or an effect whose return value is deliberately ignored. The singleton value [Unit] serves as signal.
 *
 * Kind<F, A> -> Kind<F, Unit>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello World".just<ForId, String, CharSequence>(Id.monad()).unit<ForId, String,
 * CharSequence>(Id.monad())
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("unit")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A> Kind<Kind<Kind<ForKleisli, F>, D>, A>.unit(FF: Functor<F>): Kleisli<F, D, Unit> =
    arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@unit.unit<A>() as arrow.mtl.Kleisli<F, D, kotlin.Unit>
}

/**
 * Applies [f] to an [A] inside [F] and returns the [F] structure with a tuple of the [A] value and the
 * computed [B] value as result of applying [f]
 *
 * Kind<F, A> -> Kind<F, Tuple2<A, B>>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just<ForId, String, CharSequence>(Id.monad()).fproduct<ForId, String, CharSequence,
 * String>(Id.monad(), { "$it World" })
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("fproduct")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.fproduct(FF: Functor<F>, arg1: Function1<A,
    B>): Kleisli<F, D, Tuple2<A, B>> = arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@fproduct.fproduct<A, B>(arg1) as arrow.mtl.Kleisli<F, D, arrow.core.Tuple2<A, B>>
}

/**
 * Replaces [A] inside [F] with [B] resulting in a Kind<F, B>
 *
 * Kind<F, A> -> Kind<F, B>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello World".just<ForId, String, CharSequence>(Id.monad()).`as`<ForId, String, CharSequence,
 * String>(Id.monad(), "...")
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("as")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.`as`(FF: Functor<F>, arg1: B): Kleisli<F, D,
    B> = arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@`as`.`as`<A, B>(arg1) as arrow.mtl.Kleisli<F, D, B>
}

/**
 * Pairs [B] with [A] returning a Kind<F, Tuple2<B, A>>
 *
 * Kind<F, A> -> Kind<F, Tuple2<B, A>>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just<ForId, String, CharSequence>(Id.monad()).tupleLeft<ForId, String, CharSequence,
 * String>(Id.monad(), "World")
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("tupleLeft")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.tupleLeft(FF: Functor<F>, arg1: B):
    Kleisli<F, D, Tuple2<B, A>> = arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@tupleLeft.tupleLeft<A, B>(arg1) as arrow.mtl.Kleisli<F, D, arrow.core.Tuple2<B, A>>
}

/**
 * Pairs [A] with [B] returning a Kind<F, Tuple2<A, B>>
 *
 * Kind<F, A> -> Kind<F, Tuple2<A, B>>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   "Hello".just<ForId, String, CharSequence>(Id.monad()).tupleRight<ForId, String, CharSequence,
 * String>(Id.monad(), "World")
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("tupleRight")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, A, B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.tupleRight(FF: Functor<F>, arg1: B):
    Kleisli<F, D, Tuple2<A, B>> = arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@tupleRight.tupleRight<A, B>(arg1) as arrow.mtl.Kleisli<F, D, arrow.core.Tuple2<A, B>>
}

/**
 * Given [A] is a sub type of [B], re-type this value from Kind<F, A> to Kind<F, B>
 *
 * Kind<F, A> -> Kind<F, B>
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 * import arrow.mtl.extensions.kleisli.applicative.just
 * import arrow.Kind
 *
 * fun main(args: Array<String>) {
 *   val result: Kind<*, CharSequence> =
 *   //sampleStart
 *   "Hello".just<ForId, String, CharSequence>(Id.monad()).map<ForId, String, CharSequence,
 * String>(Id.monad(), { "$it World" }).widen<ForId, String, CharSequence, String>(Id.monad())
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@JvmName("widen")
@Suppress(
    "UNCHECKED_CAST",
    "USELESS_CAST",
    "EXTENSION_SHADOWED_BY_MEMBER",
    "UNUSED_PARAMETER"
)
fun <F, D, B, A : B> Kind<Kind<Kind<ForKleisli, F>, D>, A>.widen(FF: Functor<F>): Kleisli<F, D, B> =
    arrow.mtl.Kleisli.functor<F, D>(FF).run {
  this@widen.widen<B, A>() as arrow.mtl.Kleisli<F, D, B>
}

/**
 * ank_macro_hierarchy(arrow.typeclasses.Functor)
 *
 * The [Functor] type class abstracts the ability to [map] over the computational context of a type constructor.
 * Examples of type constructors that can implement instances of the Functor type class include [Kleisli],
 * [arrow.core.Option], [arrow.core.NonEmptyList], [List] and many other data types that include a [map] function with the shape
 * `fun <F, A, B> Kind<F, A>.map(f: (A) -> B): Kind<F, B>` where `F` refers to any type constructor whose contents can be transformed.
 *
 * ```kotlin:ank:playground
 * import arrow.mtl.*
 * import arrow.mtl.extensions.kleisli.functor.*
 * import arrow.core.*
 * import arrow.core.extensions.id.monad.monad
 * import arrow.core.*
 *
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   Kleisli.functor<ForId, String>(Id.monad())
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 *
 * ### Example
 *
 * Oftentimes we find ourselves in situations where we need to transform the contents of some data type.
 * [map] allows us to safely compute over values under the assumption that they'll be there returning the
 * transformation encapsulated in the same context.
 *
 * Consider both [arrow.core.Option] and [arrow.core.Try]:
 *
 * `Option<A>` allows us to model absence and has two possible states, `Some(a: A)` if the value is not absent and `None` to represent an empty case.
 * In a similar fashion `Try<A>` may have two possible cases `Success(a: A)` for computations that succeed and `Failure(e: Throwable)` if they fail
 * with an exception.
 *
 * Both [arrow.core.Try] and [arrow.core.Option] are examples of data types that can be computed over transforming their inner results.
 *
 * ```kotlin:ank:playground
 * import arrow.*
 * import arrow.core.*
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   Try { "1".toInt() }.map { it * 2 }
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 *
 * ```kotlin:ank:playground
 * import arrow.*
 * import arrow.core.*
 *
 * fun main(args: Array<String>) {
 *   val result =
 *   //sampleStart
 *   Option(1).map { it * 2 }
 *   //sampleEnd
 *   println(result)
 * }
 * ```
 */
@Suppress(
    "UNCHECKED_CAST",
    "NOTHING_TO_INLINE"
)
inline fun <F, D> Companion.functor(FF: Functor<F>): KleisliFunctor<F, D> = object :
    arrow.mtl.extensions.KleisliFunctor<F, D> { override fun FF(): arrow.typeclasses.Functor<F> = FF
    }