/*
 * Decompiled with CFR 0.152.
 */
package io.github.microcks.util.grpc;

import com.github.os72.protocjar.Protoc;
import com.google.protobuf.DescriptorProtos;
import com.google.protobuf.Descriptors;
import io.github.microcks.domain.Exchange;
import io.github.microcks.domain.Operation;
import io.github.microcks.domain.Resource;
import io.github.microcks.domain.ResourceType;
import io.github.microcks.domain.Service;
import io.github.microcks.domain.ServiceType;
import io.github.microcks.util.MockRepositoryImportException;
import io.github.microcks.util.MockRepositoryImporter;
import io.github.microcks.util.ReferenceResolver;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ProtobufImporter
implements MockRepositoryImporter {
    private static final Logger log = LoggerFactory.getLogger(ProtobufImporter.class);
    private static final String BINARY_DESCRIPTOR_EXT = ".pbb";
    private static final String BUILTIN_LIBRARY_PREFIX = "google/protobuf";
    private final String protoDirectory;
    private final String protoFileName;
    private final ReferenceResolver referenceResolver;
    private String specContent;
    private DescriptorProtos.FileDescriptorSet fds;

    public ProtobufImporter(String protoFilePath, ReferenceResolver referenceResolver) throws IOException {
        this.referenceResolver = referenceResolver;
        File protoFile = new File(protoFilePath);
        this.protoDirectory = protoFile.getParentFile().getAbsolutePath();
        this.protoFileName = protoFile.getName();
        String[] args = new String[]{"-v3.21.8", "--include_std_types", "--include_imports", "--proto_path=" + this.protoDirectory, "--descriptor_set_out=" + this.protoDirectory + "/" + this.protoFileName + BINARY_DESCRIPTOR_EXT, this.protoFileName};
        ArrayList<File> resolvedImportsLocalFiles = null;
        try {
            byte[] bytes = Files.readAllBytes(Paths.get(protoFilePath, new String[0]));
            this.specContent = new String(bytes, StandardCharsets.UTF_8);
            if (referenceResolver != null) {
                resolvedImportsLocalFiles = new ArrayList<File>();
                this.resolveAndPrepareRemoteImports(Paths.get(protoFilePath, new String[0]), resolvedImportsLocalFiles);
            }
            int result = Protoc.runProtoc((String[])args);
            File protoFileB = new File(this.protoDirectory, this.protoFileName + BINARY_DESCRIPTOR_EXT);
            this.fds = DescriptorProtos.FileDescriptorSet.parseFrom((InputStream)new FileInputStream(protoFileB));
        }
        catch (InterruptedException ie) {
            log.error("Protobuf schema compilation has been interrupted on {}", (Object)protoFilePath);
            Thread.currentThread().interrupt();
        }
        catch (Exception e) {
            throw new IOException("Protobuf schema file parsing error on " + protoFilePath + ": " + e.getMessage());
        }
        finally {
            if (resolvedImportsLocalFiles != null) {
                resolvedImportsLocalFiles.forEach(File::delete);
            }
        }
    }

    @Override
    public List<Service> getServiceDefinitions() throws MockRepositoryImportException {
        ArrayList<Service> results = new ArrayList<Service>();
        ArrayList<Descriptors.FileDescriptor> dependencies = new ArrayList<Descriptors.FileDescriptor>();
        for (DescriptorProtos.FileDescriptorProto fdp : this.fds.getFileList()) {
            String packageName = fdp.getPackage();
            String[] parts = packageName.split("\\.");
            String version = parts.length > 2 ? parts[parts.length - 1] : packageName;
            Descriptors.FileDescriptor fd = null;
            try {
                fd = Descriptors.FileDescriptor.buildFrom((DescriptorProtos.FileDescriptorProto)fdp, (Descriptors.FileDescriptor[])dependencies.toArray(new Descriptors.FileDescriptor[dependencies.size()]), (boolean)true);
                dependencies.add(fd);
            }
            catch (Descriptors.DescriptorValidationException e) {
                throw new MockRepositoryImportException("Exception while building Protobuf descriptor, probably a missing dependency issue: " + e.getMessage());
            }
            for (Descriptors.ServiceDescriptor sd : fd.getServices()) {
                Service service = new Service();
                service.setName(sd.getFullName());
                service.setVersion(version);
                service.setType(ServiceType.GRPC);
                service.setXmlNS(fd.getPackage());
                service.setOperations(this.extractOperations(sd));
                results.add(service);
            }
        }
        return results;
    }

    @Override
    public List<Resource> getResourceDefinitions(Service service) throws MockRepositoryImportException {
        ArrayList<Resource> results = new ArrayList<Resource>();
        Resource textResource = new Resource();
        textResource.setName(service.getName() + "-" + service.getVersion() + ".proto");
        textResource.setType(ResourceType.PROTOBUF_SCHEMA);
        textResource.setContent(this.specContent);
        results.add(textResource);
        try {
            byte[] binaryPB = Files.readAllBytes(Path.of(this.protoDirectory, this.protoFileName + BINARY_DESCRIPTOR_EXT));
            String base64PB = new String(Base64.getEncoder().encode(binaryPB), StandardCharsets.UTF_8);
            Resource descResource = new Resource();
            descResource.setName(service.getName() + "-" + service.getVersion() + BINARY_DESCRIPTOR_EXT);
            descResource.setType(ResourceType.PROTOBUF_DESCRIPTOR);
            descResource.setContent(base64PB);
            results.add(descResource);
        }
        catch (Exception e) {
            log.error("Exception while encoding Protobuf binary descriptor into base64", (Throwable)e);
            throw new MockRepositoryImportException("Exception while encoding Protobuf binary descriptor into base64");
        }
        if (this.referenceResolver != null) {
            this.referenceResolver.getRelativeResolvedReferences().forEach((p, f) -> {
                Resource protoResource = new Resource();
                protoResource.setName(service.getName() + "-" + service.getVersion() + "-" + p.replace("/", "~1"));
                protoResource.setType(ResourceType.PROTOBUF_SCHEMA);
                protoResource.setPath(p);
                try {
                    protoResource.setContent(Files.readString(f.toPath(), StandardCharsets.UTF_8));
                }
                catch (IOException ioe) {
                    log.warn("Exception while setting content of {} Protobuf resource", (Object)protoResource.getName(), (Object)ioe);
                    log.warn("Pursuing on next resource as it was for information purpose only");
                }
                results.add(protoResource);
            });
            this.referenceResolver.cleanResolvedReferences();
        }
        return results;
    }

    @Override
    public List<Exchange> getMessageDefinitions(Service service, Operation operation) throws MockRepositoryImportException {
        return new ArrayList<Exchange>();
    }

    private void resolveAndPrepareRemoteImports(Path protoFilePath, List<File> resolvedImportsLocalFiles) {
        String line = null;
        try (BufferedReader reader = Files.newBufferedReader(protoFilePath, StandardCharsets.UTF_8);){
            while ((line = reader.readLine()) != null) {
                Path importPath;
                if (!(line = line.trim()).startsWith("import ")) continue;
                String importStr = line.substring("import ".length() + 1);
                if (importStr.endsWith(";")) {
                    importStr = importStr.substring(0, importStr.length() - 1);
                }
                if (importStr.endsWith("\"") || importStr.endsWith("'")) {
                    importStr = importStr.substring(0, importStr.length() - 1);
                }
                if (importStr.startsWith("\"") || importStr.startsWith("'")) {
                    importStr = importStr.substring(1);
                }
                log.debug("Found an import to resolve in protobuf: {}", (Object)importStr);
                if (importStr.startsWith(BUILTIN_LIBRARY_PREFIX) || Files.exists(importPath = protoFilePath.getParent().resolve(importStr), new LinkOption[0])) continue;
                String importContent = this.referenceResolver.getReferenceContent(importStr, StandardCharsets.UTF_8);
                try {
                    Files.createDirectories(importPath.getParent(), new FileAttribute[0]);
                    Files.createFile(importPath, new FileAttribute[0]);
                }
                catch (FileAlreadyExistsException faee) {
                    log.warn("Exception while writing protobuf dependency", (Throwable)faee);
                }
                Files.write(importPath, importContent.getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
                resolvedImportsLocalFiles.add(importPath.toFile());
                this.resolveAndPrepareRemoteImports(importPath, resolvedImportsLocalFiles);
            }
        }
        catch (Exception e) {
            log.error("Exception while retrieving protobuf dependency", (Throwable)e);
        }
    }

    private List<Operation> extractOperations(Descriptors.ServiceDescriptor service) {
        ArrayList<Operation> results = new ArrayList<Operation>();
        for (Descriptors.MethodDescriptor method : service.getMethods()) {
            Operation operation = new Operation();
            operation.setName(method.getName());
            if (method.getInputType() != null) {
                operation.setInputName("." + method.getInputType().getFullName());
                boolean hasOnlyPrimitiveArgs = ProtobufImporter.hasOnlyPrimitiveArgs(method);
                if (hasOnlyPrimitiveArgs && !method.getInputType().getFields().isEmpty()) {
                    operation.setDispatcher("QUERY_ARGS");
                    operation.setDispatcherRules(ProtobufImporter.extractOperationParams(method.getInputType().getFields()));
                }
            }
            if (method.getOutputType() != null) {
                operation.setOutputName("." + method.getOutputType().getFullName());
            }
            results.add(operation);
        }
        return results;
    }

    private static boolean hasOnlyPrimitiveArgs(Descriptors.MethodDescriptor method) {
        for (Descriptors.FieldDescriptor field : method.getInputType().getFields()) {
            Descriptors.FieldDescriptor.Type fieldType = field.getType();
            if (ProtobufImporter.isScalarType(fieldType)) continue;
            return false;
        }
        return true;
    }

    private static boolean isScalarType(Descriptors.FieldDescriptor.Type fieldType) {
        return fieldType != Descriptors.FieldDescriptor.Type.MESSAGE && fieldType != Descriptors.FieldDescriptor.Type.GROUP && fieldType != Descriptors.FieldDescriptor.Type.BYTES;
    }

    private static String extractOperationParams(List<Descriptors.FieldDescriptor> inputFields) {
        StringBuilder builder = new StringBuilder();
        for (Descriptors.FieldDescriptor inputField : inputFields) {
            builder.append(inputField.getName()).append(" && ");
        }
        return builder.substring(0, builder.length() - 4);
    }
}

