/*
 * Copyright 2017-2018 original authors
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.micronaut.http.server;

import io.micronaut.context.annotation.ConfigurationProperties;
import io.micronaut.core.convert.format.ReadableBytes;
import io.micronaut.core.util.StringUtils;
import io.micronaut.core.util.Toggleable;
import io.micronaut.http.server.cors.CorsOriginConfiguration;
import io.micronaut.runtime.ApplicationConfiguration;

import javax.inject.Inject;
import java.io.File;
import java.nio.charset.Charset;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;

/**
 * <p>A base {@link ConfigurationProperties} for servers.</p>
 *
 * @author Graeme Rocher
 * @since 1.0
 */
@ConfigurationProperties(value = HttpServerConfiguration.PREFIX, cliPrefix = "")
public class HttpServerConfiguration {

    /**
     * The default port value.
     */
    @SuppressWarnings("WeakerAccess")
    public static final int DEFAULT_PORT = 8080;

    /**
     * The prefix used for configuration.
     */

    public static final String PREFIX = "micronaut.server";

    /**
     * The default value random port.
     */
    @SuppressWarnings("WeakerAccess")
    public static final int DEFAULT_RANDOM_PORT = -1;

    /**
     * The default max request size.
     */
    @SuppressWarnings("WeakerAccess")
    public static final long DEFAULT_MAX_REQUEST_SIZE = 1024 * 1024 * 10; // 10MB

    /**
     * The default read idle time in minutes.
     */
    @SuppressWarnings("WeakerAccess")
    public static final long DEFAULT_READ_IDLE_TIME_MINUTES = 5;

    /**
     * The default write idle time in minutes.
     */
    @SuppressWarnings("WeakerAccess")
    public static final long DEFAULT_WRITE_IDLE_TIME_MINUTES = 5;

    /**
     * The default date header.
     */
    @SuppressWarnings("WeakerAccess")
    public static final boolean DEFAULT_DATEHEADER = true;

    /**
     * The default idle time.
     */
    @SuppressWarnings("WeakerAccess")
    public static final long DEFAULT_IDLE_TIME_MINUTES = 5;

    /**
     * The default value for log handled exceptions.
     */
    @SuppressWarnings("WeakerAccess")
    public static final boolean DEFAULT_LOG_HANDLED_EXCEPTIONS = false;

    private Integer port;
    private String host;
    private Integer readTimeout;
    private long maxRequestSize = DEFAULT_MAX_REQUEST_SIZE;
    private Duration readIdleTimeout = null;
    private Duration writeIdleTimeout = null;
    private Duration idleTimeout = Duration.of(DEFAULT_IDLE_TIME_MINUTES, ChronoUnit.MINUTES);
    private MultipartConfiguration multipart = new MultipartConfiguration();
    private CorsConfiguration cors = new CorsConfiguration();
    private String serverHeader;
    private boolean dateHeader = DEFAULT_DATEHEADER;
    private boolean logHandledExceptions = DEFAULT_LOG_HANDLED_EXCEPTIONS;

    private final ApplicationConfiguration applicationConfiguration;
    private Charset defaultCharset;

    /**
     * Default constructor.
     */
    public HttpServerConfiguration() {
        this.applicationConfiguration = new ApplicationConfiguration();
    }

    /**
     * @param applicationConfiguration The application configuration
     */
    @Inject
    public HttpServerConfiguration(ApplicationConfiguration applicationConfiguration) {
        if (applicationConfiguration != null) {
            this.defaultCharset = applicationConfiguration.getDefaultCharset();
        }

        this.applicationConfiguration = applicationConfiguration;
    }

    /**
     * @return The application configuration instance
     */
    public ApplicationConfiguration getApplicationConfiguration() {
        return applicationConfiguration;
    }

    /**
     * @return The default charset to use
     */
    public Charset getDefaultCharset() {
        return defaultCharset;
    }

    /**
     * @param defaultCharset The default charset to use
     */
    public void setDefaultCharset(Charset defaultCharset) {
        this.defaultCharset = defaultCharset;
    }

    /**
     * @return The default server port
     */
    public Optional<Integer> getPort() {
        return Optional.ofNullable(port);
    }

    /**
     * @return The default host
     */
    public Optional<String> getHost() {
        return Optional.ofNullable(host);
    }

    /**
     * @return The read timeout setting for the server
     */
    public Optional<Integer> getReadTimeout() {
        return Optional.ofNullable(readTimeout);
    }

    /**
     * @return Configuration for multipart / file uploads
     */
    public MultipartConfiguration getMultipart() {
        return multipart;
    }

    /**
     * @return Configuration for CORS
     */
    public CorsConfiguration getCors() {
        return cors;
    }

    /**
     * @return The maximum request body size
     */
    public long getMaxRequestSize() {
        return maxRequestSize;
    }

    /**
     * @return The default amount of time to allow read operation connections  to remain idle
     */
    public Duration getReadIdleTimeout() {
        return Optional.ofNullable(readIdleTimeout).orElse(idleTimeout);
    }

    /**
     * @return The default amount of time to allow write operation connections to remain idle
     */
    public Duration getWriteIdleTimeout() {
        return Optional.ofNullable(writeIdleTimeout).orElse(idleTimeout);
    }

    /**
     * @return The time to allow an idle connection for
     */
    public Duration getIdleTimeout() {
        return idleTimeout;
    }

    /**
     * @return The optional server header value
     */
    public Optional<String> getServerHeader() {
        return Optional.ofNullable(serverHeader);
    }

    /**
     * @return True if the date header should be set
     */
    public boolean isDateHeader() {
        return dateHeader;
    }

    /**
     * @return True if exceptions handled by either an error
     * route or exception handler should be logged
     */
    public boolean isLogHandledExceptions() {
        return logHandledExceptions;
    }

    /**
     * Sets the port to bind to. Default value ({@value #DEFAULT_RANDOM_PORT})
     *
     * @param port The port
     */
    public void setPort(int port) {
        this.port = port;
    }

    /**
     * Sets the host to bind to.
     * @param host The host
     */
    public void setHost(String host) {
        if (StringUtils.isNotEmpty(host)) {
            this.host = host;
        }
    }

    /**
     * Sets the default read timeout.
     *
     * @param readTimeout The read timeout
     */
    public void setReadTimeout(Integer readTimeout) {
        this.readTimeout = readTimeout;
    }

    /**
     * Sets the name of the server header.
     *
     * @param serverHeader The server header
     */
    public void setServerHeader(String serverHeader) {
        this.serverHeader = serverHeader;
    }

    /**
     * Sets the maximum request size. Default value ({@value #DEFAULT_MAX_REQUEST_SIZE} => // 10MB)
     *
     * @param maxRequestSize The max request size
     */
    public void setMaxRequestSize(@ReadableBytes long maxRequestSize) {
        this.maxRequestSize = maxRequestSize;
    }

    /**
     * Sets the amount of time a connection can remain idle without any reads occurring. Default value ({@value #DEFAULT_READ_IDLE_TIME_MINUTES} seconds).
     *
     * @param readIdleTimeout The read idle time
     */
    public void setReadIdleTimeout(Duration readIdleTimeout) {
        this.readIdleTimeout = readIdleTimeout;
    }

    /**
     * Sets the amount of time a connection can remain idle without any writes occurring. Default value ({@value #DEFAULT_WRITE_IDLE_TIME_MINUTES} seconds).
     *
     * @param writeIdleTimeout The write idle time
     */
    public void setWriteIdleTimeout(Duration writeIdleTimeout) {
        this.writeIdleTimeout = writeIdleTimeout;
    }

    /**
     * Sets the idle time of connections for the server. Default value ({@value #DEFAULT_IDLE_TIME_MINUTES} seconds).
     *
     * @param idleTimeout The idle time
     */
    public void setIdleTimeout(Duration idleTimeout) {
        if (idleTimeout != null) {
            this.idleTimeout = idleTimeout;
        }
    }

    /**
     * Sets the multipart configuration.
     *
     * @param multipart The multipart configuration
     */
    public void setMultipart(MultipartConfiguration multipart) {
        this.multipart = multipart;
    }

    /**
     * Sets the cors configuration.
     * @param cors The cors configuration
     */
    public void setCors(CorsConfiguration cors) {
        this.cors = cors;
    }

    /**
     * Sets whether a date header should be sent back. Default value ({@value #DEFAULT_DATEHEADER}).
     *
     * @param dateHeader True if a date header should be sent.
     */
    public void setDateHeader(boolean dateHeader) {
        this.dateHeader = dateHeader;
    }

    /**
     * Sets whether exceptions handled by either an error route or exception handler
     * should still be logged. Default value ({@value #DEFAULT_LOG_HANDLED_EXCEPTIONS }).
     *
     * @param logHandledExceptions True if exceptions should be logged
     */
    public void setLogHandledExceptions(boolean logHandledExceptions) {
        this.logHandledExceptions = logHandledExceptions;
    }

    /**
     * Configuration for multipart handling.
     */
    @ConfigurationProperties("multipart")
    public static class MultipartConfiguration implements Toggleable {

        /**
         * The default enable value.
         */
        @SuppressWarnings("WeakerAccess")
        public static final boolean DEFAULT_ENABLED = false;

        /**
         * The default max file size.
         */
        @SuppressWarnings("WeakerAccess")
        public static final long DEFAULT_MAX_FILE_SIZE = 1024 * 1024; // 1MB

        /**
         * The default disk value.
         */
        @SuppressWarnings("WeakerAccess")
        public static final boolean DEFAULT_DISK = false;

        private File location;
        private long maxFileSize = DEFAULT_MAX_FILE_SIZE;
        private boolean enabled = DEFAULT_ENABLED;
        private boolean disk = DEFAULT_DISK;

        /**
         * @return The location to store temporary files
         */
        public Optional<File> getLocation() {
            return Optional.ofNullable(location);
        }

        /**
         * @return The max file size. Defaults to 1MB
         */
        public long getMaxFileSize() {
            return maxFileSize;
        }

        /**
         * @return Whether file uploads are enabled. Defaults to true.
         */
        @Override
        public boolean isEnabled() {
            return enabled;
        }

        /**
         * @return Whether to use disk. Defaults to false.
         */
        public boolean isDisk() {
            return disk;
        }

        /**
         * Sets the location to store files.
         * @param location The location
         */
        public void setLocation(File location) {
            this.location = location;
        }

        /**
         * Sets the max file size. Default value ({@value #DEFAULT_MAX_FILE_SIZE} => 1MB).
         * @param maxFileSize The max file size
         */
        public void setMaxFileSize(@ReadableBytes long maxFileSize) {
            this.maxFileSize = maxFileSize;
        }

        /**
         * Sets whether multipart processing is enabled. Default value ({@value #DEFAULT_ENABLED}).
         * @param enabled True if it is enabled
         */
        public void setEnabled(boolean enabled) {
            this.enabled = enabled;
        }

        /**
         * Sets whether to buffer data to disk or not. Default value ({@value #DEFAULT_DISK}).
         * @param disk True if data should be written to disk
         */
        public void setDisk(boolean disk) {
            this.disk = disk;
        }
    }

    /**
     * Configuration for CORS.
     */
    @ConfigurationProperties("cors")
    public static class CorsConfiguration implements Toggleable {

        public static final boolean DEFAULT_ENABLED = false;

        private boolean enabled = DEFAULT_ENABLED;

        private Map<String, CorsOriginConfiguration> configurations = Collections.emptyMap();

        private Map<String, CorsOriginConfiguration> defaultConfiguration = new LinkedHashMap<>(1);

        /**
         * @return Whether cors is enabled. Defaults to false.
         */
        @Override
        public boolean isEnabled() {
            return enabled;
        }

        /**
         * @return The cors configurations
         */
        public Map<String, CorsOriginConfiguration> getConfigurations() {
            if (enabled && configurations.isEmpty()) {
                if (defaultConfiguration.isEmpty()) {
                    defaultConfiguration.put("default", new CorsOriginConfiguration());
                }
                return defaultConfiguration;
            }
            return configurations;
        }

        /**
         * Sets whether CORS is enabled. Default value ({@value #DEFAULT_ENABLED})
         * @param enabled True if CORS is enabled
         */
        public void setEnabled(boolean enabled) {
            this.enabled = enabled;
        }

        /**
         * Sets the CORS configurations.
         * @param configurations The CORS configurations
         */
        public void setConfigurations(Map<String, CorsOriginConfiguration> configurations) {
            this.configurations = configurations;
        }
    }
}
