/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.rxjava.core.http;

import java.util.Map;
import rx.Observable;
import rx.Single;
import io.vertx.rxjava.core.buffer.Buffer;
import io.vertx.core.http.GoAway;
import io.vertx.core.http.Http2Settings;
import io.vertx.core.AsyncResult;
import io.vertx.core.Handler;
import io.vertx.rxjava.core.net.SocketAddress;

/**
 * Represents an HTTP connection.
 * <p/>
 * HTTP/1.x connection provides an limited implementation, the following methods are implemented:
 * <ul>
 *   <li>{@link io.vertx.rxjava.core.http.HttpConnection#close}</li>
 *   <li>{@link io.vertx.rxjava.core.http.HttpConnection#closeHandler}</li>
 *   <li>{@link io.vertx.rxjava.core.http.HttpConnection#exceptionHandler}</li>
 * </ul>
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpConnection original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rxjava.RxGen(io.vertx.core.http.HttpConnection.class)
public class HttpConnection {

  public static final io.vertx.lang.rxjava.TypeArg<HttpConnection> __TYPE_ARG = new io.vertx.lang.rxjava.TypeArg<>(
    obj -> new HttpConnection((io.vertx.core.http.HttpConnection) obj),
    HttpConnection::getDelegate
  );

  private final io.vertx.core.http.HttpConnection delegate;
  
  public HttpConnection(io.vertx.core.http.HttpConnection delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.http.HttpConnection getDelegate() {
    return delegate;
  }

  /**
   * @return the current connection window size or <code>-1</code> for HTTP/1.x
   */
  public int getWindowSize() { 
    int ret = delegate.getWindowSize();
    return ret;
  }

  /**
   * Update the current connection wide window size to a new size.
   * <p/>
   * Increasing this value, gives better performance when several data streams are multiplexed
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param windowSize the new window size
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection setWindowSize(int windowSize) { 
    delegate.setWindowSize(windowSize);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpConnection#goAway} with a last stream id <code>2^31-1</code>.
   * @param errorCode 
   * @return 
   */
  public HttpConnection goAway(long errorCode) { 
    delegate.goAway(errorCode);
    return this;
  }

  /**
   * Like {@link io.vertx.rxjava.core.http.HttpConnection#goAway} with no buffer.
   * @param errorCode 
   * @param lastStreamId 
   * @return 
   */
  public HttpConnection goAway(long errorCode, int lastStreamId) { 
    delegate.goAway(errorCode, lastStreamId);
    return this;
  }

  /**
   * Send a go away frame to the remote endpoint of the connection.
   * <p/>
   * <ul>
   *   <li>a  frame is sent to the to the remote endpoint with the <code>errorCode</code> and <code>debugData</code></li>
   *   <li>any stream created after the stream identified by <code>lastStreamId</code> will be closed</li>
   *   <li>for an  is different than <code>0</code> when all the remaining streams are closed this connection will be closed automatically</li>
   * </ul>
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param errorCode the  error code
   * @param lastStreamId the last stream id
   * @param debugData additional debug data sent to the remote endpoint
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection goAway(long errorCode, int lastStreamId, Buffer debugData) { 
    delegate.goAway(errorCode, lastStreamId, debugData.getDelegate());
    return this;
  }

  /**
   * Set an handler called when a  frame is received.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection goAwayHandler(Handler<GoAway> handler) { 
    delegate.goAwayHandler(handler);
    return this;
  }

  /**
   * Set an handler called when a  frame has been sent or received and all connections are closed.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection shutdownHandler(Handler<Void> handler) { 
    delegate.shutdownHandler(handler);
    return this;
  }

  /**
   * Initiate a connection shutdown, a go away frame is sent and the connection is closed when all current active streams
   * are closed or after a time out of 30 seconds.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection shutdown() { 
    delegate.shutdown();
    return this;
  }

  /**
   * Initiate a connection shutdown, a go away frame is sent and the connection is closed when all current streams
   * will be closed or the <code>timeout</code> is fired.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param timeoutMs the timeout in milliseconds
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection shutdown(long timeoutMs) { 
    delegate.shutdown(timeoutMs);
    return this;
  }

  /**
   * Set a close handler. The handler will get notified when the connection is closed.
   * @param handler the handler to be notified
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection closeHandler(Handler<Void> handler) { 
    delegate.closeHandler(handler);
    return this;
  }

  /**
   * Close the connection and all the currently active streams.
   * <p/>
   * An HTTP/2 connection will send a  frame before.
   */
  public void close() { 
    delegate.close();
  }

  /**
   * @return the latest server settings acknowledged by the remote endpoint - this is not implemented for HTTP/1.x
   */
  public Http2Settings settings() { 
    Http2Settings ret = delegate.settings();
    return ret;
  }

  /**
   * Send to the remote endpoint an update of the server settings.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param settings the new settings
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection updateSettings(Http2Settings settings) { 
    delegate.updateSettings(settings);
    return this;
  }

  /**
   * Send to the remote endpoint an update of this endpoint settings
   * <p/>
   * The <code>completionHandler</code> will be notified when the remote endpoint has acknowledged the settings.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param settings the new settings
   * @param completionHandler the handler notified when the settings have been acknowledged by the remote endpoint
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection updateSettings(Http2Settings settings, Handler<AsyncResult<Void>> completionHandler) { 
    delegate.updateSettings(settings, completionHandler);
    return this;
  }

  /**
   * Send to the remote endpoint an update of this endpoint settings
   * <p/>
   * The <code>completionHandler</code> will be notified when the remote endpoint has acknowledged the settings.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param settings the new settings
   * @return 
   * @deprecated use {@link #rxUpdateSettings} instead
   */
  @Deprecated()
  public Observable<Void> updateSettingsObservable(Http2Settings settings) { 
    io.vertx.rx.java.ObservableFuture<Void> completionHandler = io.vertx.rx.java.RxHelper.observableFuture();
    updateSettings(settings, completionHandler.toHandler());
    return completionHandler;
  }

  /**
   * Send to the remote endpoint an update of this endpoint settings
   * <p/>
   * The <code>completionHandler</code> will be notified when the remote endpoint has acknowledged the settings.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param settings the new settings
   * @return 
   */
  public Single<Void> rxUpdateSettings(Http2Settings settings) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      updateSettings(settings, fut);
    }));
  }

  /**
   * @return the current remote endpoint settings for this connection - this is not implemented for HTTP/1.x
   */
  public Http2Settings remoteSettings() { 
    Http2Settings ret = delegate.remoteSettings();
    return ret;
  }

  /**
   * Set an handler that is called when remote endpoint {@link io.vertx.core.http.Http2Settings} are updated.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param handler the handler for remote endpoint settings
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection remoteSettingsHandler(Handler<Http2Settings> handler) { 
    delegate.remoteSettingsHandler(handler);
    return this;
  }

  /**
   * Send a  frame to the remote endpoint.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param data the 8 bytes data of the frame
   * @param pongHandler an async result handler notified with pong reply or the failure
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection ping(Buffer data, Handler<AsyncResult<Buffer>> pongHandler) { 
    delegate.ping(data.getDelegate(), new Handler<AsyncResult<io.vertx.core.buffer.Buffer>>() {
      public void handle(AsyncResult<io.vertx.core.buffer.Buffer> ar) {
        if (ar.succeeded()) {
          pongHandler.handle(io.vertx.core.Future.succeededFuture(Buffer.newInstance(ar.result())));
        } else {
          pongHandler.handle(io.vertx.core.Future.failedFuture(ar.cause()));
        }
      }
    });
    return this;
  }

  /**
   * Send a  frame to the remote endpoint.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param data the 8 bytes data of the frame
   * @return 
   * @deprecated use {@link #rxPing} instead
   */
  @Deprecated()
  public Observable<Buffer> pingObservable(Buffer data) { 
    io.vertx.rx.java.ObservableFuture<Buffer> pongHandler = io.vertx.rx.java.RxHelper.observableFuture();
    ping(data, pongHandler.toHandler());
    return pongHandler;
  }

  /**
   * Send a  frame to the remote endpoint.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param data the 8 bytes data of the frame
   * @return 
   */
  public Single<Buffer> rxPing(Buffer data) { 
    return Single.create(new io.vertx.rx.java.SingleOnSubscribeAdapter<>(fut -> {
      ping(data, fut);
    }));
  }

  /**
   * Set an handler notified when a  frame is received from the remote endpoint.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param handler the handler to be called when a  is received
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection pingHandler(Handler<Buffer> handler) { 
    delegate.pingHandler(new Handler<io.vertx.core.buffer.Buffer>() {
      public void handle(io.vertx.core.buffer.Buffer event) {
        handler.handle(Buffer.newInstance(event));
      }
    });
    return this;
  }

  /**
   * Set an handler called when a connection error happens
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public HttpConnection exceptionHandler(Handler<Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  /**
   * @return the remote address for this connection
   */
  public SocketAddress remoteAddress() { 
    if (cached_0 != null) {
      return cached_0;
    }
    SocketAddress ret = SocketAddress.newInstance(delegate.remoteAddress());
    cached_0 = ret;
    return ret;
  }

  /**
   * @return the remote address for this connection
   */
  public SocketAddress localAddress() { 
    if (cached_1 != null) {
      return cached_1;
    }
    SocketAddress ret = SocketAddress.newInstance(delegate.localAddress());
    cached_1 = ret;
    return ret;
  }

  private SocketAddress cached_0;
  private SocketAddress cached_1;

  public static HttpConnection newInstance(io.vertx.core.http.HttpConnection arg) {
    return arg != null ? new HttpConnection(arg) : null;
  }
}
