/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.rxjava.core.parsetools;

import java.util.Map;
import rx.Observable;
import rx.Single;
import io.vertx.rxjava.core.buffer.Buffer;
import io.vertx.core.Handler;

/**
 * A helper class which allows you to easily parse protocols which are delimited by a sequence of bytes, or fixed
 * size records.
 * <p>
 * Instances of this class take as input {@link io.vertx.rxjava.core.buffer.Buffer} instances containing raw bytes,
 * and output records.
 * <p>
 * For example, if I had a simple ASCII text protocol delimited by '\n' and the input was the following:
 * <p>
 * <pre>
 * buffer1:HELLO\nHOW ARE Y
 * buffer2:OU?\nI AM
 * buffer3: DOING OK
 * buffer4:\n
 * </pre>
 * Then the output would be:<p>
 * <pre>
 * buffer1:HELLO
 * buffer2:HOW ARE YOU?
 * buffer3:I AM DOING OK
 * </pre>
 * Instances of this class can be changed between delimited mode and fixed size record mode on the fly as
 * individual records are read, this allows you to parse protocols where, for example, the first 5 records might
 * all be fixed size (of potentially different sizes), followed by some delimited records, followed by more fixed
 * size records.
 * <p>
 * Instances of this class can't currently be used for protocols where the text is encoded with something other than
 * a 1-1 byte-char mapping.
 * <p>
 * Please see the documentation for more information.
 *
 * <p/>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.parsetools.RecordParser original} non RX-ified interface using Vert.x codegen.
 */

@io.vertx.lang.rxjava.RxGen(io.vertx.core.parsetools.RecordParser.class)
public class RecordParser implements io.vertx.core.Handler<Buffer> {

  public static final io.vertx.lang.rxjava.TypeArg<RecordParser> __TYPE_ARG = new io.vertx.lang.rxjava.TypeArg<>(
    obj -> new RecordParser((io.vertx.core.parsetools.RecordParser) obj),
    RecordParser::getDelegate
  );

  private final io.vertx.core.parsetools.RecordParser delegate;
  
  public RecordParser(io.vertx.core.parsetools.RecordParser delegate) {
    this.delegate = delegate;
  }

  public io.vertx.core.parsetools.RecordParser getDelegate() {
    return delegate;
  }

  public void setOutput(Handler<Buffer> output) { 
    delegate.setOutput(new Handler<io.vertx.core.buffer.Buffer>() {
      public void handle(io.vertx.core.buffer.Buffer event) {
        output.handle(Buffer.newInstance(event));
      }
    });
  }

  /**
   * Create a new <code>RecordParser</code> instance, initially in delimited mode, and where the delimiter can be represented
   * by the String <code></code> delim endcoded in latin-1 . Don't use this if your String contains other than latin-1 characters.
   * <p>
   * <code>output</code> Will receive whole records which have been parsed.
   * @param delim the initial delimiter string
   * @param output handler that will receive the output
   * @return 
   */
  public static RecordParser newDelimited(String delim, Handler<Buffer> output) { 
    RecordParser ret = RecordParser.newInstance(io.vertx.core.parsetools.RecordParser.newDelimited(delim, new Handler<io.vertx.core.buffer.Buffer>() {
      public void handle(io.vertx.core.buffer.Buffer event) {
        output.handle(Buffer.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create a new <code>RecordParser</code> instance, initially in delimited mode, and where the delimiter can be represented
   * by the <code>Buffer</code> delim.
   * <p>
   * <code>output</code> Will receive whole records which have been parsed.
   * @param delim the initial delimiter buffer
   * @param output handler that will receive the output
   * @return 
   */
  public static RecordParser newDelimited(Buffer delim, Handler<Buffer> output) { 
    RecordParser ret = RecordParser.newInstance(io.vertx.core.parsetools.RecordParser.newDelimited(delim.getDelegate(), new Handler<io.vertx.core.buffer.Buffer>() {
      public void handle(io.vertx.core.buffer.Buffer event) {
        output.handle(Buffer.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Create a new <code>RecordParser</code> instance, initially in fixed size mode, and where the record size is specified
   * by the <code>size</code> parameter.
   * <p>
   * <code>output</code> Will receive whole records which have been parsed.
   * @param size the initial record size
   * @param output handler that will receive the output
   * @return 
   */
  public static RecordParser newFixed(int size, Handler<Buffer> output) { 
    RecordParser ret = RecordParser.newInstance(io.vertx.core.parsetools.RecordParser.newFixed(size, new Handler<io.vertx.core.buffer.Buffer>() {
      public void handle(io.vertx.core.buffer.Buffer event) {
        output.handle(Buffer.newInstance(event));
      }
    }));
    return ret;
  }

  /**
   * Flip the parser into delimited mode, and where the delimiter can be represented
   * by the String <code>delim</code> encoded in latin-1 . Don't use this if your String contains other than latin-1 characters.
   * <p>
   * This method can be called multiple times with different values of delim while data is being parsed.
   * @param delim the new delimeter
   */
  public void delimitedMode(String delim) { 
    delegate.delimitedMode(delim);
  }

  /**
   * Flip the parser into delimited mode, and where the delimiter can be represented
   * by the delimiter <code>delim</code>.
   * <p>
   * This method can be called multiple times with different values of delim while data is being parsed.
   * @param delim the new delimiter
   */
  public void delimitedMode(Buffer delim) { 
    delegate.delimitedMode(delim.getDelegate());
  }

  /**
   * Flip the parser into fixed size mode, where the record size is specified by <code>size</code> in bytes.
   * <p>
   * This method can be called multiple times with different values of size while data is being parsed.
   * @param size the new record size
   */
  public void fixedSizeMode(int size) { 
    delegate.fixedSizeMode(size);
  }

  /**
   * This method is called to provide the parser with data.
   * @param buffer a chunk of data
   */
  public void handle(Buffer buffer) { 
    delegate.handle(buffer.getDelegate());
  }


  public static  RecordParser newInstance(io.vertx.core.parsetools.RecordParser arg) {
    return arg != null ? new RecordParser(arg) : null;
  }
}
