/**********************************************************************
Copyright (c) 2005 Roland Szabo and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.jpox.cache;

import java.io.IOException;
import java.io.Serializable;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;

import javax.jdo.JDOException;
import javax.jdo.identity.SingleFieldIdentity;

import org.jpox.store.OID;
import org.jpox.util.JPOXLogger;
import org.jpox.util.Localiser;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;
import net.sf.ehcache.ObjectExistsException;

/**
 * Plugin for ehcache that allows the user to use different cahces for different classes.
 * Based on the EhcacheLevel2Cache class.
 * @version $Revision: 1.5 $
 */
public class EhcacheClassBasedLevel2Cache implements Level2Cache
{
    /** Localisation of messages. */
    private static final Localiser LOCALISER_EHCACHE = Localiser.getInstance("org.jpox.cache.Localisation_ehcache",EhcacheClassBasedLevel2Cache.class.getClassLoader());

    private final CacheManager cacheManager;

    private final Cache defaultCache;

    private final HashMap caches = new HashMap();

    private Cache getCacheForClass(String pcClassName)
    {
        Cache cache = (Cache) caches.get(pcClassName);
        if (cache == null)
        {
            if (JPOXLogger.CACHE.isDebugEnabled())
            {
                JPOXLogger.CACHE.debug(LOCALISER_EHCACHE.msg("Cache.EHCache.Initialising", pcClassName));
            }
            if (cacheManager.cacheExists(pcClassName))
            {
                if (JPOXLogger.CACHE.isDebugEnabled())
                {
                    JPOXLogger.CACHE.debug(LOCALISER_EHCACHE.msg("Cache.EHCache.Exists", pcClassName));
                }
                cache = cacheManager.getCache(pcClassName);
            }
            else
            {
                // if defaultCache is null, there should be a warning here for the user,
                // that he either needs to define the cache for the class, or give a default                
                if (JPOXLogger.CACHE.isDebugEnabled())
                {
                    JPOXLogger.CACHE.debug(LOCALISER_EHCACHE.msg("Cache.EHCache.CacheDoesntExist"));
                }
                if (defaultCache == null)
                {
                    JPOXLogger.CACHE.error(LOCALISER_EHCACHE.msg("Cache.EHCache.CacheDoestExistNoDefault", pcClassName));
                }
                cache = defaultCache;
            }
            caches.put(pcClassName, cache);
        }
        return cache;
    }

    private Cache getCacheForId(Object id)
    {
        if (id instanceof SingleFieldIdentity)
        {
            return getCacheForClass(((SingleFieldIdentity) id).getTargetClassName());
        }
        if (id instanceof OID)
        {
            return getCacheForClass(((OID) id).getPcClass());
        }
        return defaultCache;
    }

    /**
     * Constructor.
     * @param props Any properties to control the cache
     */
    public EhcacheClassBasedLevel2Cache(Properties props)
    {
        try
        {
            cacheManager = CacheManager.create(CacheManager.class.getResource(props.getProperty("ConfigurationFile")));
        }
        catch (CacheException e)
        {
            throw new JDOException(LOCALISER_EHCACHE.msg("Cache.EHCache.CacheManagerInitialiseFails", e));
        }
        String cacheName = props.getProperty("CacheName");
        if (cacheName != null && cacheName.length() > 0)
        {
            if (!cacheManager.cacheExists(cacheName))
            {
                try
                {
                    cacheManager.addCache(cacheName);
                }
                catch (IllegalStateException e)
                {
                    throw new JDOException(LOCALISER_EHCACHE.msg("Cache.EHCache.CreateDefaultFails", e));
                }
                catch (ObjectExistsException e)
                {
                    throw new JDOException(LOCALISER_EHCACHE.msg("Cache.EHCache.CreateDefaultFails", e));
                }
                catch (CacheException e)
                {
                    throw new JDOException(LOCALISER_EHCACHE.msg("Cache.EHCache.CreateDefaultFails", e));
                }
            }
            defaultCache = cacheManager.getCache(cacheName);
        }
        else
        {
            defaultCache = null;
        }
    }

    /**
     * Method to clear the cache
     * @see org.jpox.cache.Level2Cache#clear()
     */
    public void clear()
    {
        try
        {
            for (Iterator i = caches.values().iterator(); i.hasNext();)
            {
                ((Cache) i.next()).removeAll();
            }
            defaultCache.removeAll();
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Accessor for whether the cache contains the specified id.
     * @see org.jpox.cache.Level2Cache#containsOid(java.lang.Object)
     */
    public boolean containsOid(Object oid)
    {
        try
        {
            return getCacheForId(oid).getKeys().contains(oid);
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * Accessor for an object in the cache.
     * @see org.jpox.cache.Level2Cache#get(java.lang.Object)
     */
    public CachedPC get(Object oid)
    {
        try
        {
            Element element = getCacheForId(oid).get((Serializable) oid);
            if (element == null)
            {
                return null;
            }
            return toPC(element);
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Convert from Element to a cacheable object
     * @param object the Element
     * @return the cacheable object
     */
    private CachedPC toPC(Element object)
    {
        return (CachedPC) ((CacheElement) object.getValue()).getValue();
    }

    /**
     * Convert from PersistenceCapable to Element
     * @param oid the id
     * @param object the PersistenceCapable
     * @return the Element
     */
    private Element toElement(Object oid, CachedPC object)
    {
        return new Element((Serializable) oid, new CacheElement(object));
    }

    /**
     * Accessor for the number of pinned objects - not supported by Ehcache.
     * @see org.jpox.cache.Level2Cache#getNumberOfPinnedObjects()
     */
    public int getNumberOfPinnedObjects()
    {
        throw new UnsupportedOperationException("getNumberOfPinnedObjects() method not yet supported by EHCache plugin");
    }

    /**
     * Accessor for the number of unpinned objects - not supported by Ehcache.
     * @see org.jpox.cache.Level2Cache#getNumberOfUnpinnedObjects()
     */
    public int getNumberOfUnpinnedObjects()
    {
        throw new UnsupportedOperationException("getNumberOfUnpinnedObjects() method not yet supported by EHCache plugin");
    }

    /**
     * Accessor for the size of the cache.
     * @see org.jpox.cache.Level2Cache#getSize()
     */
    public int getSize()
    {
        try
        {
            int size = defaultCache.getSize();
            for (Iterator i = caches.values().iterator(); i.hasNext();)
            {
                size += ((Cache) i.next()).getSize();
            }
            return size;
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * Accessor for whether the cache is empty
     * @see org.jpox.cache.Level2Cache#isEmpty()
     */
    public boolean isEmpty()
    {
        return getSize() == 0;
    }

    /**
     * Method to add an object to the cache under its id
     * @param oid The identity
     * @param pc The cacheable object
     */
    public CachedPC put(Object oid, CachedPC pc)
    {
        if (oid == null || pc == null)
        {
            return null;
        }

        // Check if the object is disconnected from its StateManager/PM
        if (pc.getPersistenceCapable().jdoGetPersistenceManager() != null)
        {
            return null;
        }

        getCacheForId(oid).put(toElement(oid, pc));

        return pc;
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#evict(java.lang.Object)
     */
    public void evict(Object oid)
    {
        Object pc = get(oid);
        if (pc != null)
        {
            getCacheForId(oid).remove(toElement(oid, (CachedPC) pc));
        }
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#evictAll()
     */
    public void evictAll()
    {
        clear();
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#evictAll(java.lang.Class, boolean)
     */
    public void evictAll(Class pcClass, boolean subclasses)
    {
        throw new UnsupportedOperationException("evict(Class, boolean) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#evictAll(java.util.Collection)
     */
    public void evictAll(Collection oids)
    {
        if (oids == null)
        {
            return;
        }

        Iterator iter = oids.iterator();
        while (iter.hasNext())
        {
            evict(iter.next());
        }
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#evictAll(java.lang.Object[])
     */
    public void evictAll(Object[] oids)
    {
        if (oids == null)
        {
            return;
        }

        for (int i = 0; i < oids.length; i++)
        {
            evict(oids[i]);
        }
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#pin(java.lang.Object)
     */
    public void pin(Object oid)
    {
        throw new UnsupportedOperationException("pinAll(Object) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#pinAll(java.lang.Class, boolean)
     */
    public void pinAll(Class cls, boolean subs)
    {
        throw new UnsupportedOperationException("pinAll(Class, boolean) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#pinAll(java.util.Collection)
     */
    public void pinAll(Collection oids)
    {
        throw new UnsupportedOperationException("pinAll(Collection) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#pinAll(java.lang.Object[])
     */
    public void pinAll(Object[] oids)
    {
        throw new UnsupportedOperationException("pinAll(Object[]) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#unpin(java.lang.Object)
     */
    public void unpin(Object oid)
    {
        throw new UnsupportedOperationException("unpin(Object) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#unpinAll(java.lang.Class, boolean)
     */
    public void unpinAll(Class cls, boolean subs)
    {
        throw new UnsupportedOperationException("unpinAll(Class, boolean) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#unpinAll(java.util.Collection)
     */
    public void unpinAll(Collection oids)
    {
        throw new UnsupportedOperationException("unpinAll(Collection) method not yet supported by EHCache plugin");
    }

    /**
     * @see javax.jdo.datastore.DataStoreCache#unpinAll(java.lang.Object[])
     */
    public void unpinAll(Object[] oids)
    {
        throw new UnsupportedOperationException("unpinAll(Object[]) method not yet supported by EHCache plugin");
    }

    /**
     * Workaround class for the required serializable
     */
    private class CacheElement implements Serializable
    {
        private final Object value;

        public CacheElement(Object value)
        {
            this.value = value;
        }

        public Object getValue()
        {
            return value;
        }

        public boolean equals(Object arg0)
        {
            if (arg0 == this)
            {
                return true;
            }
            if (!(arg0 instanceof CacheElement))
            {
                return false;
            }
            return ((CacheElement) arg0).value.equals(this.value);
        }

        public int hashCode()
        {
            return this.value.hashCode();
        }
    }
}