/**********************************************************************
Copyright (c) 2005 Erik Bengtson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.jpox.cache;

import java.io.IOException;
import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.Properties;

import javax.jdo.JDOException;

import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheException;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;
import net.sf.ehcache.ObjectExistsException;

/**
 * Simple implementation of a plugin for use of Ehcache
 * caching product with JPOX. Please refer to
 * <a href="http://ehcache.sourceforge.net">ehcache.sourceforge.net</a>
 * for full details of their product. 
 * 
 * @version $Revision: 1.6 $
 */
public class EhcacheLevel2Cache implements Level2Cache
{
    /** The cache manager */
    private final CacheManager cacheManager;
    
    /** The cache */
    private final Cache cache;
    

    /**
     * Constructor.
     * @param props Any properties to control the cache
     */
    public EhcacheLevel2Cache(Properties props)
    {
        try
        {
            cacheManager = CacheManager.create(CacheManager.class.getResource(props.getProperty("ConfigurationFile")));
        }
        catch (CacheException e)
        {
            throw new JDOException("Cant create cache");
        }
        if( !cacheManager.cacheExists(props.getProperty("CacheName")) )
        {
            try
            {
                cacheManager.addCache(props.getProperty("CacheName"));
            }
            catch (IllegalStateException e)
            {
                e.printStackTrace();
                throw new JDOException("Cant create cache");
            }
            catch (ObjectExistsException e)
            {
                e.printStackTrace();
                throw new JDOException("Cant create cache");
            }
            catch (CacheException e)
            {
                e.printStackTrace();
                throw new JDOException("Cant create cache");
            }
        }
        cache = cacheManager.getCache(props.getProperty("CacheName"));
    }

    /**
     * Method to clear the cache
     * @see org.jpox.cache.Level2Cache#clear()
     */
    public void clear()
    {
        try
        {
            cache.removeAll();
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (IOException e)
        {
            e.printStackTrace();
        }
    }

    /**
     * Accessor for whether the cache contains the specified id.
     * @see org.jpox.cache.Level2Cache#containsOid(java.lang.Object)
     */
    public boolean containsOid(Object oid)
    {
        try
        {
            return cache.getKeys().contains(oid);
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return false;
    }

    /**
     * Accessor for an object in the cache.
     * @see org.jpox.cache.Level2Cache#get(java.lang.Object)
     */
    public CachedPC get(Object oid)
    {
        try
        {
            Element element = cache.get((Serializable) oid);
            if (element == null)
            {
                return null;
            }
            return toPC(element);
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return null;
    }
    
    /**
     * Convert from Element to a cacheable object
     * @param object the Element
     * @return the cacheable object
     */
    private CachedPC toPC(Element object)
    {
        return (CachedPC) ((CacheElement)object.getValue()).getValue();
    }

    /**
     * Convert from PersistenceCapable to Element
     * @param oid the id
     * @param object the PersistenceCapable
     * @return the Element
     */
    private Element toElement(Object oid, CachedPC object)
    {
        return new Element( (Serializable) oid, new CacheElement(object));
    }    

    /**
     * Accessor for the number of pinned objects - not supported by Ehcache.
     * @see org.jpox.cache.Level2Cache#getNumberOfPinnedObjects()
     */
    public int getNumberOfPinnedObjects()
    {
        throw new UnsupportedOperationException("getNumberOfPinnedObjects() method not yet supported by Ehcache plugin");
    }

    /**
     * Accessor for the number of unpinned objects - not supported by Ehcache.
     * @see org.jpox.cache.Level2Cache#getNumberOfUnpinnedObjects()
     */
    public int getNumberOfUnpinnedObjects()
    {
        throw new UnsupportedOperationException("getNumberOfUnpinnedObjects() method not yet supported by Ehcache plugin");
    }

    /**
     * Accessor for the size of the cache.
     * @see org.jpox.cache.Level2Cache#getSize()
     */
    public int getSize()
    {
        try
        {
            return cache.getSize();
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return 0;
    }

    /**
     * Accessor for whether the cache is empty
     * @see org.jpox.cache.Level2Cache#isEmpty()
     */
    public boolean isEmpty()
    {
        try
        {
            return cache.getSize() == 0;
        }
        catch (IllegalStateException e)
        {
            e.printStackTrace();
        }
        catch (CacheException e)
        {
            e.printStackTrace();
        }
        return true;
    }

    /**
     * Method to add an object to the cache under its id
     * @param oid The identity
     * @param pc The cacheable object
     */
    public CachedPC put(Object oid, CachedPC pc)
    {
        if (oid == null || pc == null)
        {
            return null;
        }

        // Check if the object is disconnected from its StateManager/PM
        if (pc.getPersistenceCapable().jdoGetPersistenceManager() != null)
        {
            return null;
        }

        cache.put(toElement(oid, pc));
        
        return pc;
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#evict(java.lang.Object)
     */
    public void evict(Object oid)
    {
        Object pc = get(oid);
        if (pc != null)
        {
            cache.remove(toElement(oid, (CachedPC)pc));
        }
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#evictAll()
     */
    public void evictAll()
    {
        clear();
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#evictAll(java.lang.Class, boolean)
     */
    public void evictAll(Class pcClass, boolean subclasses)
    {
        throw new UnsupportedOperationException("evict(Class, boolean) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#evictAll(java.util.Collection)
     */
    public void evictAll(Collection oids)
    {
        if (oids == null)
        {
            return;
        }

        Iterator iter = oids.iterator();
        while (iter.hasNext())
        {
            evict(iter.next());
        }
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#evictAll(java.lang.Object[])
     */
    public void evictAll(Object[] oids)
    {
        if (oids == null)
        {
            return;
        }

        for (int i=0;i<oids.length;i++)
        {
            evict(oids[i]);
        }
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#pin(java.lang.Object)
     */
    public void pin(Object oid)
    {
        throw new UnsupportedOperationException("pinAll(Object) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#pinAll(java.lang.Class, boolean)
     */
    public void pinAll(Class cls, boolean subs)
    {
        throw new UnsupportedOperationException("pinAll(Class, boolean) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#pinAll(java.util.Collection)
     */
    public void pinAll(Collection oids)
    {
        throw new UnsupportedOperationException("pinAll(Collection) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#pinAll(java.lang.Object[])
     */
    public void pinAll(Object[] oids)
    {
        throw new UnsupportedOperationException("pinAll(Object[]) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#unpin(java.lang.Object)
     */
    public void unpin(Object oid)
    {
        throw new UnsupportedOperationException("unpin(Object) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#unpinAll(java.lang.Class, boolean)
     */
    public void unpinAll(Class cls, boolean subs)
    {
        throw new UnsupportedOperationException("unpinAll(Class, boolean) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#unpinAll(java.util.Collection)
     */
    public void unpinAll(Collection oids)
    {
        throw new UnsupportedOperationException("unpinAll(Collection) method not yet supported by Ehcache plugin");
    }

    /**
     * 
     * @see javax.jdo.datastore.DataStoreCache#unpinAll(java.lang.Object[])
     */
    public void unpinAll(Object[] oids)
    {
        throw new UnsupportedOperationException("unpinAll(Object[]) method not yet supported by Ehcache plugin");
    }
    
    /**
     * Workaround class for the required serializable
     * @version $Revision: 1.6 $
     */
    private class CacheElement implements Serializable
    {
        private final Object value;
        
        public CacheElement(Object value)
        {
            this.value = value;
        }
        
        public Object getValue()
        {
            return value;
        }
        
        public boolean equals(Object arg0)
        {
            if (arg0 == this)
            {
                return true;
            }
            if (!(arg0 instanceof CacheElement))
            {
                return false;
            }
            return ((CacheElement)arg0).value.equals(this.value);
        }

        public int hashCode()
        {
            return this.value.hashCode();
        }
    }
}