/*
 * Decompiled with CFR 0.152.
 */
package org.cp.elements.lang;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import org.cp.elements.lang.Assert;
import org.cp.elements.lang.LangExtensions;
import org.cp.elements.lang.ObjectUtils;
import org.cp.elements.lang.StringUtils;
import org.cp.elements.util.ComparatorResultBuilder;

public class Version
implements Comparable<Version> {
    protected static final String RELEASE_DATE_TIME_FORMAT = "yyyy-MMMM-dd-HH-mm-ss";
    private int buildNumber = 0;
    private int qualifierNumber = 0;
    private final int major;
    private final int minor;
    private final int maintenance;
    private LocalDateTime releaseDateTime;
    private Qualifier qualifier = Qualifier.UNDEFINED;

    public static Version from(int major, int minor) {
        return Version.from(major, minor, 0, Qualifier.UNDEFINED, 0);
    }

    public static Version from(int major, int minor, int maintenance) {
        return Version.from(major, minor, maintenance, Qualifier.UNDEFINED, 0);
    }

    public static Version from(int major, int minor, int maintenance, Qualifier qualifier) {
        return Version.from(major, minor, maintenance, qualifier, 0);
    }

    public static Version from(int major, int minor, int maintenance, Qualifier qualifier, int qualifierNumber) {
        return new Version(major, minor, maintenance, qualifier, qualifierNumber);
    }

    public static Version parse(String version) {
        Assert.hasText(version, "The version [%s] must be specified", version);
        String[] versionNumbers = version.split("\\.");
        Assert.isTrue(versionNumbers.length > 1, "Version [%s] must minimally consist of major and minor version numbers", version);
        switch (versionNumbers.length) {
            case 2: {
                return Version.parseMajorMinor(versionNumbers);
            }
            case 3: {
                return Version.parseMajorMinorMaintenance(versionNumbers);
            }
            case 4: {
                return Version.parseMajorMinorMaintenanceQualifier(versionNumbers);
            }
        }
        throw new IllegalArgumentException(String.format("Unrecognized format for version [%s]", version));
    }

    private static int parseInt(String value) {
        return Integer.parseInt(StringUtils.getDigits(value));
    }

    private static Version parseMajorMinor(String[] versionNumbers) {
        return Version.from(Version.parseInt(versionNumbers[0]), Version.parseInt(versionNumbers[1]));
    }

    private static Version parseMajorMinorMaintenance(String[] versionNumbers) {
        return Version.from(Version.parseInt(versionNumbers[0]), Version.parseInt(versionNumbers[1]), Version.parseInt(versionNumbers[2]));
    }

    private static Version parseMajorMinorMaintenanceQualifier(String[] versionNumbers) {
        return Version.from(Version.parseInt(versionNumbers[0]), Version.parseInt(versionNumbers[1]), Version.parseInt(versionNumbers[2]), Version.parseQualifier(versionNumbers[3]), Version.parseQualifierNumber(versionNumbers[3]));
    }

    private static Qualifier parseQualifier(String qualifier) {
        return Qualifier.resolve(qualifier);
    }

    private static int parseQualifierNumber(String qualifier) {
        try {
            return Version.parseInt(qualifier);
        }
        catch (NumberFormatException ignore) {
            return 0;
        }
    }

    public Version(int major, int minor) {
        this(major, minor, 0, Qualifier.UNDEFINED, 0);
    }

    public Version(int major, int minor, int maintenance) {
        this(major, minor, maintenance, Qualifier.UNDEFINED, 0);
    }

    public Version(int major, int minor, int maintenance, Qualifier qualifier) {
        this(major, minor, maintenance, qualifier, 0);
    }

    public Version(int major, int minor, int maintenance, Qualifier qualifier, int qualifierNumber) {
        LangExtensions.assertThat(major).throwing(new IllegalArgumentException(String.format("Major version [%d] must be greater than equal to 0", major))).isGreaterThanEqualTo(0);
        LangExtensions.assertThat(minor).throwing(new IllegalArgumentException(String.format("Minor version [%d] must be greater than equal to 0", minor))).isGreaterThanEqualTo(0);
        LangExtensions.assertThat(maintenance).throwing(new IllegalArgumentException(String.format("Maintenance version [%d] must be greater than equal to 0", maintenance))).isGreaterThanEqualTo(0);
        this.major = major;
        this.minor = minor;
        this.maintenance = maintenance;
        this.qualifier = ObjectUtils.defaultIfNull(qualifier, Qualifier.UNDEFINED);
        this.qualifierNumber = Math.max(qualifierNumber, 0);
    }

    public boolean isMilestone() {
        return this.getQualifier().isMilestone();
    }

    public boolean isReleaseCandidate() {
        return this.getQualifier().isReleaseCandidate();
    }

    public boolean isRelease() {
        return this.getQualifier().isRelease();
    }

    public boolean isSnapshot() {
        return this.getQualifier().isSnapshot();
    }

    public boolean isUndefined() {
        return this.getQualifier().isUndefined();
    }

    public int getBuildNumber() {
        return Math.max(this.buildNumber, 0);
    }

    public int getMajor() {
        return this.major;
    }

    public int getMinor() {
        return this.minor;
    }

    public int getMaintenance() {
        return this.maintenance;
    }

    public Qualifier getQualifier() {
        return ObjectUtils.defaultIfNull(this.qualifier, Qualifier.UNDEFINED);
    }

    public int getQualifierNumber() {
        return Math.max(this.qualifierNumber, 0);
    }

    public LocalDateTime getReleaseDateTime() {
        return this.releaseDateTime;
    }

    @Override
    public int compareTo(Version version) {
        return ComparatorResultBuilder.create().doCompare(this.getMajor(), version.getMajor()).doCompare(this.getMinor(), version.getMinor()).doCompare(this.getMaintenance(), version.getMaintenance()).doCompare(this.getQualifier().ordinal(), version.getQualifier().ordinal()).doCompare(this.getQualifierNumber(), version.getQualifierNumber()).getResult();
    }

    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof Version)) {
            return false;
        }
        Version that = (Version)obj;
        return ObjectUtils.equals(this.getBuildNumber(), that.getBuildNumber()) && ObjectUtils.equals(this.getMajor(), that.getMajor()) && ObjectUtils.equals(this.getMinor(), that.getMinor()) && ObjectUtils.equals(this.getMaintenance(), that.getMaintenance()) && ObjectUtils.equals((Object)this.getQualifier(), (Object)that.getQualifier()) && ObjectUtils.equals(this.getQualifierNumber(), that.getQualifierNumber());
    }

    public int hashCode() {
        int hashValue = 17;
        hashValue = 37 * hashValue + ObjectUtils.hashCode(this.getBuildNumber());
        hashValue = 37 * hashValue + ObjectUtils.hashCode(this.getMajor());
        hashValue = 37 * hashValue + ObjectUtils.hashCode(this.getMinor());
        hashValue = 37 * hashValue + ObjectUtils.hashCode(this.getMaintenance());
        hashValue = 37 * hashValue + ObjectUtils.hashCode((Object)this.getQualifier());
        hashValue = 37 * hashValue + ObjectUtils.hashCode(this.getQualifierNumber());
        return hashValue;
    }

    public String toString() {
        int buildNumber = this.getBuildNumber();
        Qualifier qualifier = this.getQualifier();
        return String.format("%1$d.%2$d.%3$d%4$s%5$s%6$s", this.getMajor(), this.getMinor(), this.getMaintenance(), this.toQualifierString(), this.toBuildNumberString(), this.toReleaseDateTimeString());
    }

    private String toBuildNumberString() {
        int buildNumber = this.getBuildNumber();
        return buildNumber > 0 ? String.format(" build %d", buildNumber) : "";
    }

    private String toQualifierString() {
        Qualifier qualifier = this.getQualifier();
        int qualifierNumber = this.getQualifierNumber();
        return qualifier.isUndefined() ? "" : (qualifierNumber > 0 ? String.format(".%s%d", qualifier.getAbbreviation(), qualifierNumber) : String.format(".%s", qualifier.name()));
    }

    private String toReleaseDateTimeString() {
        LocalDateTime releaseDateTime = this.getReleaseDateTime();
        return releaseDateTime == null ? "" : String.format(" on %s", releaseDateTime.format(DateTimeFormatter.ofPattern(RELEASE_DATE_TIME_FORMAT)));
    }

    public Version on(LocalDateTime releaseDateTime) {
        this.releaseDateTime = releaseDateTime;
        return this;
    }

    public Version with(int buildNumber) {
        this.buildNumber = Math.max(buildNumber, 0);
        return this;
    }

    public Version with(Qualifier qualifier) {
        return this.with(qualifier, 0);
    }

    public Version with(Qualifier qualifier, int qualifierNumber) {
        this.qualifier = ObjectUtils.defaultIfNull(qualifier, Qualifier.UNDEFINED);
        this.qualifierNumber = Math.max(qualifierNumber, 0);
        return this;
    }

    public static enum Qualifier {
        ALPHA("ALPHA", "Alpha"),
        BETA("BETA", "Beta"),
        BUILD_SNAPSHOT("BUILD-SNAPSHOT", "Build Snapshot"),
        ITERATION("IT", "Iteration"),
        MILESTONE("M", "Milestone"),
        RELEASE_CANDIDATE("RC", "Release Candidate"),
        RELEASE("RELEASE", "Release"),
        SNAPSHOT("SNAPSHOT", "Snapshot"),
        UNDEFINED("UNDEFINED", "Undefined");

        private final String abbreviation;
        private final String description;

        public static Qualifier resolve(String version) {
            if (StringUtils.hasText(version)) {
                String trimmedLowerCaseVersion = version.trim().toLowerCase();
                String versionQualifier = version.substring(version.lastIndexOf(".") + 1);
                for (Qualifier qualifier : Qualifier.values()) {
                    String qualifierName = qualifier.name().toLowerCase();
                    String alternateQualifierName = qualifierName.replace("_", "-");
                    if (!trimmedLowerCaseVersion.endsWith(qualifierName) && !trimmedLowerCaseVersion.endsWith(alternateQualifierName) && !versionQualifier.startsWith(qualifier.getAbbreviation())) continue;
                    return qualifier;
                }
            }
            return UNDEFINED;
        }

        private Qualifier(String abbreviation, String description) {
            this.abbreviation = abbreviation;
            this.description = description;
        }

        public boolean isAlpha() {
            return this == ALPHA;
        }

        public boolean isBeta() {
            return this == BETA;
        }

        public boolean isBuildSnapshot() {
            return this == BUILD_SNAPSHOT;
        }

        public boolean isIteration() {
            return this == ITERATION;
        }

        public boolean isMilestone() {
            return this == MILESTONE;
        }

        public boolean isReleaseCandidate() {
            return this == RELEASE_CANDIDATE;
        }

        public boolean isRelease() {
            return this == RELEASE;
        }

        public boolean isSnapshot() {
            return this == SNAPSHOT;
        }

        public boolean isUndefined() {
            return this == UNDEFINED;
        }

        public boolean isNotUndefined() {
            return !this.isUndefined();
        }

        public String getAbbreviation() {
            return this.abbreviation;
        }

        public String getDescription() {
            return this.description;
        }

        public String toString() {
            return this.getDescription();
        }
    }
}

