/**********************************************************************
Copyright (c) 2002 Mike Martin (TJDO) and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
 

Contributors:
2003 Andy Jefferson - coding standards
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.adapter;

import java.math.BigInteger;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Types;
import java.util.ArrayList;
import java.util.List;

import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.IdentifierType;
import org.datanucleus.store.mapped.expression.BooleanExpression;
import org.datanucleus.store.mapped.expression.NumericExpression;
import org.datanucleus.store.mapped.expression.ScalarExpression;
import org.datanucleus.store.mapped.expression.StringExpression;
import org.datanucleus.store.mapped.expression.StringLiteral;
import org.datanucleus.store.mapped.expression.ScalarExpression.IllegalArgumentTypeException;
import org.datanucleus.store.rdbms.schema.RDBMSColumnInfo;
import org.datanucleus.store.rdbms.schema.SQLTypeInfo;
import org.datanucleus.store.rdbms.table.Table;
import org.datanucleus.store.schema.StoreSchemaHandler;

/**
 * Provides methods for adapting SQL language elements to the DB2 database.
 * @see DatabaseAdapter
 */
public class DB2Adapter extends DatabaseAdapter
{
    /**
     * A string containing the list of DB2 keywords 
     * This list is normally obtained dynamically from the driver using
     * DatabaseMetaData.getSQLKeywords()
     * 
     * Based on database DB2 version 7
     */
    public static final String DB2_RESERVED_WORDS =
        "ACCESS,ALIAS,ALLOW,ASUTIME,AUDIT,AUX,AUXILIARY,BUFFERPOOL," +
        "CAPTURE,CCSID,CLUSTER,COLLECTION,COLLID,COMMENT,CONCAT," +
        "CONTAINS,COUNT_BIG,CURRENT_LC_PATH,CURRENT_SERVER," +
        "CURRENT_TIMEZONE,DATABASE,DAYS,DB2GENERAL,DB2SQL,DBA," +
        "DBINFO,DBSPACE,DISALLOW,DSSIZE,EDITPROC,ERASE,EXCLUSIVE," +
        "EXPLAIN,FENCED,FIELDPROC,FILE,FINAL,GENERATED,GRAPHIC,HOURS," +
        "IDENTIFIED,INDEX,INTEGRITY,ISOBID,JAVA,LABEL,LC_CTYPE,LINKTYPE," +
        "LOCALE,LOCATORS,LOCK,LOCKSIZE,LONG,MICROSECOND,MICROSECONDS," +
        "MINUTES,MODE,MONTHS,NAME,NAMED,NHEADER,NODENAME,NODENUMBER," +
        "NULLS,NUMPARTS,OBID,OPTIMIZATION,OPTIMIZE,PACKAGE,PAGE," +
        "PAGES,PART,PCTFREE,PCTINDEX,PIECESIZE,PLAN,PRIQTY,PRIVATE," +
        "PROGRAM,PSID,QYERYNO,RECOVERY,RENAME,RESET,RESOURCE,RRN,RUN," +
        "SCHEDULE,SCRATCHPAD,SECONDS,SECQTY,SECURITY,SHARE,SIMPLE," +
        "SOURCE,STANDARD,STATISTICS,STAY,STOGROUP,STORES,STORPOOL," +
        "STYLE,SUBPAGES,SYNONYM,TABLESPACE,TYPE,VALIDPROC,VARIABLE," +
        "VARIANT,VCAT,VOLUMES,WLM,YEARS";
    
    /**
     * Constructs a DB2 adapter based on the given JDBC metadata.
     * @param metadata the database metadata.
     */
    public DB2Adapter(DatabaseMetaData metadata)
    {
        super(metadata);

        reservedKeywords.addAll(parseKeywordList(DB2_RESERVED_WORDS));

        // Update supported options
        supportedOptions.add(LOCK_WITH_SELECT_FOR_UPDATE);
        supportedOptions.add(IDENTITY_COLUMNS);
        supportedOptions.add(SEQUENCES);
        supportedOptions.add(ANALYSIS_METHODS);
        supportedOptions.remove(BOOLEAN_COMPARISON);
        supportedOptions.remove(DEFERRED_CONSTRAINTS);
        supportedOptions.remove(NULLS_IN_CANDIDATE_KEYS);
        supportedOptions.remove(NULLS_KEYWORD_IN_COLUMN_OPTIONS);

        supportedOptions.remove(FK_DELETE_ACTION_DEFAULT);
        supportedOptions.remove(FK_UPDATE_ACTION_DEFAULT);
        supportedOptions.remove(FK_UPDATE_ACTION_CASCADE);
        supportedOptions.remove(FK_UPDATE_ACTION_NULL);
    }

    /**
     * Initialise the types for this datastore.
     * @param handler SchemaHandler that we initialise the types for
     * @param mconn Managed connection to use
     */
    public void initialiseTypes(StoreSchemaHandler handler, ManagedConnection mconn)
    {
        super.initialiseTypes(handler, mconn);

        // Add on any missing JDBC types
        SQLTypeInfo sqlType = new org.datanucleus.store.rdbms.schema.DB2TypeInfo(
            "FLOAT", (short)Types.FLOAT, 53, null, null, null, 1, false, (short)2,
            false, false, false, null, (short)0, (short)0, 0);
        addSQLTypeForJDBCType(handler, mconn, (short)Types.FLOAT, sqlType, true);

        sqlType = new org.datanucleus.store.rdbms.schema.DB2TypeInfo(
            "NUMERIC", (short)Types.NUMERIC, 31, null, null, "PRECISION,SCALE", 1,
             false, (short)2, false, false, false, null, (short)0, (short)31, 0);
        addSQLTypeForJDBCType(handler, mconn, (short)Types.NUMERIC, sqlType, true);

        sqlType = new org.datanucleus.store.rdbms.schema.DB2TypeInfo(
            "BIGINT", (short)Types.BIGINT, 20, null, null, null, 1, false, (short)2,
            false, true, false, null, (short)0, (short)0, 10);
        addSQLTypeForJDBCType(handler, mconn, (short)Types.BIGINT, sqlType, true);
    }

    public String getVendorID()
    {
        return "db2";
    }
    
    public String getSchemaName(Connection conn) throws SQLException
    {
        Statement stmt = conn.createStatement();
        
        try
        {
            String stmtText = "VALUES (CURRENT SCHEMA)";
            ResultSet rs = stmt.executeQuery(stmtText);

            try
            {
                if (!rs.next())
                {
                    throw new NucleusDataStoreException("No result returned from " + stmtText).setFatal();
                }

                return rs.getString(1).trim();
            }
            finally
            {
                rs.close();
            }
        }
        finally
        {
            stmt.close();
        }
    }

    /**
     * Method to return the maximum length of a datastore identifier of the specified type.
     * If no limit exists then returns -1
     * @param identifierType Type of identifier (see IdentifierFactory.TABLE, etc)
     * @return The max permitted length of this type of identifier
     */
    public int getDatastoreIdentifierMaxLength(IdentifierType identifierType)
    {
        if (identifierType == IdentifierType.CANDIDATE_KEY)
        {
            return 18;
        }
        else if (identifierType == IdentifierType.FOREIGN_KEY)
        {
            return 18;
        }
        else if (identifierType == IdentifierType.INDEX)
        {
            return 18;
        }
        else if (identifierType == IdentifierType.PRIMARY_KEY)
        {
            return 18;
        }
        else
        {
            return super.getDatastoreIdentifierMaxLength(identifierType);
        }
    }

    public SQLTypeInfo newSQLTypeInfo(ResultSet rs)
    {
        return new org.datanucleus.store.rdbms.schema.DB2TypeInfo(rs);
    }

    /**
     * Method to create a column info for the current row.
     * Overrides the dataType/columnSize/decimalDigits to cater for DB2 particularities.
     * @param rs ResultSet from DatabaseMetaData.getColumns()
     * @return column info
     */
    public RDBMSColumnInfo newRDBMSColumnInfo(ResultSet rs)
    {
        RDBMSColumnInfo info = new RDBMSColumnInfo(rs);

        short dataType = info.getDataType();
        switch (dataType)
        {
            case Types.DATE:
            case Types.TIME:
            case Types.TIMESTAMP:
                // Values > 0 inexplicably get returned here.
                info.setDecimalDigits(0);
                break;
            default:
                break;
        }

        return info;
    }

    public int getUnlimitedLengthPrecisionValue(SQLTypeInfo typeInfo)
    {
        if (typeInfo.getDataType() == java.sql.Types.BLOB || typeInfo.getDataType() == java.sql.Types.CLOB)
        {
            return 1 << 30;
        }
        else
        {
            return super.getUnlimitedLengthPrecisionValue(typeInfo);
        }
    }

    public String getDropTableStatement(DatastoreContainerObject table)
    {
        return "DROP TABLE " + table.toString();
    }

	/**
	 * Accessor for the auto-increment sql statement for this datastore.
     * @param table Name of the table that the autoincrement is for
     * @param columnName Name of the column that the autoincrement is for
	 * @return The statement for getting the latest auto-increment key
	 **/
	public String getAutoIncrementStmt(Table table, String columnName)
	{
		return "VALUES IDENTITY_VAL_LOCAL()";
	}

	/**
	 * Accessor for the auto-increment keyword for generating DDLs (CREATE TABLEs...).
	 * @return The keyword for a column using auto-increment
	 **/
	public String getAutoIncrementKeyword()
	{
		return "generated always as identity (start with 1)";
	}

    /**
     * Continuation string to use where the SQL statement goes over more than 1
     * line. DB2 doesn't convert newlines into continuation characters and so
     * we just provide a space so that it accepts the statement.
     * @return Continuation string.
     **/
    public String getContinuationString()
    {
        return " ";
    }
    
    // ---------------------------- Sequence Support ---------------------------

    /**
     * Accessor for the sequence statement to create the sequence.
     * @param sequence_name Name of the sequence 
     * @param min Minimum value for the sequence
     * @param max Maximum value for the sequence
     * @param start Start value for the sequence
     * @param increment Increment value for the sequence
     * @param cache_size Cache size for the sequence
     * @return The statement for getting the next id from the sequence
     **/
    public String getSequenceCreateStmt(String sequence_name,
            Integer min,Integer max, Integer start,Integer increment, Integer cache_size)
    {
        if (sequence_name == null)
        {
            throw new NucleusUserException(LOCALISER.msg("051028"));
        }
        
        StringBuffer stmt = new StringBuffer("CREATE SEQUENCE ");
        stmt.append(sequence_name);
        stmt.append(" AS INTEGER ");

        if (start != null)
        {
            stmt.append(" START WITH " + start);
        }
        if (increment != null)
        {
            stmt.append(" INCREMENT BY " + increment);
        }
        if (min != null)
        {
            stmt.append(" MINVALUE " + min);
        }
        if (max != null)
        {
            stmt.append(" MAXVALUE " + max);
        }        
        if (cache_size != null)
        {
            stmt.append(" CACHE " + cache_size);
        }
        else
        {
            stmt.append(" NOCACHE");
        }
        
        return stmt.toString();
    }        
    
    /**
     * Accessor for the statement for getting the next id from the sequence for this datastore.
     * @param sequence_name Name of the sequence 
     * @return The statement for getting the next id for the sequence
     **/
    public String getSequenceNextStmt(String sequence_name)
    {
        if (sequence_name == null)
        {
            throw new NucleusUserException(LOCALISER.msg("051028"));
        }
        StringBuffer stmt=new StringBuffer("VALUES NEXTVAL FOR ");
        stmt.append(sequence_name);

        return stmt.toString();
    }

    // ------------------------------- Query Expression methods ------------------------------

    /**
     * A String conversion that converts a String literal to String expression. It will allow
     * the String to only be evaluated at runtime.
     * In SQL, it should compile something like:
     * <p>
     * <blockquote>
     * <pre>
     * CAST(999999 AS VARCHAR(32672))
     * </pre>
     * </blockquote>
     * </p>
     * @param expr The NumericExpression
     * @return the StringExpression
     */
    public StringExpression toStringExpression(StringLiteral expr)
    {
        List args = new ArrayList();
        args.add(expr);        
        List types = new ArrayList();
        types.add("VARCHAR(32672)");        
        return new StringExpression("CAST", args, types);        
    }

    
    /**
     * Method to translate all chars in this expression to the <code>fromExpr</code> which 
     * corresponds to <code>toExpr</code>.
     * @return The expression.
     **/
    public StringExpression translateMethod(ScalarExpression expr, ScalarExpression toExpr, ScalarExpression fromExpr)
    {
        ArrayList args = new ArrayList();
        args.add(expr);
        args.add(toExpr);
        args.add(fromExpr);

        return new StringExpression("TRANSLATE", args);
    }    

    /**
     * Method to handle the starts with operation.
     * @param source The expression with the searched string
     * @param str The expression for the search string 
     * @return The expression.
     **/
    public BooleanExpression startsWithMethod(ScalarExpression source, ScalarExpression str)
    {
        ScalarExpression integerLiteral = getMapping(BigInteger.class, source).newLiteral(source.getQueryExpression(), BigInteger.ONE);
        ArrayList args = new ArrayList();
        if (str instanceof StringLiteral)
        {
            args.add(toStringExpression((StringLiteral) str));
        }
        else
        {
            args.add(str);
        }
        if (source instanceof StringLiteral)
        {
            args.add(toStringExpression((StringLiteral) source));
        }
        else
        {
            args.add(source);
        }
		//DB2 8 manual
        //a VARCHAR the maximum length is 4 000 bytes 
        //LOCATE( CAST(stringSearched AS VARCHAR(4000)), SearchString, [StartPosition] )
        return new BooleanExpression(new StringExpression("LOCATE", args),ScalarExpression.OP_EQ,integerLiteral);
    }
    
    /**
     * <p>
     * If only one operand expression is of type String, then string conversion
     * is performed on the other operand to produce a string at run time. The
     * result is a reference to a String object (newly created, unless the
     * expression is a compile-time constant expression (�15.28))that is the
     * concatenation of the two operand strings. The characters of the left-hand
     * operand precede the characters of the right-hand operand in the newly
     * created string. If an operand of type String is null, then the string
     * "null" is used instead of that operand. "null" is used instead of that
     * operand.
     * </p>
     * <p>
     * Concatenates two or more character or binary strings, columns, or a
     * combination of strings and column names into one expression (a string
     * operator).
     * </p>
     * @param operand1 the left expression
     * @param operand2 the right expression
     * @return The Expression for concatenation
     */
    public ScalarExpression concatOperator(ScalarExpression operand1, ScalarExpression operand2)
    {
        /*
         * We cast it to VARCHAR otherwise the concatenation in derby it is promoted to
         * LONG VARCHAR.
         * 
         * In Derby, ? string parameters are promoted to LONG VARCHAR, and Derby
         * does not allow comparisons between LONG VARCHAR types. so the below 
         * example would be invalid in Derby
         * (THIS.FIRSTNAME||?) = ?
         * 
         * Due to that we convert it to
         *  
         * (CAST(THIS.FIRSTNAME||? AS VARCHAR(4000))) = ?
         * 
         * The only issue with this solution is for columns bigger than 4000 chars.
         * 
         * Secondly, if both operands are parameters, derby does not allow concatenation
         * e.g.
         * 
         * ? || ? is not allowed by derby
         * 
         * so we do
         * 
         * CAST( ? AS VARCHAR(4000) ) || CAST( ? AS VARCHAR(4000) )
         * 
         * If both situations happen,
         *  
         * (CAST(CAST( ? AS VARCHAR(4000) ) || CAST( ? AS VARCHAR(4000) ) AS VARCHAR(4000))) = ? 
         */
        List types = new ArrayList();
        types.add("VARCHAR(4000)");        

        List argsOp1 = new ArrayList();
        argsOp1.add(operand1);        

        List argsOp2 = new ArrayList();
        argsOp2.add(operand2);        
        
        List args = new ArrayList();
        args.add(super.concatOperator(new StringExpression("CAST", argsOp1, types), new StringExpression("CAST", argsOp2, types)));        

        return new StringExpression("CAST", args, types);
    }
    /**
     * Returns whether this string ends with the specified string.
     * @param leftOperand the source string
     * @param rightOperand The string to compare against.
     * @return Whether it ends with the string.
     **/
    public BooleanExpression endsWithMethod(ScalarExpression leftOperand, ScalarExpression rightOperand)
    {
        if (!(rightOperand instanceof StringExpression))
        {
            throw new IllegalArgumentTypeException(rightOperand);
        }
        List types = new ArrayList();
        types.add("VARCHAR(4000)");        

        List argsOp1 = new ArrayList();
        argsOp1.add(leftOperand);        
        
        StringLiteral pct = new StringLiteral(leftOperand.getQueryExpression(), leftOperand.getMapping(), '%');
        return new BooleanExpression(new StringExpression("CAST", argsOp1, types), ScalarExpression.OP_LIKE, pct.add(leftOperand.getQueryExpression().getStoreManager().getDatastoreAdapter().getEscapedPatternExpression(rightOperand)));
    }

    /**
     * Method to generate a modulus expression. The binary % operator is said to
     * yield the remainder of its operands from an implied division; the
     * left-hand operand is the dividend and the right-hand operand is the
     * divisor. This returns MOD(expr1, expr2).
     * @param operand1 the left expression
     * @param operand2 the right expression
     * @return The Expression for modulus
     */
    public NumericExpression modOperator(ScalarExpression operand1, ScalarExpression operand2)
    {
        List  args = new ArrayList();

        List types = new ArrayList();
        types.add("BIGINT");        
        
        List argsOp1 = new ArrayList();
        argsOp1.add(operand1);        
        args.add(new NumericExpression("CAST", argsOp1, types));

        List argsOp2 = new ArrayList();
        argsOp2.add(operand2);        
        args.add(new NumericExpression("CAST", argsOp2, types));

        return new NumericExpression("MOD", args);
    } 

    /**
     * Accessor for a numeric expression to represent the method call, with passed argument.
     * @param method The method (case insensitive)
     * @param expr The argument to the method
     * @return The numeric expression that results
     */
    public NumericExpression getNumericExpressionForMethod(String method, ScalarExpression expr)
    {
        if (method.equalsIgnoreCase("length"))
        {
            ArrayList args = new ArrayList();
            args.add(expr);
            return new NumericExpression("LENGTH", args);
        }
        else
        {
            return super.getNumericExpressionForMethod(method, expr);
        }
    }

    public StringExpression substringMethod(StringExpression str,
                                            NumericExpression begin)
    {
        ArrayList args = new ArrayList();
        if( str instanceof StringLiteral )
        {
            args.add(toStringExpression((StringLiteral) str));
        }
        else
        {
            args.add(str);
        }
        args.add(begin.add(getMapping(BigInteger.class, str).newLiteral(str.getQueryExpression(), BigInteger.ONE)));
        //DB2 8.0 manual
        //SUBSTR( string, start )
        return new StringExpression("SUBSTR", args);
    }

    public StringExpression substringMethod(StringExpression str,
                                               NumericExpression begin,
                                               NumericExpression end)
    {
        ArrayList args = new ArrayList();
        if( str instanceof StringLiteral )
        {
            args.add(toStringExpression((StringLiteral) str));
        }
        else
        {
            args.add(str);
        }
        args.add(begin.add(getMapping(BigInteger.class, str).newLiteral(str.getQueryExpression(), BigInteger.ONE)));
        args.add(end.sub(begin));
        //DB2 8.0
        //SUBSTR( string, start, length )
        return new StringExpression("SUBSTR", args);
    }
    
    /**
     * Returns the appropriate SQL expression for the java query "trim" method.
     * It should return something like:
     * <pre>LTRIM(RTRIM(str))</pre>
     * @param str The first argument to the trim() method.
     * @param leading Whether to trim leading spaces
     * @param trailing Whether to trim trailing spaces
     * @return  The text of the SQL expression.
     */
    public StringExpression trimMethod(StringExpression str, boolean leading, boolean trailing)
    {
        ArrayList args = new ArrayList();
        args.add(str);
        if (leading && trailing)
        {
            StringExpression strExpr = new StringExpression("RTRIM", args);
            args.clear();
            args.add(strExpr);
            return new StringExpression("LTRIM", args);
        }
        else if (leading)
        {
            return new StringExpression("LTRIM", args);
        }
        else if (trailing)
        {
            return new StringExpression("RTRIM", args);
        }
        return str;
    }

    /**
     * Returns the appropriate SQL expression for the JDOQL String.indexOf() method.
     * It should return something like:
     * <p>
     * <blockquote><pre>
     * LOCATE(str, substr [,pos])-1
     * </pre></blockquote>
     * since LOCATE returns the first character as position 1. Similarly the "pos" is based on the first
     * position being 1.
     * </p>
     * @param source The expression we want to search.
     * @param str The argument to the indexOf() method.
     * @param from The from position
     * @return The text of the SQL expression.
     */
    public NumericExpression indexOfMethod(ScalarExpression source, ScalarExpression str, NumericExpression from)
    {
        ScalarExpression integerLiteral = getMapping(BigInteger.class, source).newLiteral(source.getQueryExpression(), BigInteger.ONE);

        ArrayList args = new ArrayList();
        args.add(source);
        List argsOp0 = new ArrayList();
        argsOp0.add(str);        
        List types = new ArrayList();
        types.add("VARCHAR(4000)"); // max 4000 according DB2 docs        
        args.add(new StringExpression("CAST", argsOp0, types));
        if (from != null)
        {
            types = new ArrayList();
            types.add("BIGINT");        

            List argsOp1 = new ArrayList();
            argsOp1.add(new NumericExpression(from, ScalarExpression.OP_ADD, integerLiteral));        
           
            // Add 1 to the passed in value so that it is of origin 1 to be compatible with LOCATE
            // Make sure argument is typed as BIGINT  
            args.add(new NumericExpression("CAST", argsOp1, types));            
        }
        NumericExpression locateExpr = new NumericExpression("LOCATE", args);

        // Subtract 1 from the result of LOCATE to be consistent with Java strings
        // TODO Would be nice to put this in parentheses
        return new NumericExpression(locateExpr, ScalarExpression.OP_SUB, integerLiteral);
    }
}