/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
 **********************************************************************/
package org.datanucleus.store.rdbms.query;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Stack;
import java.util.StringTokenizer;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.FetchPlan;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.exceptions.NucleusUserException;
import org.datanucleus.metadata.AbstractClassMetaData;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.IdentityType;
import org.datanucleus.metadata.MetaDataManager;
import org.datanucleus.metadata.Relation;
import org.datanucleus.query.QueryUtils;
import org.datanucleus.query.compiler.CompilationComponent;
import org.datanucleus.query.compiler.QueryCompilation;
import org.datanucleus.query.evaluator.AbstractExpressionEvaluator;
import org.datanucleus.query.expression.CastExpression;
import org.datanucleus.query.expression.ClassExpression;
import org.datanucleus.query.expression.CreatorExpression;
import org.datanucleus.query.expression.DyadicExpression;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.query.expression.InvokeExpression;
import org.datanucleus.query.expression.JoinExpression;
import org.datanucleus.query.expression.Literal;
import org.datanucleus.query.expression.OrderExpression;
import org.datanucleus.query.expression.ParameterExpression;
import org.datanucleus.query.expression.PrimaryExpression;
import org.datanucleus.query.expression.SubqueryExpression;
import org.datanucleus.query.expression.VariableExpression;
import org.datanucleus.query.expression.JoinExpression.JoinType;
import org.datanucleus.query.symbol.Symbol;
import org.datanucleus.store.mapped.DatastoreClass;
import org.datanucleus.store.mapped.DatastoreContainerObject;
import org.datanucleus.store.mapped.StatementClassMapping;
import org.datanucleus.store.mapped.StatementMappingIndex;
import org.datanucleus.store.mapped.StatementNewObjectMapping;
import org.datanucleus.store.mapped.StatementResultMapping;
import org.datanucleus.store.mapped.mapping.CollectionMapping;
import org.datanucleus.store.mapped.mapping.DatastoreMapping;
import org.datanucleus.store.mapped.mapping.EmbeddedMapping;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.mapped.mapping.MapMapping;
import org.datanucleus.store.mapped.mapping.MappingConsumer;
import org.datanucleus.store.mapped.mapping.StringMapping;
import org.datanucleus.store.mapped.mapping.TemporalMapping;
import org.datanucleus.store.query.QueryCompilerSyntaxException;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.sql.SQLStatement;
import org.datanucleus.store.rdbms.sql.SQLStatementHelper;
import org.datanucleus.store.rdbms.sql.SQLTable;
import org.datanucleus.store.rdbms.sql.expression.BooleanExpression;
import org.datanucleus.store.rdbms.sql.expression.BooleanSubqueryExpression;
import org.datanucleus.store.rdbms.sql.expression.NewObjectExpression;
import org.datanucleus.store.rdbms.sql.expression.NumericSubqueryExpression;
import org.datanucleus.store.rdbms.sql.expression.ParameterLiteral;
import org.datanucleus.store.rdbms.sql.expression.SQLExpression;
import org.datanucleus.store.rdbms.sql.expression.SQLExpressionFactory;
import org.datanucleus.store.rdbms.sql.expression.SQLLiteral;
import org.datanucleus.store.rdbms.sql.expression.StringSubqueryExpression;
import org.datanucleus.store.rdbms.sql.expression.SubqueryExpressionComponent;
import org.datanucleus.store.rdbms.sql.expression.TemporalSubqueryExpression;
import org.datanucleus.store.rdbms.sql.expression.UnboundExpression;
import org.datanucleus.store.rdbms.table.CollectionTable;
import org.datanucleus.store.rdbms.table.ElementContainerTable;
import org.datanucleus.util.ClassUtils;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;
import org.datanucleus.util.StringUtils;

/**
 * Class which maps a compiled (generic) query to an SQL query for RDBMS datastores.
 * Takes in an SQLStatement, and use of compile() will update the SQLStatement to reflect
 * the filter, result, grouping, having, ordering etc.
 */
public class QueryToSQLMapper extends AbstractExpressionEvaluator implements QueryGenerator
{
    /** Localiser for messages. */
    protected static final Localiser LOCALISER = Localiser.getInstance(
        "org.datanucleus.store.rdbms.Localisation", RDBMSStoreManager.class.getClassLoader());

    final String candidateAlias;

    final AbstractClassMetaData candidateCmd;

    final QueryCompilation compilation;

    /** Input parameter values, keyed by the parameter name. Will be null if compiled pre-execution. */
    final Map parameters;

    /** Positional parameter that we are up to (-1 implies not being used). */
    int positionalParamNumber = -1;

    /** SQL statement that we are updating. */
    final SQLStatement stmt;

    /** Definition of mapping for the results of this SQL statement (candidate). */
    final StatementClassMapping resultDefinitionForClass;

    /** Definition of mapping for the results of this SQL statement (result). */
    final StatementResultMapping resultDefinition;

    Map<Object, SQLExpression> expressionForParameter;

    final RDBMSStoreManager storeMgr;

    final FetchPlan fetchPlan;

    final SQLExpressionFactory exprFactory;

    ObjectManager om;

    ClassLoaderResolver clr;

    Map<String, Object> compileProperties = new HashMap();

    /** State variable for the component being compiled. */
    CompilationComponent compileComponent;

    /** State variable for the number of the next parameter. */
    int paramNumber = 1; // JDBC parameters start at 1

    /** Stack of expressions, used for compilation of the query into SQL. */
    Stack<SQLExpression> stack = new Stack();

    /** Map of SQLTable/mapping keyed by the name of the primary that it relates to. */
    Map<String, SQLTableMapping> sqlTableByPrimary = new HashMap<String, SQLTableMapping>();

    Map<String, JavaTypeMapping> paramMappingForName = new HashMap();

    boolean caseInsensitive = false;

    /** Parent mapper if we are processing a subquery. */
    public QueryToSQLMapper parentMapper = null;

    org.datanucleus.store.rdbms.sql.SQLJoin.JoinType defaultJoinType = null;

    /**
     * State variable for whether this query is precompilable (hence whether it is cacheable).
     * Or in other words, whether we can compile it without knowing parameter values.
     */
    boolean precompilable = true;

    class SQLTableMapping
    {
        SQLTable table;
        AbstractClassMetaData cmd;
        JavaTypeMapping mapping;
        public SQLTableMapping(SQLTable tbl, AbstractClassMetaData cmd, JavaTypeMapping m)
        {
            this.table = tbl;
            this.cmd = cmd;
            this.mapping = m;
        }
        public String toString()
        {
            return "SQLTableMapping: tbl=" + table + 
                " class=" + (cmd != null ? cmd.getFullClassName() : "null") + 
                " mapping=" + mapping;
        }
    }

    /**
     * Constructor.
     * @param stmt SQL Statement to update with the query contents.
     * @param compilation The generic query compilation
     * @param parameters Parameters needed
     * @param resultDefForClass Definition of results for the statement (populated here)
     * @param resultDef Definition of results if we have a result clause (populated here)
     * @param cmd Metadata for the candidate class
     * @param fetchPlan Fetch Plan to apply
     * @param om ObjectManager
     */
    public QueryToSQLMapper(SQLStatement stmt, QueryCompilation compilation, Map parameters,
            StatementClassMapping resultDefForClass, StatementResultMapping resultDef,
            AbstractClassMetaData cmd, FetchPlan fetchPlan, ObjectManager om)
    {
        this.parameters = parameters;
        this.compilation = compilation;
        this.stmt = stmt;
        this.resultDefinitionForClass = resultDefForClass;
        this.resultDefinition = resultDef;
        this.candidateAlias = compilation.getCandidateAlias();
        this.fetchPlan = fetchPlan;
        this.storeMgr = stmt.getRDBMSManager();
        this.exprFactory = stmt.getRDBMSManager().getSQLExpressionFactory();
        this.candidateCmd = cmd;
        this.om = om;
        this.clr = om.getClassLoaderResolver();

        if (compilation.getQueryLanguage().equalsIgnoreCase("JPQL"))
        {
            caseInsensitive = true;
        }

        this.stmt.setQueryGenerator(this);

        // Register the candidate
        SQLTableMapping tblMapping =
            new SQLTableMapping(stmt.getPrimaryTable(), candidateCmd, stmt.getPrimaryTable().getTable().getIdMapping());
        setSQLTableMappingForAlias(candidateAlias, tblMapping);
    }

    /**
     * Method to set the default join type to be used.
     * By default we use the nullability of the field to define whether to use LEFT OUTER, or INNER.
     * In JPQL if a relation isn't specified explicitly in the FROM clause then it should use INNER
     * hence where this method will be used, to define the default.
     * @param joinType The default join type
     */
    void setDefaultJoinType(org.datanucleus.store.rdbms.sql.SQLJoin.JoinType joinType)
    {
        this.defaultJoinType = joinType;
    }

    void setParentMapper(QueryToSQLMapper parent)
    {
        this.parentMapper = parent;
    }

    /**
     * Accessor for the query language that this query pertains to.
     * Can be used to decide how to handle the input.
     * @return The query language
     */
    public String getQueryLanguage()
    {
        return compilation.getQueryLanguage();
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.query.QueryGenerator#getClassLoaderResolver()
     */
    public ClassLoaderResolver getClassLoaderResolver()
    {
        return clr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.query.QueryGenerator#getCompilationComponent()
     */
    public CompilationComponent getCompilationComponent()
    {
        return compileComponent;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.query.QueryGenerator#getObjectManager()
     */
    public ObjectManager getObjectManager()
    {
        return om;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.query.QueryGenerator#getProperty(java.lang.String)
     */
    public Object getProperty(String name)
    {
        return compileProperties.get(name);
    }

    /**
     * Accessor for whether the query is precompilable (doesn't need to know parameter values
     * to be able to compile it).
     * @return Whether the query is precompilable and hence cacheable
     */
    public boolean isPrecompilable()
    {
        return precompilable;
    }

    /**
     * Method to update the supplied SQLStatement with the components of the specified query.
     * During the compilation process this updates the SQLStatement "compileComponent" to the
     * component of the query being compiled.
     */
    public void compile()
    {
        compileFrom();
        compileFilter();
        compileResult();
        compileGrouping();
        compileHaving();
        compileOrdering();

        if (compilation.getExprResult() == null)
        {
            // Select of the candidate (no result)
            // Note this is done last etc since the filter/grouping/ordering etc may bring in the necessary 
            // tables for the candidate fields being selected here
            if (candidateCmd.getIdentityType() == IdentityType.NONDURABLE)
            {
                // Nondurable identity cases have no "id" for later fetching so get all fields now
                if (NucleusLogger.QUERY.isDebugEnabled())
                {
                    NucleusLogger.QUERY.debug(LOCALISER.msg("052520", candidateCmd.getFullClassName()));
                }
                fetchPlan.setGroup("all");
            }

            if (stmt.allUnionsForSamePrimaryTable())
            {
                // Select fetch-plan members of the candidate (and optionally the next level of sub-objects)
                // Don't select next level when we are processing a subquery
                SQLStatementHelper.selectFetchPlanOfCandidateInStatement(stmt, resultDefinitionForClass, 
                    candidateCmd, fetchPlan, parentMapper == null ? 1 : 0);
            }
            else
            {
                // Select identity of the candidates since use different base tables
                // TODO complete-table will come through here but maybe ought to be treated differently
                SQLStatementHelper.selectIdentityOfCandidateInStatement(stmt, resultDefinitionForClass, 
                    candidateCmd);
            }
        }

        // Check for variables that haven't been bound to the query (declared but not used)
        Collection<String> symbols = compilation.getSymbolTable().getSymbolNames();
        Iterator<String> symIter = symbols.iterator();
        while (symIter.hasNext())
        {
            Symbol sym = compilation.getSymbolTable().getSymbol(symIter.next());
            if (sym.getType() == Symbol.VARIABLE)
            {
                if (compilation.getCompilationForSubquery(sym.getQualifiedName()) == null &&
                    !hasSQLTableMappingForAlias(sym.getQualifiedName()))
                {
                    // Variable not a subquery, nor had its table allocated
                    throw new QueryCompilerSyntaxException("Query has variable \"" + sym.getQualifiedName() + "\" which is not bound to the query");
                }
            }
        }
    }

    /**
     * Method to compile the FROM clause of the query into the SQLStatement.
     */
    protected void compileFrom()
    {
        if (compilation.getExprFrom() != null)
        {
            // Process all ClassExpression(s) in the FROM, adding joins to the statement as required
            compileComponent = CompilationComponent.FROM;
            Expression[] fromExprs = compilation.getExprFrom();
            for (int i=0;i<fromExprs.length;i++)
            {
                ClassExpression clsExpr = (ClassExpression)fromExprs[i];
                compileFromClassExpression(clsExpr);
            }
            compileComponent = null;
        }
    }

    /**
     * Method to compile the WHERE clause of the query into the SQLStatement.
     */
    protected void compileFilter()
    {
        if (compilation.getExprFilter() != null)
        {
            // Apply the filter to the SQLStatement
            compileComponent = CompilationComponent.FILTER;
            if (QueryUtils.expressionHasOrOperator(compilation.getExprFilter()))
            {
                compileProperties.put("Filter.OR", true);
            }
            if (QueryUtils.expressionHasNotOperator(compilation.getExprFilter()))
            {
                // Really we want to know if there is a NOT contains, but just check NOT for now
                compileProperties.put("Filter.NOT", true);
            }
            BooleanExpression filterExpr = (BooleanExpression)compilation.getExprFilter().evaluate(this);
            stmt.whereAnd(filterExpr, true);
            compileComponent = null;
        }
    }

    /**
     * Method to compile the result clause of the query into the SQLStatement.
     */
    protected void compileResult()
    {
        if (compilation.getExprUpdate() != null)
        {
            // Update statement, so no select
            compileComponent = CompilationComponent.UPDATE;
            Expression[] updateExprs = compilation.getExprUpdate();
            SQLExpression[] updateSqlExprs = new SQLExpression[updateExprs.length];
            for (int i=0;i<updateExprs.length;i++)
            {
                // "field = value"
                DyadicExpression updateExpr = (DyadicExpression)updateExprs[i];

                // Left-side has to be PrimaryExpression
                SQLExpression leftSqlExpr = null;
                if (updateExpr.getLeft() instanceof PrimaryExpression)
                {
                    processPrimaryExpression((PrimaryExpression)updateExpr.getLeft());
                    leftSqlExpr = stack.pop();
                }
                else
                {
                    throw new NucleusException("Dont currently support update clause containing left expression of type " + updateExpr.getLeft());
                }

                // Right-side can be Literal, or Parameter, or PrimaryExpression
                SQLExpression rightSqlExpr = null;
                if (updateExpr.getRight() instanceof Literal)
                {
                    processLiteral((Literal)updateExpr.getRight());
                    rightSqlExpr = stack.pop();
                }
                else if (updateExpr.getRight() instanceof ParameterExpression)
                {
                    processParameterExpression((ParameterExpression)updateExpr.getRight());
                    rightSqlExpr = stack.pop();
                }
                else if (updateExpr.getRight() instanceof PrimaryExpression)
                {
                    processPrimaryExpression((PrimaryExpression)updateExpr.getRight());
                    rightSqlExpr = stack.pop();
                }
                else if (updateExpr.getRight() instanceof DyadicExpression)
                {
                    updateExpr.getRight().evaluate(this);
                    rightSqlExpr = stack.pop();
                }
                else
                {
                    throw new NucleusException("Dont currently support update clause containing right expression of type " + updateExpr.getRight());
                }

                if (leftSqlExpr != null && rightSqlExpr != null)
                {
                    updateSqlExprs[i] = leftSqlExpr.eq(rightSqlExpr);
                }
            }
            stmt.setUpdates(updateSqlExprs);
            compileComponent = null;
        }
        else if (compilation.getExprResult() != null)
        {
            compileComponent = CompilationComponent.RESULT;

            // Select any result expressions
            Expression[] resultExprs = compilation.getExprResult();
            for (int i=0;i<resultExprs.length;i++)
            {
                String alias = resultExprs[i].getAlias();
                if (resultExprs[i] instanceof InvokeExpression)
                {
                    processInvokeExpression((InvokeExpression)resultExprs[i]);
                    SQLExpression sqlExpr = stack.pop();
                    validateExpressionForResult(sqlExpr);
                    int col = stmt.select(sqlExpr, alias);
                    StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                    idx.setColumnPositions(new int[] {col});
                    if (alias != null)
                    {
                        idx.setColumnAlias(alias);
                    }
                    resultDefinition.addMappingForResultExpression(i, idx);
                }
                else if (resultExprs[i] instanceof PrimaryExpression)
                {
                    PrimaryExpression primExpr = (PrimaryExpression)resultExprs[i];
                    if (primExpr.getId().equals(candidateAlias))
                    {
                        // "this", so select fetch plan fields
                        StatementClassMapping map = new StatementClassMapping(candidateCmd.getFullClassName(), null);
                        SQLStatementHelper.selectFetchPlanOfCandidateInStatement(stmt, map, 
                            candidateCmd, fetchPlan, 1);
                        resultDefinition.addMappingForResultExpression(i, map);
                    }
                    else
                    {
                        processPrimaryExpression(primExpr);
                        SQLExpression sqlExpr = stack.pop();
                        validateExpressionForResult(sqlExpr);
                        if (sqlExpr instanceof SQLLiteral)
                        {
                            int col = stmt.select(sqlExpr, alias);
                            StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                            idx.setColumnPositions(new int[] {col});
                            if (alias != null)
                            {
                                idx.setColumnAlias(alias);
                            }
                            resultDefinition.addMappingForResultExpression(i, idx);
                        }
                        else
                        {
                            int[] cols = stmt.select(sqlExpr.getSQLTable(), sqlExpr.getJavaTypeMapping(), alias);
                            StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                            idx.setColumnPositions(cols);
                            if (alias != null)
                            {
                                idx.setColumnAlias(alias);
                            }
                            resultDefinition.addMappingForResultExpression(i, idx);
                        }
                    }
                }
                else if (resultExprs[i] instanceof ParameterExpression)
                {
                    processParameterExpression((ParameterExpression)resultExprs[i], true);
                    SQLExpression sqlExpr = stack.pop();
                    validateExpressionForResult(sqlExpr);
                    int col = stmt.select(sqlExpr, alias);
                    StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                    idx.setColumnPositions(new int[] {col});
                    if (alias != null)
                    {
                        idx.setColumnAlias(alias);
                    }
                    resultDefinition.addMappingForResultExpression(i, idx);
                }
                else if (resultExprs[i] instanceof VariableExpression)
                {
                    processVariableExpression((VariableExpression)resultExprs[i]);
                    SQLExpression sqlExpr = stack.pop();
                    validateExpressionForResult(sqlExpr);
                    if (sqlExpr instanceof UnboundExpression)
                    {
                        // Variable wasn't bound in the compilation so far, so handle as cross-join
                        processUnboundExpression((UnboundExpression) sqlExpr);
                        sqlExpr = stack.pop();
                        NucleusLogger.QUERY.debug(">> QueryToSQL.exprResult variable was still unbound, so binding via cross-join");
                    }
                    StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                    if (sqlExpr.getNumberOfSubExpressions() == 1)
                    {
                        int col = stmt.select(sqlExpr, alias);
                        idx.setColumnPositions(new int[] {col});
                    }
                    else
                    {
                        int[] cols = new int[sqlExpr.getNumberOfSubExpressions()];
                        for (int j=0;j<sqlExpr.getNumberOfSubExpressions();j++)
                        {
                            String colAlias = (alias != null ? (alias + "_" + j) : null);
                            int col = stmt.select(sqlExpr.getSubExpression(j), colAlias);
                            cols[j] = col;
                        }
                        idx.setColumnPositions(cols);
                    }
                    if (alias != null)
                    {
                        idx.setColumnAlias(alias);
                    }
                    resultDefinition.addMappingForResultExpression(i, idx);
                }
                else if (resultExprs[i] instanceof Literal)
                {
                    processLiteral((Literal)resultExprs[i]);
                    SQLExpression sqlExpr = stack.pop();
                    validateExpressionForResult(sqlExpr);
                    int col = stmt.select(sqlExpr, alias);
                    StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                    idx.setColumnPositions(new int[] {col});
                    if (alias != null)
                    {
                        idx.setColumnAlias(alias);
                    }
                    resultDefinition.addMappingForResultExpression(i, idx);
                }
                else if (resultExprs[i] instanceof CreatorExpression)
                {
                    processCreatorExpression((CreatorExpression)resultExprs[i]);
                    NewObjectExpression sqlExpr = (NewObjectExpression)stack.pop();
                    StatementNewObjectMapping stmtMap = getStatementMappingForNewObjectExpression(sqlExpr);
                    resultDefinition.addMappingForResultExpression(i, stmtMap);
                }
                else if (resultExprs[i] instanceof DyadicExpression)
                {
                    // Something like {a+b} perhaps. Maybe we should check for invalid expressions?
                    resultExprs[i].evaluate(this);
                    SQLExpression sqlExpr = stack.pop();
                    int col = stmt.select(sqlExpr, alias);
                    StatementMappingIndex idx = new StatementMappingIndex(sqlExpr.getJavaTypeMapping());
                    idx.setColumnPositions(new int[] {col});
                    if (alias != null)
                    {
                        idx.setColumnAlias(alias);
                    }
                    resultDefinition.addMappingForResultExpression(i, idx);
                }
                else
                {
                    throw new NucleusException("Dont currently support result clause containing expression of type " + resultExprs[i]);
                }
            }

            if (stmt.getNumberOfSelects() == 0)
            {
                // Nothing selected so likely the user had some "new MyClass()" expression, so select "1"
                stmt.select(exprFactory.newLiteral(stmt,
                    storeMgr.getMappingManager().getMapping(Integer.class), 1), null);
            }
            compileComponent = null;
        }
    }

    /**
     * Method that validates that the specified expression is valid for use in a result clause.
     * Throws a NucleusUserException if it finds a multi-value mapping selected (collection, map).
     * @param sqlExpr The SQLExpression
     */
    protected void validateExpressionForResult(SQLExpression sqlExpr)
    {
        JavaTypeMapping m = sqlExpr.getJavaTypeMapping();
        if (m != null)
        {
            if (m instanceof CollectionMapping || m instanceof MapMapping)
            {
                throw new NucleusUserException("Cannot select multi-valued objects in a result clause of a query");
            }
        }
    }

    /**
     * Method to compile the grouping clause of the query into the SQLStatement.
     */
    protected void compileGrouping()
    {
        if (compilation.getExprGrouping() != null)
        {
            // Apply any grouping to the statement
            compileComponent = CompilationComponent.GROUPING;
            Expression[] groupExprs = compilation.getExprGrouping();
            for (int i = 0; i < groupExprs.length; i++)
            {
                Expression groupExpr = groupExprs[i];
                SQLExpression sqlGroupExpr = (SQLExpression)groupExpr.evaluate(this);
                stmt.addGroupingExpression(sqlGroupExpr);
            }
            compileComponent = null;
        }
    }

    /**
     * Method to compile the having clause of the query into the SQLStatement.
     */
    protected void compileHaving()
    {
        if (compilation.getExprHaving() != null)
        {
            // Apply any having to the statement
            compileComponent = CompilationComponent.HAVING;
            Expression havingExpr = compilation.getExprHaving();
            Object havingEval = havingExpr.evaluate(this);
            if (!(havingEval instanceof BooleanExpression))
            {
                // Non-boolean having clause should be user exception
                throw new NucleusUserException(LOCALISER.msg("021051", havingExpr));
            }
            stmt.setHaving((BooleanExpression)havingEval);
            compileComponent = null;
        }
    }

    /**
     * Method to compile the ordering clause of the query into the SQLStatement.
     */
    protected void compileOrdering()
    {
        if (compilation.getExprOrdering() != null)
        {
            compileComponent = CompilationComponent.ORDERING;
            Expression[] orderingExpr = compilation.getExprOrdering();
            SQLExpression[] orderSqlExprs = new SQLExpression[orderingExpr.length];
            boolean[] directions = new boolean[orderingExpr.length];
            for (int i = 0; i < orderingExpr.length; i++)
            {
                OrderExpression orderExpr = (OrderExpression)orderingExpr[i];
                orderSqlExprs[i] = (SQLExpression)orderExpr.getLeft().evaluate(this);
                String orderDir = orderExpr.getSortOrder();
                directions[i] = ((orderDir == null || orderDir.equals("ascending")) ? false : true);
            }
            stmt.setOrdering(orderSqlExprs, directions);
            compileComponent = null;
        }
    }

    /**
     * Method to take a ClassExpression (in a FROM clause) and process the candidate and any
     * linked JoinExpression(s), adding joins to the SQLStatement as required.
     * @param clsExpr The ClassExpression
     */
    protected void compileFromClassExpression(ClassExpression clsExpr)
    {
        Symbol clsExprSym = clsExpr.getSymbol();
        Class baseCls = clsExprSym.getValueType();
        SQLTable candSqlTbl = stmt.getPrimaryTable();
        MetaDataManager mmgr = storeMgr.getMetaDataManager();
        AbstractClassMetaData cmd = mmgr.getMetaDataForClass(baseCls, clr);
        if (baseCls != null && baseCls != compilation.getCandidateClass())
        {
            // Not candidate class so must be cross join (JPA spec 4.4.5)
            DatastoreClass candTbl = storeMgr.getDatastoreClass(baseCls.getName(), clr);
            candSqlTbl = stmt.crossJoin(candTbl, clsExpr.getAlias(), null);
            SQLTableMapping tblMapping = new SQLTableMapping(candSqlTbl, cmd, candTbl.getIdMapping());
            setSQLTableMappingForAlias(clsExpr.getAlias(), tblMapping);
        }

        if (clsExpr.getCandidateExpression() != null && parentMapper != null)
        {
            // User defined the candidate of the subquery as an implied join to the outer query
            // e.g SELECT c FROM Customer c WHERE EXISTS (SELECT o FROM c.orders o ...)
            // so add the join to the outer query
            StringTokenizer tokeniser = new StringTokenizer(clsExpr.getCandidateExpression(), ".");
            String outerAlias = tokeniser.nextToken();
            String joinedField = tokeniser.nextToken();
            // TODO Cater for more than single relation join (possible with JPQL?)

            // Find the SQLTable etc in the outer query so we can join to it
            SQLTableMapping outerSqlTblMapping = parentMapper.getSQLTableMappingForAlias(outerAlias);
            SQLTable outerSqlTbl = outerSqlTblMapping.table;
            AbstractClassMetaData outerCmd = outerSqlTblMapping.cmd;
            AbstractMemberMetaData mmd = outerCmd.getMetaDataForMember(joinedField);
            int relationType = mmd.getRelationType(clr);
            switch (relationType)
            {
                case Relation.ONE_TO_ONE_UNI:
                {
                    SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                        outerSqlTbl, outerSqlTbl.getTable().getMemberMapping(mmd));
                    SQLExpression innerExpr = exprFactory.newExpression(stmt, candSqlTbl,
                        candSqlTbl.getTable().getIdMapping());
                    stmt.whereAnd(outerExpr.eq(innerExpr), false);
                    break;
                }
                case Relation.ONE_TO_ONE_BI:
                {
                    if (mmd.getMappedBy() != null)
                    {
                        AbstractMemberMetaData relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getIdMapping());
                        SQLExpression innerExpr = exprFactory.newExpression(stmt, candSqlTbl,
                            candSqlTbl.getTable().getMemberMapping(relMmd));
                        stmt.whereAnd(outerExpr.eq(innerExpr), false);
                    }
                    else
                    {
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getMemberMapping(mmd));
                        SQLExpression innerExpr = exprFactory.newExpression(stmt, candSqlTbl,
                            candSqlTbl.getTable().getIdMapping());
                        stmt.whereAnd(outerExpr.eq(innerExpr), false);
                    }
                    break;
                }
                case Relation.ONE_TO_MANY_UNI:
                {
                    AbstractMemberMetaData relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                    if (mmd.getJoinMetaData() != null || relMmd.getJoinMetaData() != null)
                    {
                        // Join to join table, then to related table
                        ElementContainerTable joinTbl = (ElementContainerTable)storeMgr.getDatastoreContainerObject(mmd);
                        SQLTable joinSqlTbl = stmt.innerJoin(candSqlTbl, candSqlTbl.getTable().getIdMapping(),
                            joinTbl, null, joinTbl.getElementMapping(), null, null);
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getIdMapping());
                        SQLExpression joinExpr = exprFactory.newExpression(stmt, joinSqlTbl,
                            joinTbl.getOwnerMapping());
                        stmt.whereAnd(outerExpr.eq(joinExpr), false);
                    }
                    else
                    {
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getIdMapping());
                        SQLExpression innerExpr = exprFactory.newExpression(stmt, candSqlTbl,
                            candSqlTbl.getTable().getMemberMapping(relMmd));
                        stmt.whereAnd(outerExpr.eq(innerExpr), false);
                    }
                    break;
                }
                case Relation.ONE_TO_MANY_BI:
                {
                    AbstractMemberMetaData relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                    if (mmd.getJoinMetaData() != null || relMmd.getJoinMetaData() != null)
                    {
                        // Join to join table, then to related table
                        ElementContainerTable joinTbl = (ElementContainerTable)storeMgr.getDatastoreContainerObject(mmd);
                        SQLTable joinSqlTbl = stmt.innerJoin(candSqlTbl, candSqlTbl.getTable().getIdMapping(),
                            joinTbl, null, joinTbl.getElementMapping(), null, null);
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getIdMapping());
                        SQLExpression joinExpr = exprFactory.newExpression(stmt, joinSqlTbl,
                            joinTbl.getOwnerMapping());
                        stmt.whereAnd(outerExpr.eq(joinExpr), false);
                    }
                    else
                    {
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getIdMapping());
                        SQLExpression innerExpr = exprFactory.newExpression(stmt, candSqlTbl,
                            candSqlTbl.getTable().getMemberMapping(relMmd));
                        stmt.whereAnd(outerExpr.eq(innerExpr), false);
                    }
                    break;
                }
                case Relation.MANY_TO_ONE_BI:
                {
                    AbstractMemberMetaData relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                    if (mmd.getJoinMetaData() != null || relMmd.getJoinMetaData() != null)
                    {
                        // Join to join table, then to related table
                        ElementContainerTable joinTbl = (CollectionTable)storeMgr.getDatastoreContainerObject(relMmd);
                        SQLTable joinSqlTbl = stmt.innerJoin(candSqlTbl, candSqlTbl.getTable().getIdMapping(),
                            joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getIdMapping());
                        SQLExpression joinExpr = exprFactory.newExpression(stmt, joinSqlTbl,
                            joinTbl.getElementMapping());
                        stmt.whereAnd(outerExpr.eq(joinExpr), false);
                    }
                    else
                    {
                        SQLExpression outerExpr = exprFactory.newExpression(stmt.getParentStatement(),
                            outerSqlTbl, outerSqlTbl.getTable().getMemberMapping(mmd));
                        SQLExpression innerExpr = exprFactory.newExpression(stmt, candSqlTbl,
                            candSqlTbl.getTable().getIdMapping());
                        stmt.whereAnd(outerExpr.eq(innerExpr), false);
                    }
                }
                default:
                    break;
            }
        }

        // Process all join expressions
        Expression rightExpr = clsExpr.getRight();
        SQLTable sqlTbl = candSqlTbl;
        while (rightExpr != null)
        {
            if (rightExpr instanceof JoinExpression)
            {
                JoinExpression joinExpr = (JoinExpression)rightExpr;
                JoinType joinType = joinExpr.getType();
                String joinAlias = joinExpr.getAlias();
                PrimaryExpression joinPrimExpr = joinExpr.getPrimaryExpression();
                String joinPrimExprId = joinPrimExpr.getId();
                String joinTableGroupName = null;
                if (joinPrimExprId.toUpperCase().startsWith(candidateAlias.toUpperCase()))
                {
                    // Name table group of joined-to as per the relation
                    // Note : this will only work for one level out from the candidate TODO Extend this
                    joinTableGroupName = joinPrimExprId;
                }

                List<String> joinPrimTuples = joinPrimExpr.getTuples();
                Iterator<String> iter = joinPrimTuples.iterator();
                iter.next(); // Omit reference tuple since this is either the candidate or a cross join (above)
                while (iter.hasNext())
                {
                    String id = iter.next();
                    AbstractMemberMetaData mmd = cmd.getMetaDataForMember(id);
                    int relationType = mmd.getRelationType(clr);
                    DatastoreClass relTable = null;
                    AbstractMemberMetaData relMmd = null;
                    // TODO Cater for map in 1-N relations?
                    switch (relationType)
                    {
                        case Relation.ONE_TO_ONE_UNI:
                            relTable = storeMgr.getDatastoreClass(mmd.getTypeName(), clr);
                            cmd = mmgr.getMetaDataForClass(mmd.getType(), clr);
                            if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                            {
                                sqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getMemberMapping(mmd),
                                    relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                            }
                            else
                            {
                                sqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getMemberMapping(mmd),
                                    relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                            }
                            break;
                        case Relation.ONE_TO_ONE_BI:
                            relTable = storeMgr.getDatastoreClass(mmd.getTypeName(), clr);
                            cmd = storeMgr.getMetaDataManager().getMetaDataForClass(mmd.getType(), clr);
                            if (mmd.getMappedBy() != null)
                            {
                                relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    sqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        relTable, null, relTable.getMemberMapping(relMmd), null, joinTableGroupName);
                                }
                                else
                                {
                                    sqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(),
                                        relTable, null, relTable.getMemberMapping(relMmd), null, joinTableGroupName);
                                }
                            }
                            else
                            {
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    sqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getMemberMapping(mmd), relTable, null, 
                                        relTable.getIdMapping(), null, joinTableGroupName);
                                }
                                else
                                {
                                    sqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getMemberMapping(mmd), relTable, null, 
                                        relTable.getIdMapping(), null, joinTableGroupName);
                                }
                            }
                            break;
                        case Relation.ONE_TO_MANY_BI:
                            relTable = storeMgr.getDatastoreClass(mmd.getCollection().getElementType(), clr);
                            cmd = mmd.getCollection().getElementClassMetaData(clr, mmgr);
                            relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                            if (mmd.getJoinMetaData() != null || relMmd.getJoinMetaData() != null)
                            {
                                // Join to join table, then to related table
                                ElementContainerTable joinTbl = (ElementContainerTable)storeMgr.getDatastoreContainerObject(mmd);
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    SQLTable joinSqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                                    sqlTbl = stmt.innerJoin(joinSqlTbl, joinTbl.getElementMapping(), 
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                                else
                                {
                                    SQLTable joinSqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                                    sqlTbl = stmt.leftOuterJoin(joinSqlTbl, joinTbl.getElementMapping(), 
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                            }
                            else
                            {
                                // Join to related table
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    sqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(),
                                        relTable, null, relTable.getMemberMapping(relMmd), null, joinTableGroupName);
                                }
                                else
                                {
                                    sqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(),
                                        relTable, null, relTable.getMemberMapping(relMmd), null, joinTableGroupName);
                                }
                            }
                            break;
                        case Relation.ONE_TO_MANY_UNI:
                            relTable = storeMgr.getDatastoreClass(mmd.getCollection().getElementType(), clr);
                            cmd = mmd.getCollection().getElementClassMetaData(clr, mmgr);
                            if (mmd.getJoinMetaData() != null)
                            {
                                // Join to join table, then to related table
                                ElementContainerTable joinTbl = (ElementContainerTable)storeMgr.getDatastoreContainerObject(mmd);
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    SQLTable joinSqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                                    sqlTbl = stmt.innerJoin(joinSqlTbl, joinTbl.getElementMapping(),
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                                else
                                {
                                    SQLTable joinSqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                                    sqlTbl = stmt.leftOuterJoin(joinSqlTbl, joinTbl.getElementMapping(), 
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                            }
                            else
                            {
                                // Join to related table
                                JavaTypeMapping relMapping = relTable.getExternalMapping(mmd, MappingConsumer.MAPPING_TYPE_EXTERNAL_FK);
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    sqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(),
                                        relTable, null, relMapping, null, joinTableGroupName);
                                }
                                else
                                {
                                    sqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(),
                                        relTable, null, relMapping, null, joinTableGroupName);
                                }
                            }
                            break;
                        case Relation.MANY_TO_MANY_BI:
                            relTable = storeMgr.getDatastoreClass(mmd.getCollection().getElementType(), clr);
                            cmd = mmd.getCollection().getElementClassMetaData(clr, mmgr);
                            relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                            // Join to join table, then to related table
                            CollectionTable joinTbl = (CollectionTable)storeMgr.getDatastoreContainerObject(mmd);
                            if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                            {
                                SQLTable joinSqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                    joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                                sqlTbl = stmt.innerJoin(joinSqlTbl, joinTbl.getElementMapping(), 
                                    relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                            }
                            else
                            {
                                SQLTable joinSqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                    joinTbl, null, joinTbl.getOwnerMapping(), null, null);
                                sqlTbl = stmt.leftOuterJoin(joinSqlTbl, joinTbl.getElementMapping(), 
                                    relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                            }
                            break;
                        case Relation.MANY_TO_ONE_BI:
                            relTable = storeMgr.getDatastoreClass(mmd.getTypeName(), clr);
                            cmd = storeMgr.getMetaDataManager().getMetaDataForClass(mmd.getType(), clr);
                            relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                            if (mmd.getJoinMetaData() != null || relMmd.getJoinMetaData() != null)
                            {
                                // Join to join table, then to related table
                                joinTbl = (CollectionTable)storeMgr.getDatastoreContainerObject(relMmd);
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    SQLTable joinSqlTbl = stmt.innerJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        joinTbl, null, joinTbl.getElementMapping(), null, null);
                                    sqlTbl = stmt.innerJoin(joinSqlTbl, joinTbl.getOwnerMapping(), 
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                                else
                                {
                                    SQLTable joinSqlTbl = stmt.leftOuterJoin(sqlTbl, sqlTbl.getTable().getIdMapping(), 
                                        joinTbl, null, joinTbl.getElementMapping(), null, null);
                                    sqlTbl = stmt.leftOuterJoin(joinSqlTbl, joinTbl.getOwnerMapping(), 
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                            }
                            else
                            {
                                // Join to owner table
                                JavaTypeMapping fkMapping = sqlTbl.getTable().getMemberMapping(mmd);
                                if (joinType == JoinType.JOIN_INNER || joinType == JoinType.JOIN_INNER_FETCH)
                                {
                                    sqlTbl = stmt.innerJoin(sqlTbl, fkMapping,
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                                else
                                {
                                    sqlTbl = stmt.leftOuterJoin(sqlTbl, fkMapping,
                                        relTable, null, relTable.getIdMapping(), null, joinTableGroupName);
                                }
                            }
                            break;
                        default:
                            break;
                    }
                }

                SQLTableMapping tblMapping = new SQLTableMapping(sqlTbl, cmd, sqlTbl.getTable().getIdMapping());
                setSQLTableMappingForAlias(joinAlias, tblMapping);
            }

            // Move on to next join in the chain
            rightExpr = rightExpr.getRight();
        }
    }

    /**
     * Convenience method to convert a NewObjectExpression into a StatementNewObjectMapping.
     * Handles recursive new object calls (where a new object is an arg to a new object construction).
     * @param expr The NewObjectExpression
     * @return The mapping for the new object
     */
    protected StatementNewObjectMapping getStatementMappingForNewObjectExpression(NewObjectExpression expr)
    {
        List argExprs = expr.getConstructorArgExpressions();
        StatementNewObjectMapping stmtMap = new StatementNewObjectMapping(expr.getNewClass());
        if (argExprs != null)
        {
            Iterator<SQLExpression> argIter = argExprs.iterator();
            int j = 0;
            while (argIter.hasNext())
            {
                SQLExpression argExpr = argIter.next();
                if (argExpr instanceof SQLLiteral)
                {
                    stmtMap.addConstructorArgMapping(j, ((SQLLiteral)argExpr).getValue());
                }
                else if (argExpr instanceof NewObjectExpression)
                {
                    stmtMap.addConstructorArgMapping(j,
                        getStatementMappingForNewObjectExpression((NewObjectExpression)argExpr));
                }
                else
                {
                    StatementMappingIndex idx = new StatementMappingIndex(argExpr.getJavaTypeMapping());
                    if (argExpr.getNumberOfSubExpressions() == 1)
                    {
                        int col = stmt.select(argExpr, null);
                        idx.setColumnPositions(new int[] {col});
                        stmtMap.addConstructorArgMapping(j, idx);
                    }
                    else
                    {
                        int[] cols = new int[argExpr.getNumberOfSubExpressions()];
                        for (int i=0;i<argExpr.getNumberOfSubExpressions();i++)
                        {
                            int col = stmt.select(argExpr.getSubExpression(i), null);
                            cols[i] = col;
                        }
                        idx.setColumnPositions(cols);
                        stmtMap.addConstructorArgMapping(j, idx);
                    }
                }
                j++;
            }
        }
        return stmtMap;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processAndExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processAndExpression(Expression expr)
    {
        BooleanExpression right = (BooleanExpression)stack.pop();
        BooleanExpression left = (BooleanExpression)stack.pop();
        if (left.getSQLStatement() != null && right.getSQLStatement() != null &&
            left.getSQLStatement() != right.getSQLStatement())
        {
            if (left.getSQLStatement() == stmt &&
                right.getSQLStatement().isChildStatementOf(stmt))
            {
                // Apply right to its sub-statement now and return left
                right.getSQLStatement().whereAnd(right, true);
                stack.push(left);
                return left;
            }
            else if (right.getSQLStatement() == stmt &&
                left.getSQLStatement().isChildStatementOf(stmt))
            {
                // Apply left to its sub-statement now and return right
                left.getSQLStatement().whereAnd(left, true);
                stack.push(right);
                return right;
            }
            // TODO Cater for more situations
        }

        BooleanExpression opExpr = left.and(right);
        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processOrExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processOrExpression(Expression expr)
    {
        BooleanExpression right = (BooleanExpression)stack.pop();
        BooleanExpression left = (BooleanExpression)stack.pop();
        if (left.getSQLStatement() != null && right.getSQLStatement() != null &&
            left.getSQLStatement() != right.getSQLStatement())
        {
            if (left.getSQLStatement() == stmt && right.getSQLStatement().isChildStatementOf(stmt))
            {
                // Apply right to its sub-statement now and return left
                right.getSQLStatement().whereAnd(right, true);
                stack.push(left);
                return left;
            }
            else if (right.getSQLStatement() == stmt && left.getSQLStatement().isChildStatementOf(stmt))
            {
                // Apply left to its sub-statement now and return right
                left.getSQLStatement().whereAnd(left, true);
                stack.push(right);
                return right;
            }
            // TODO Cater for more situations
        }

        BooleanExpression opExpr = left.ior(right);
        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processEqExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processEqExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }

        if (left.isParameter() && right.isParameter())
        {
            if (left.isParameter() && left instanceof SQLLiteral && ((SQLLiteral)left).getValue() != null)
            {
                // Change this parameter to a plain literal
                useParameterExpressionAsLiteral((SQLLiteral)left);
            }
            if (right.isParameter() && right instanceof SQLLiteral && ((SQLLiteral)right).getValue() != null)
            {
                // Change this parameter to a plain literal
                useParameterExpressionAsLiteral((SQLLiteral)right);
            }
        }

        if (left.isParameter())
        {
            updateParameterMapping(left, right.getJavaTypeMapping());
        }
        else if (right.isParameter())
        {
            updateParameterMapping(right, left.getJavaTypeMapping());
        }
        if (left instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression) left);
            left = stack.pop();
        }
        if (right instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression)right);
            right = stack.pop();
        }
        BooleanExpression opExpr = left.eq(right);

        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processNoteqExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processNoteqExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }

        if (left.isParameter() && right.isParameter())
        {
            if (left.isParameter() && left instanceof SQLLiteral && ((SQLLiteral)left).getValue() != null)
            {
                useParameterExpressionAsLiteral((SQLLiteral)left);
            }
            if (right.isParameter() && right instanceof SQLLiteral && ((SQLLiteral)right).getValue() != null)
            {
                useParameterExpressionAsLiteral((SQLLiteral)right);
            }
        }
        if (left.isParameter())
        {
            updateParameterMapping(left, right.getJavaTypeMapping());
        }
        else if (right.isParameter())
        {
            updateParameterMapping(right, left.getJavaTypeMapping());
        }

        if (left instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression) left);
            left = stack.pop();
        }
        if (right instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression)right);
            right = stack.pop();
        }
        BooleanExpression opExpr = left.ne(right);

        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processGteqExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processGteqExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }
        if (left.isParameter())
        {
            updateParameterMapping(left, right.getJavaTypeMapping());
        }
        else if (right.isParameter())
        {
            updateParameterMapping(right, left.getJavaTypeMapping());
        }
        if (left instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression) left);
            left = stack.pop();
        }
        if (right instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression)right);
            right = stack.pop();
        }

        BooleanExpression opExpr = left.ge(right);

        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processGtExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processGtExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }
        if (left.isParameter())
        {
            updateParameterMapping(left, right.getJavaTypeMapping());
        }
        else if (right.isParameter())
        {
            updateParameterMapping(right, left.getJavaTypeMapping());
        }
        if (left instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression) left);
            left = stack.pop();
        }
        if (right instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression)right);
            right = stack.pop();
        }

        BooleanExpression opExpr = left.gt(right);

        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLteqExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processLteqExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }
        if (left.isParameter())
        {
            updateParameterMapping(left, right.getJavaTypeMapping());
        }
        else if (right.isParameter())
        {
            updateParameterMapping(right, left.getJavaTypeMapping());
        }
        if (left instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression) left);
            left = stack.pop();
        }
        if (right instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression)right);
            right = stack.pop();
        }

        BooleanExpression opExpr = left.le(right);

        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLtExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processLtExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }
        if (left.isParameter())
        {
            updateParameterMapping(left, right.getJavaTypeMapping());
        }
        else if (right.isParameter())
        {
            updateParameterMapping(right, left.getJavaTypeMapping());
        }
        if (left instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression) left);
            left = stack.pop();
        }
        if (right instanceof UnboundExpression)
        {
            processUnboundExpression((UnboundExpression)right);
            right = stack.pop();
        }

        BooleanExpression opExpr = left.lt(right);

        stack.push(opExpr);
        return opExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLiteral(org.datanucleus.query.expression.Literal)
     */
    protected Object processLiteral(Literal expr)
    {
        Object litValue = expr.getLiteral();
        if (litValue instanceof Class)
        {
            // Convert Class literals (instanceof) into StringLiteral
            litValue = ((Class)litValue).getName();
        }
        JavaTypeMapping m = null;
        if (litValue != null)
        {
            m = exprFactory.getMappingForType(litValue.getClass(), false);
        }
        SQLExpression sqlExpr = exprFactory.newLiteral(stmt, m, litValue);
        stack.push(sqlExpr);
        return sqlExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processPrimaryExpression(org.datanucleus.query.expression.PrimaryExpression)
     */
    protected Object processPrimaryExpression(PrimaryExpression expr)
    {
        SQLExpression sqlExpr = null;

        if (expr.getLeft() != null)
        {
            if (expr.getLeft() instanceof ParameterExpression)
            {
                // "{paramExpr}.field[.field[.field]]"
                precompilable = false; // Need parameter values to process this
                ParameterExpression paramExpr = (ParameterExpression)expr.getLeft();

                Symbol paramSym = compilation.getSymbolTable().getSymbol(paramExpr.getId());
                if (paramSym.getValueType().isArray())
                {
                    // Special case : array "methods" (particularly "length")
                    String first = expr.getTuples().get(0);
                    processParameterExpression(paramExpr, true);
                    SQLExpression paramSqlExpr = stack.pop();
                    sqlExpr = exprFactory.invokeMethod(stmt, "ARRAY", first, paramSqlExpr, null);
                    stack.push(sqlExpr);
                    return sqlExpr;
                }
                else
                {
                    // Create Literal for the parameter (since we need to perform operations on it)
                    processParameterExpression(paramExpr, true);
                    SQLExpression paramSqlExpr = stack.pop();
                    SQLLiteral lit = (SQLLiteral)paramSqlExpr;
                    Object paramValue = lit.getValue();

                    List<String> tuples = expr.getTuples();
                    Iterator<String> tuplesIter = tuples.iterator();
                    Object objValue = paramValue;
                    while (tuplesIter.hasNext())
                    {
                        String fieldName = tuplesIter.next();
                        objValue = getValueForObjectField(objValue, fieldName);
                        if (objValue == null)
                        {
                            break;
                        }
                    }

                    if (objValue == null)
                    {
                        sqlExpr = exprFactory.newLiteral(stmt, null, null);
                        stack.push(sqlExpr);
                        return sqlExpr;
                    }
                    else
                    {
                        JavaTypeMapping m = exprFactory.getMappingForType(objValue.getClass(), false);
                        sqlExpr = exprFactory.newLiteral(stmt, m, objValue);
                        stack.push(sqlExpr);
                        return sqlExpr;
                    }
                }
            }
            else if (expr.getLeft() instanceof VariableExpression)
            {
                // "{varExpr}.field[.field[.field]]"
                VariableExpression varExpr = (VariableExpression)expr.getLeft();
                processVariableExpression(varExpr);
                SQLExpression varSqlExpr = stack.pop();
                NucleusLogger.QUERY.debug(">> QueryToSQL.processPrimary left(var)=" + expr.getLeft() + 
                    " id=" + expr.getId() + " varSqlExpr=" + varSqlExpr);
                if (varSqlExpr instanceof UnboundExpression)
                {
                    processUnboundExpression((UnboundExpression)varSqlExpr);
                    varSqlExpr = stack.pop();
                    NucleusLogger.QUERY.debug(">> QueryToSQL.processPrimary variable is still unbound, so binding via cross-join varSqlExpr=" + varSqlExpr);
                }
                NucleusLogger.QUERY.debug(">> QueryToSQL.processPrimary varType=" + varSqlExpr.getJavaTypeMapping().getType() + 
                    " varExpr.tbl=" + varSqlExpr.getSQLTable() +
                    " stmt=" + stmt +
                    " varExpr.stmt=" + varSqlExpr.getSQLStatement());

                Class varType = clr.classForName(varSqlExpr.getJavaTypeMapping().getType());
                if (varSqlExpr.getSQLStatement() == stmt.getParentStatement())
                {
                    // Use parent mapper to get the mapping for this field since it has the table
                    SQLTableMapping sqlMapping =
                        parentMapper.getSQLTableMappingForPrimaryExpression(stmt, null, expr.getTuples(), false);
                    if (sqlMapping == null)
                    {
                        throw new NucleusException("PrimaryExpression " + expr.getId() + " is not yet supported");
                    }
                    // TODO Cater for the table required to join to not being the primary table of the outer query
                    // This should check on
                    // getDatastoreAdapter().supportsOption(RDBMSAdapter.ACCESS_PARENTQUERY_IN_SUBQUERY))

                    sqlExpr = exprFactory.newExpression(varSqlExpr.getSQLStatement(), 
                        sqlMapping.table, sqlMapping.mapping);
                    stack.push(sqlExpr);
                    return sqlExpr;
                }

                SQLTableMapping varTblMapping = getSQLTableMappingForAlias(varExpr.getId());
                if (varTblMapping == null)
                {
                    throw new NucleusUserException("Variable " + varExpr.getId() + " is not yet bound, so cannot get field " + expr.getId());
                }
                if (varTblMapping.cmd == null)
                {
                    throw new NucleusUserException("Variable " + varExpr.getId() + " of type " + varType.getName() + " cannot evaluate " + expr.getId());
                }

                SQLTableMapping sqlMapping =
                    getSQLTableMappingForPrimaryExpression(varSqlExpr.getSQLStatement(), varExpr.getId(), 
                        expr.getTuples(), false);
                NucleusLogger.QUERY.debug(">> QueryToSQL.processPrimary var=" + varExpr.getId() +
                    " tuples=" + StringUtils.collectionToString(expr.getTuples()) +
                    " sqlMapping=" + sqlMapping);

                sqlExpr = exprFactory.newExpression(sqlMapping.table.getSQLStatement(), sqlMapping.table,
                    sqlMapping.mapping);
                stack.push(sqlExpr);
                return sqlExpr;
            }
            else if (expr.getLeft() instanceof CastExpression)
            {
                // "(castExpr)field[.field[.field]]" where castExpr = "(castType)primExpr"
                CastExpression castExpr = (CastExpression)expr.getLeft();

                // Evaluate the expression being cast
                PrimaryExpression castLeftExpr = (PrimaryExpression)castExpr.getLeft();
                processPrimaryExpressionInternal(castLeftExpr, true); // Force any join since we need further operations
                SQLExpression castLeftSqlExpr = stack.pop();

                // Cast the expression
                String castClassName = castExpr.getClassName();
                SQLExpression castClassNameExpr = exprFactory.newLiteral(stmt, 
                    exprFactory.getMappingForType(String.class, false), castClassName);
                sqlExpr = castLeftSqlExpr.cast(castClassNameExpr);

                // Add the cast expression to the table mapping
                String exprPrimName = "CAST_" + castLeftExpr.getId();
                AbstractClassMetaData castCmd = om.getMetaDataManager().getMetaDataForClass(castClassName, clr);
                SQLTableMapping tblMapping = new SQLTableMapping(sqlExpr.getSQLTable(), castCmd, sqlExpr.getJavaTypeMapping());
                setSQLTableMappingForAlias(exprPrimName, tblMapping);

                // Evaluate the primary of this cast expression
                SQLTableMapping sqlMapping =
                    getSQLTableMappingForPrimaryExpression(stmt, exprPrimName, expr.getTuples(), false);
                if (sqlMapping == null)
                {
                    throw new NucleusException("PrimaryExpression " + expr + " is not yet supported");
                }
                sqlExpr = exprFactory.newExpression(stmt, sqlMapping.table, sqlMapping.mapping);
                stack.push(sqlExpr);
                return sqlExpr;
            }
            else if (expr.getLeft() instanceof InvokeExpression)
            {
                processInvokeExpression((InvokeExpression)expr.getLeft());
                SQLExpression invokeSqlExpr = stack.pop();
                DatastoreContainerObject tbl = invokeSqlExpr.getSQLTable().getTable();
                if (tbl instanceof DatastoreClass)
                {
                    // Table of a class, so assume to have field in the table of the class
                    // TODO Allow joins to superclasses if required
                    if (expr.getTuples().size() > 1)
                    {
                        throw new NucleusUserException("Dont currently support evaluating " + expr.getId() +
                            " on " + invokeSqlExpr);
                    }
                    JavaTypeMapping mapping = ((DatastoreClass)tbl).getMemberMapping(expr.getId());
                    if (mapping == null)
                    {
                        throw new NucleusUserException("Dont currently support evaluating " + expr.getId() +
                            " on " + invokeSqlExpr +
                            ". The field " + expr.getId() + " doesnt exist in table " + tbl);
                    }

                    sqlExpr = exprFactory.newExpression(stmt, invokeSqlExpr.getSQLTable(), mapping);
                    stack.push(sqlExpr);
                    return sqlExpr;
                }
                else
                {
                    // Join table!
                    throw new NucleusUserException("Dont currently support evaluating " + expr.getId() +
                        " on " + invokeSqlExpr +
                        " with invoke having table of " + tbl);
                }
            }
            else
            {
                throw new NucleusUserException("Dont currently support PrimaryExpression with 'left' of " + expr.getLeft());
            }
        }

        // Real primary expression ("field.field", "alias.field.field" etc)
        return processPrimaryExpressionInternal(expr, false);
    }

    private SQLExpression processPrimaryExpressionInternal(PrimaryExpression primExpr, boolean forceJoin)
    {
        // Real primary expression ("field.field", "alias.field.field" etc)
        SQLTableMapping sqlMapping =
            getSQLTableMappingForPrimaryExpression(stmt, null, primExpr.getTuples(), forceJoin);
        if (sqlMapping == null)
        {
            throw new NucleusException("PrimaryExpression " + primExpr.getId() + " is not yet supported");
        }
        SQLExpression sqlExpr = exprFactory.newExpression(stmt, sqlMapping.table, sqlMapping.mapping);
        stack.push(sqlExpr);
        return sqlExpr;
    }

    /**
     * Method to take in a PrimaryExpression and return the SQLTable mapping info that it signifies.
     * If the primary expression implies joining to other objects then adds the joins to the statement.
     * Only adds joins if necessary; so if there is a further component after the required join, or if
     * the "forceJoin" flag is set.
     * @param theStmt SQLStatement to use when looking for tables etc
     * @param exprName Name for an expression that this primary is relative to (optional)
     *                 If not specified then the tuples are relative to the candidate.
     *                 If specified then should have an entry in sqlTableByPrimary under this name.
     * @param tuples Tuples of the primary expression
     * @param forceJoin Whether to force a join if a relation member
     * @return The SQL table mapping information for the specified primary
     */
    private SQLTableMapping getSQLTableMappingForPrimaryExpression(SQLStatement theStmt, String exprName,
            List<String> tuples, boolean forceJoin)
    {
        SQLTableMapping sqlMapping = null;

        // Find source object
        Iterator<String> iter = tuples.iterator();
        String first = tuples.get(0);
        String primaryName = null;
        if (exprName != null)
        {
            // Primary relative to some object etc
            sqlMapping = getSQLTableMappingForAlias(exprName);
            primaryName = exprName;
        }
        else
        {
            if (hasSQLTableMappingForAlias(first))
            {
                // Start from a candidate (e.g JPQL alias)
                sqlMapping = getSQLTableMappingForAlias(first);
                primaryName = first;
                iter.next(); // Skip first tuple
            }

            if (sqlMapping == null)
            {
                if (parentMapper != null && parentMapper.hasSQLTableMappingForAlias(first))
                {
                    // Try parent query
                    sqlMapping = parentMapper.getSQLTableMappingForAlias(first);
                    primaryName = first;
                    iter.next(); // Skip first tuple
                }
            }
            if (sqlMapping == null)
            {
                // Field of candidate, so use candidate
                sqlMapping = getSQLTableMappingForAlias(candidateAlias);
                primaryName = candidateAlias;
            }
        }

        AbstractClassMetaData cmd = sqlMapping.cmd;
        JavaTypeMapping mapping = null;
        if (sqlMapping != null)
        {
            mapping = sqlMapping.mapping;
        }

        while (iter.hasNext())
        {
            String component = iter.next();
            primaryName += "." + component; // fully-qualified primary name

            // Derive SQLTableMapping for this component
            SQLTableMapping sqlMappingNew = getSQLTableMappingForAlias(primaryName);
            if (sqlMappingNew == null)
            {
                // Table not present for this primary
                AbstractMemberMetaData mmd = cmd.getMetaDataForMember(component);
                if (mmd == null)
                {
                    // Not valid member name
                    throw new NucleusUserException("Query contains access of " + primaryName + " yet this field/property doesnt exist");
                }

                // Find the table and the mapping for this field in the table
                SQLTable sqlTbl = null;
                if (mapping instanceof EmbeddedMapping)
                {
                    // Embedded into the current table
                    sqlTbl = sqlMapping.table;
                    mapping = ((EmbeddedMapping)mapping).getJavaTypeMapping(component);
                }
                else
                {
                    DatastoreClass table = storeMgr.getDatastoreClass(cmd.getFullClassName(), clr);
                    mapping = table.getMemberMapping(mmd);
                    sqlTbl = SQLStatementHelper.getSQLTableForMappingOfTable(theStmt, sqlMapping.table, mapping);
                }

                int relationType = mmd.getRelationType(clr);
                switch (relationType)
                {
                    case Relation.NONE :
                        sqlMappingNew = new SQLTableMapping(sqlTbl, cmd, mapping);
                        cmd = sqlMappingNew.cmd;
                        setSQLTableMappingForAlias(primaryName, sqlMappingNew);
                        break;

                    case Relation.ONE_TO_ONE_UNI :
                    case Relation.ONE_TO_ONE_BI :
                    case Relation.MANY_TO_ONE_BI :
                        if (mmd.getMappedBy() != null)
                        {
                            // FK in other table so join to that first
                            AbstractMemberMetaData relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                            if (relMmd.getAbstractClassMetaData().isEmbeddedOnly())
                            {
                                // Member is embedded, so keep same SQL table mapping
                                sqlMappingNew = sqlMapping;
                                cmd = relMmd.getAbstractClassMetaData();
                            }
                            else
                            {
                                // Member is in own table, so move to that SQL table mapping
                                DatastoreClass relTable = storeMgr.getDatastoreClass(mmd.getTypeName(), clr);
                                JavaTypeMapping relMapping = relTable.getMemberMapping(relMmd);
                                // Join to related table unless we already have the join in place
                                sqlTbl = theStmt.getTable(relTable, primaryName);
                                if (sqlTbl == null)
                                {
                                    sqlTbl = SQLStatementHelper.addJoinForOneToOneRelation(theStmt, 
                                        sqlMapping.table.getTable().getIdMapping(), sqlMapping.table,
                                        relMapping, relTable, null, null, primaryName, defaultJoinType);
                                }

                                if (iter.hasNext())
                                {
                                    sqlMappingNew = new SQLTableMapping(sqlTbl, relMmd.getAbstractClassMetaData(),
                                        relTable.getIdMapping());
                                    cmd = sqlMappingNew.cmd;
                                }
                                else
                                {
                                    sqlMappingNew = new SQLTableMapping(sqlTbl, cmd, relTable.getIdMapping());
                                    cmd = sqlMappingNew.cmd;
                                }
                            }
                        }
                        else
                        {
                            // FK is at this side, so only join if further component provided, or if forcing
                            if (iter.hasNext() || forceJoin)
                            {
                                AbstractClassMetaData relCmd = null;
                                JavaTypeMapping relMapping = null;
                                DatastoreClass relTable = null;
                                if (relationType == Relation.ONE_TO_ONE_BI)
                                {
                                    AbstractMemberMetaData relMmd = mmd.getRelatedMemberMetaData(clr)[0];
                                    relCmd = relMmd.getAbstractClassMetaData();
                                }
                                else
                                {
                                    relCmd = om.getMetaDataManager().getMetaDataForClass(mmd.getTypeName(), clr);
                                }
                                if (relCmd.isEmbeddedOnly())
                                {
                                    // Member is embedded so keep same SQL table mapping
                                    sqlMappingNew = sqlMapping;
                                    cmd = relCmd;
                                }
                                else
                                {
                                    // Member is in own table, so move to that SQL table mapping
                                    relTable = storeMgr.getDatastoreClass(relCmd.getFullClassName(), clr);
                                    relMapping = relTable.getIdMapping();

                                    // Join to other table unless we already have the join in place
                                    sqlTbl = theStmt.getTable(relTable, primaryName);
                                    if (sqlTbl == null)
                                    {
                                        sqlTbl = SQLStatementHelper.addJoinForOneToOneRelation(theStmt, mapping, 
                                            sqlMapping.table, relMapping, relTable, null, null, primaryName, 
                                            defaultJoinType);
                                    }

                                    sqlMappingNew = new SQLTableMapping(sqlTbl, relCmd, relMapping);
                                    cmd = sqlMappingNew.cmd;
                                    setSQLTableMappingForAlias(primaryName, sqlMappingNew);
                                }
                            }
                            else
                            {
                                sqlMappingNew = new SQLTableMapping(sqlTbl, cmd, mapping);
                                cmd = sqlMappingNew.cmd;
                                // Don't register the SQLTableMapping for this alias since only using FK
                            }
                        }
                        break;

                    case Relation.ONE_TO_MANY_UNI :
                    case Relation.ONE_TO_MANY_BI :
                    case Relation.MANY_TO_MANY_BI :
                        // Can't reference further than a collection/map so just return its mapping here
                        sqlMappingNew = new SQLTableMapping(sqlTbl, cmd, mapping);
                        cmd = sqlMappingNew.cmd;
                        setSQLTableMappingForAlias(primaryName, sqlMappingNew);
                        break;

                    default :
                        break;
                }
            }
            else
            {
                cmd = sqlMappingNew.cmd;
            }

            sqlMapping = sqlMappingNew;
        }

        return sqlMapping;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processParameterExpression(org.datanucleus.query.expression.ParameterExpression)
     */
    @Override
    protected Object processParameterExpression(ParameterExpression expr)
    {
        return processParameterExpression(expr, false);
    }

    /**
     * Method to process a parameter expression. The optional argument controls whether we should
     * create this as a parameter or as a literal (i.e the param value is known etc).
     * If the parameter doesn't have its value defined then returns ParameterLiteral
     * otherwise we get an XXXLiteral of the (declared) type of the parameter
     * @param expr The ParameterExpression
     * @param asLiteral Whether to create a SQLLiteral rather than a parameter literal
     */
    protected Object processParameterExpression(ParameterExpression expr, boolean asLiteral)
    {
        // Find the parameter value if supplied
        Object paramValue = null;
        if (parameters != null && parameters.size() > 0)
        {
            // Check if the parameter has a value
            if (parameters.containsKey(expr.getId()))
            {
                // Named parameter
                paramValue = parameters.get(expr.getId());
            }
            else
            {
                int position = positionalParamNumber;
                if (positionalParamNumber < 0)
                {
                    position = 0;
                }
                if (parameters.containsKey(new Integer(position)))
                {
                    paramValue = parameters.get(new Integer(position));
                    positionalParamNumber = position+1;
                }
            }
        }

        // Find the type to use for the parameter
        JavaTypeMapping m = paramMappingForName.get(expr.getId());
        if (m == null)
        {
            // Try to determine from provided parameter value or from symbol table (declared type)
            if (paramValue != null)
            {
                // Use the type of the input parameter value
                m = exprFactory.getMappingForType(paramValue.getClass(), false);
                if (expr.getSymbol() != null && expr.getSymbol().getValueType() != null)
                {
                    if (!QueryUtils.queryParameterTypesAreCompatible(expr.getSymbol().getValueType(), paramValue.getClass()))
                    {
                        throw new QueryCompilerSyntaxException("Supplied parameter " + expr.getId() +
                            " is declared as " + expr.getSymbol().getValueType().getName() +
                            " yet a value of type " + paramValue.getClass().getName() + " was supplied");
                    }
                    if (expr.getSymbol().getValueType() != paramValue.getClass())
                    {
                        // Mark as not precompilable since the supplied type implies a subclass of the declared type
                        precompilable = false;
                    }
                }
            }
            else if (expr.getSymbol() != null && expr.getSymbol().getValueType() != null)
            {
                // Use the declared type of the parameter (explicit params)
                m = exprFactory.getMappingForType(expr.getSymbol().getValueType(), false);
            }
        }

        if (paramValue == null && expr.getSymbol() != null)
        {
            precompilable = false;
        }

        SQLExpression sqlExpr = null;
        if (asLiteral && paramValue != null)
        {
            // We have the value and user wants a normal SQLLiteral so return as that
            sqlExpr = exprFactory.newLiteral(stmt, m, paramValue);
        }
        else
        {
            // Create a "parameter" Literal
            sqlExpr = exprFactory.newLiteralParameter(stmt, m, paramValue, expr.getId());
            if (sqlExpr instanceof ParameterLiteral)
            {
                ((ParameterLiteral)sqlExpr).setName(expr.getId());
            }

            if (expressionForParameter == null)
            {
                expressionForParameter = new HashMap<Object, SQLExpression>();
            }
            expressionForParameter.put(expr.getId(), sqlExpr);

            paramMappingForName.put(expr.getId(), m);
        }

        stack.push(sqlExpr);
        return sqlExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processInvokeExpression(org.datanucleus.query.expression.InvokeExpression)
     */
    protected Object processInvokeExpression(InvokeExpression expr)
    {
        // Find object that we invoke on
        Expression invokedExpr = expr.getLeft();
        SQLExpression invokedSqlExpr = null;
        if (invokedExpr == null)
        {
            // Static method
        }
        else if (invokedExpr instanceof PrimaryExpression)
        {
            processPrimaryExpression((PrimaryExpression)invokedExpr);
            invokedSqlExpr = stack.pop();
        }
        else if (invokedExpr instanceof Literal)
        {
            processLiteral((Literal)invokedExpr);
            invokedSqlExpr = stack.pop();
        }
        else if (invokedExpr instanceof ParameterExpression)
        {
            // TODO This next line should depend on what method is being invoked on the param
            precompilable = false; // Need parameter value to be able to invoke on it
            processParameterExpression((ParameterExpression)invokedExpr, true);
            invokedSqlExpr = stack.pop();
        }
        else if (invokedExpr instanceof InvokeExpression)
        {
            processInvokeExpression((InvokeExpression)invokedExpr);
            invokedSqlExpr = stack.pop();
        }
        else if (invokedExpr instanceof VariableExpression)
        {
            processVariableExpression((VariableExpression)invokedExpr);
            invokedSqlExpr = stack.pop();
        }
        else
        {
            throw new NucleusException("Dont currently support invoke expression " + invokedExpr);
        }

        String operation = expr.getOperation();
        List args = expr.getArguments();
        List sqlExprArgs = null;
        if (args != null)
        {
            sqlExprArgs = new ArrayList<SQLExpression>();
            Iterator<Expression> iter = args.iterator();
            while (iter.hasNext())
            {
                Expression argExpr = iter.next();
                if (argExpr instanceof PrimaryExpression)
                {
                    processPrimaryExpression((PrimaryExpression)argExpr);
                    sqlExprArgs.add(stack.pop());
                }
                else if (argExpr instanceof ParameterExpression)
                {
                    processParameterExpression((ParameterExpression)argExpr);
                    sqlExprArgs.add(stack.pop());
                }
                else if (argExpr instanceof InvokeExpression)
                {
                    processInvokeExpression((InvokeExpression)argExpr);
                    sqlExprArgs.add(stack.pop());
                }
                else if (argExpr instanceof Literal)
                {
                    processLiteral((Literal)argExpr);
                    sqlExprArgs.add(stack.pop());
                }
                else if (argExpr instanceof DyadicExpression)
                {
                    // Evaluate using this evaluator
                    argExpr.evaluate(this);
                    sqlExprArgs.add(stack.pop());
                }
                else if (argExpr instanceof VariableExpression)
                {
                    processVariableExpression((VariableExpression)argExpr);
                    sqlExprArgs.add(stack.pop());
                }
                else
                {
                    throw new NucleusException("Dont currently support invoke expression argument " + argExpr);
                }
            }
        }

        // Invoke the method
        SQLExpression sqlExpr = null;
        if (invokedSqlExpr != null)
        {
            sqlExpr = invokedSqlExpr.invoke(operation, sqlExprArgs);
        }
        else
        {
            sqlExpr = exprFactory.invokeMethod(stmt, null, operation, null, sqlExprArgs);
        }

        stack.push(sqlExpr);
        return sqlExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processSubqueryExpression(org.datanucleus.query.expression.SubqueryExpression)
     */
    @Override
    protected Object processSubqueryExpression(SubqueryExpression expr)
    {
        String keyword = expr.getKeyword();
        Expression subqueryExpr = expr.getRight();
        if (subqueryExpr instanceof VariableExpression)
        {
            processVariableExpression((VariableExpression)subqueryExpr);
            SQLExpression subquerySqlExpr = stack.pop();
            NucleusLogger.GENERAL.info(">> QueryToSQL.processSubquery " + expr +
                " sql=" + subquerySqlExpr + " keyword=" + keyword);
            if (keyword != null && keyword.equals("EXISTS"))
            {
                // EXISTS expressions need to be Boolean
                if (subquerySqlExpr instanceof org.datanucleus.store.rdbms.sql.expression.SubqueryExpression)
                {
                    SQLStatement subStmt = 
                        ((org.datanucleus.store.rdbms.sql.expression.SubqueryExpression)subquerySqlExpr).getSubqueryStatement();
                    subquerySqlExpr = new BooleanSubqueryExpression(stmt, keyword, subStmt);
                }
                else
                {
                    SQLStatement subStmt = ((SubqueryExpressionComponent)subquerySqlExpr).getSubqueryStatement();
                    subquerySqlExpr = new BooleanSubqueryExpression(stmt, keyword, subStmt);
                }
            }
            else if (subquerySqlExpr instanceof org.datanucleus.store.rdbms.sql.expression.SubqueryExpression)
            {
                SQLStatement subStmt = 
                    ((org.datanucleus.store.rdbms.sql.expression.SubqueryExpression)subquerySqlExpr).getSubqueryStatement();
                subquerySqlExpr = new BooleanSubqueryExpression(stmt, keyword, subStmt);
            }
            else if (subquerySqlExpr instanceof NumericSubqueryExpression)
            {
                // Apply keyword (e.g ALL, SOME, ANY) to numeric expressions
                ((NumericSubqueryExpression)subquerySqlExpr).setKeyword(keyword);
            }
            stack.push(subquerySqlExpr);
            return subquerySqlExpr;
        }
        else
        {
            throw new NucleusException("Dont currently support SubqueryExpression " + keyword +
                " for type " + subqueryExpr);
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processAddExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processAddExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }

        SQLExpression resultExpr = left.add(right);
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processDivExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processDivExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression resultExpr = left.div(right);
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processMulExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processMulExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression resultExpr = left.mul(right);
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processSubExpression(org.datanucleus.query.expression.Expression)
     */
    protected Object processSubExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        if (left instanceof ParameterLiteral && !(right instanceof ParameterLiteral))
        {
            left = replaceParameterLiteral((ParameterLiteral)left, right.getJavaTypeMapping());
        }
        else if (right instanceof ParameterLiteral && !(left instanceof ParameterLiteral))
        {
            right = replaceParameterLiteral((ParameterLiteral)right, left.getJavaTypeMapping());
        }

        SQLExpression resultExpr = left.sub(right);
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processDistinctExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processDistinctExpression(Expression expr)
    {
        SQLExpression sqlExpr = stack.pop();
        sqlExpr.distinct();
        stack.push(sqlExpr);
        return sqlExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processComExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processComExpression(Expression expr)
    {
        // Bitwise complement - only for integer values
        SQLExpression sqlExpr = stack.pop();
        SQLExpression resultExpr = sqlExpr.com();
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processModExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processModExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression resultExpr = left.mod(right);
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processNegExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processNegExpression(Expression expr)
    {
        SQLExpression sqlExpr = stack.pop();
        SQLExpression resultExpr = sqlExpr.neg();
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processNotExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processNotExpression(Expression expr)
    {
        // Logical complement - only for boolean values
        SQLExpression sqlExpr = stack.pop();
        SQLExpression resultExpr = sqlExpr.not();
        stack.push(resultExpr);
        return resultExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processCastExpression(org.datanucleus.query.expression.CastExpression)
     */
    @Override
    protected Object processCastExpression(CastExpression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression instanceofExpr = left.cast(right);
        stack.push(instanceofExpr);
        return instanceofExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processIsExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processIsExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression instanceofExpr = left.is(right, false);
        stack.push(instanceofExpr);
        return instanceofExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processIsnotExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processIsnotExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression instanceofExpr = left.is(right, true);
        stack.push(instanceofExpr);
        return instanceofExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processInExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processInExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        SQLExpression instanceofExpr = left.in(right);
        stack.push(instanceofExpr);
        return instanceofExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processCreatorExpression(org.datanucleus.query.expression.CreatorExpression)
     */
    @Override
    protected Object processCreatorExpression(CreatorExpression expr)
    {
        String className = expr.getId();
        Class cls = clr.classForName(className);
        List<SQLExpression> ctrArgExprs = null;
        List args = expr.getArguments();
        if (args != null)
        {
            ctrArgExprs = new ArrayList<SQLExpression>(args.size());
            Iterator iter = args.iterator();
            while (iter.hasNext())
            {
                Expression argExpr = (Expression)iter.next();
                SQLExpression sqlExpr = (SQLExpression)evaluate(argExpr);
                ctrArgExprs.add(sqlExpr);
            }
        }

        NewObjectExpression newExpr = new NewObjectExpression(stmt, cls, ctrArgExprs);
        stack.push(newExpr);
        return newExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processLikeExpression(org.datanucleus.query.expression.Expression)
     */
    @Override
    protected Object processLikeExpression(Expression expr)
    {
        SQLExpression right = stack.pop();
        SQLExpression left = stack.pop();
        List args = new ArrayList();
        args.add(right);

        SQLExpression likeExpr = exprFactory.invokeMethod(stmt, String.class.getName(), "like", left, args);
        stack.push(likeExpr);
        return likeExpr;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.query.evaluator.AbstractExpressionEvaluator#processVariableExpression(org.datanucleus.query.expression.VariableExpression)
     */
    @Override
    protected Object processVariableExpression(VariableExpression expr)
    {
        Symbol varSym = expr.getSymbol();
        String varName = varSym.getQualifiedName();
        NucleusLogger.QUERY.debug(">> QueryToSQL.processVariable expr=" + expr + " var=" + varName);
        if (hasSQLTableMappingForAlias(varName))
        {
            // Variable already found
            SQLTableMapping tblMapping = getSQLTableMappingForAlias(varName);
            NucleusLogger.QUERY.debug(">> QueryToSQL.processVariable (existing) var=" + varName +
                " tableMapping=" + tblMapping + " stmt=" + tblMapping.table.getSQLStatement());
            SQLExpression sqlExpr = exprFactory.newExpression(tblMapping.table.getSQLStatement(), 
                tblMapping.table, tblMapping.mapping);
            stack.push(sqlExpr);
            return sqlExpr;
        }
        else if (compilation.getCompilationForSubquery(varName) != null)
        {
            // Subquery variable
            QueryCompilation subCompilation = compilation.getCompilationForSubquery(varName);
            AbstractClassMetaData subCmd = om.getMetaDataManager().getMetaDataForClass(subCompilation.getCandidateClass(), om.getClassLoaderResolver());

            // Create subquery statement, using any provided alias if possible
            String subAlias = null;
            if (subCompilation.getCandidateAlias() != null && !subCompilation.getCandidateAlias().equals(candidateAlias))
            {
                subAlias = subCompilation.getCandidateAlias();
            }
            StatementResultMapping subqueryResultMapping = new StatementResultMapping();
            // TODO Fix "avg(something)" arg - not essential but is a hack right now
            SQLStatement subStmt = RDBMSQueryUtils.getStatementForCandidates(stmt, subCmd,
                null, om, subCompilation.getCandidateClass(), false, "avg(something)", subAlias, null);
            QueryToSQLMapper sqlMapper = new QueryToSQLMapper(subStmt, subCompilation, parameters,
                null, subqueryResultMapping, subCmd, fetchPlan, om);
            sqlMapper.setParentMapper(this);
            sqlMapper.compile();
            if (subqueryResultMapping.getNumberOfResultExpressions() > 1)
            {
                throw new NucleusUserException("Number of result expressions in subquery should be 1");
            }

            SQLExpression subExpr = null;
            NucleusLogger.QUERY.debug(">> QueryToSQL.processVariable (subquery) variable=" + varName +
                " subqueryResult=" + subqueryResultMapping);
            // TODO Cater for subquery select of its own candidate
            if (subqueryResultMapping.getNumberOfResultExpressions() == 0)
            {
                subExpr = new org.datanucleus.store.rdbms.sql.expression.SubqueryExpression(stmt, subStmt);
            }
            else
            {
                JavaTypeMapping subMapping = 
                    ((StatementMappingIndex)subqueryResultMapping.getMappingForResultExpression(0)).getMapping();
                if (subMapping instanceof TemporalMapping)
                {
                    subExpr = new TemporalSubqueryExpression(stmt, subStmt);
                }
                else if (subMapping instanceof StringMapping)
                {
                    subExpr = new StringSubqueryExpression(stmt, subStmt);
                }
                else
                {
                    subExpr = new NumericSubqueryExpression(stmt, subStmt);
                }
            }
            stack.push(subExpr);
            return subExpr;
        }
        else if (stmt.getParentStatement() != null && parentMapper != null && 
                parentMapper.candidateAlias != null && parentMapper.candidateAlias.equals(varName))
        {
            // Variable in subquery linking back to parent query
            SQLExpression varExpr = exprFactory.newExpression(stmt.getParentStatement(),
                stmt.getParentStatement().getPrimaryTable(),
                stmt.getParentStatement().getPrimaryTable().getTable().getIdMapping());
            stack.push(varExpr);
            return varExpr;
        }
        else
        {
            // Variable never met before, so return as UnboundExpression - process later if needing binding
            NucleusLogger.QUERY.debug(">> QueryToSQL.processVariable (unbound) variable=" + varName +
                " is not yet bound so returning UnboundExpression");
            UnboundExpression unbExpr = new UnboundExpression(stmt, varName);
            stack.push(unbExpr);
            return unbExpr;
        }
    }

    protected SQLExpression processUnboundExpression(UnboundExpression expr)
    {
        String varName = expr.getVariableName();
        Symbol varSym = compilation.getSymbolTable().getSymbol(varName);
        AbstractClassMetaData cmd = om.getMetaDataManager().getMetaDataForClass(varSym.getValueType(), clr);
        if (cmd != null)
        {
            // Variable is persistent type, so add cross join (may need changing later on in compilation)
            NucleusLogger.QUERY.debug(">> QueryToSQL.processVariable var=" + varName + " yet unbound, so adding as cross-join");
            DatastoreClass varTable = storeMgr.getDatastoreClass(varSym.getValueType().getName(), clr);
            SQLTable varSqlTbl = stmt.crossJoin(varTable, "VAR_" + varName, null);
            SQLTableMapping varSqlTblMapping = new SQLTableMapping(varSqlTbl, cmd, varTable.getIdMapping());
            setSQLTableMappingForAlias(varName, varSqlTblMapping);
            SQLExpression sqlExpr = exprFactory.newExpression(stmt, varSqlTbl, varTable.getIdMapping());
            stack.push(sqlExpr);
            return sqlExpr;
        }
        throw new NucleusUserException("Variable " + varName + " is unbound and cannot be determined");
    }

    /**
     * Convenience method to update the mapping used by a parameter literal. This is typically called
     * when we have an expression like "field == :param" and we want the field and parameter to use the same
     * datastore mapping type (with a java mapping we can use different types of datastore mappings and
     * so could have overridden the default for the java type).
     * @param paramExpr The parameter
     * @param mapping The mapping to use
     */
    protected void updateParameterMapping(SQLExpression paramExpr, JavaTypeMapping mapping)
    {
        if (mapping == null)
        {
            return;
        }

        boolean needsUpdating = false;
        JavaTypeMapping paramMapping = paramExpr.getJavaTypeMapping();
        if (paramMapping != null && paramMapping.getNumberOfDatastoreMappings() != mapping.getNumberOfDatastoreMappings())
        {
            needsUpdating = true;
        }
        else
        {
            for (int i=0;i<paramMapping.getNumberOfDatastoreMappings();i++)
            {
                DatastoreMapping colMapping = paramMapping.getDatastoreMapping(i);
                if (colMapping == null || colMapping.getClass() != mapping.getDatastoreMapping(i).getClass())
                {
                    needsUpdating = true;
                    break;
                }
            }
        }

        if (needsUpdating)
        {
            // Update the mapping in the expression, and also in the parameterDefinition
            paramExpr.setJavaTypeMapping(mapping);
        }
    }

    /**
     * Convenience method to return a parameter-based literal using the supplied mapping to replace
     * the provided ParameterLiteral (generated before its type was known).
     * @param paramLit The parameter literal
     * @param mapping Mapping to use
     * @return The replacement expression
     */
    protected SQLExpression replaceParameterLiteral(ParameterLiteral paramLit, JavaTypeMapping mapping)
    {
        return exprFactory.newLiteralParameter(stmt, mapping, paramLit.getValue(),
            paramLit.getParameterName());
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.query.QueryGenerator#useParameterExpressionAsLiteral(org.datanucleus.store.rdbms.sql.expression.SQLLiteral)
     */
    public void useParameterExpressionAsLiteral(SQLLiteral paramLiteral)
    {
        paramLiteral.setNotParameter();
        precompilable = false;
    }

    /**
     * Convenience method to return the value of a field of the supplied object.
     * If the object is null then returns null for the field.
     * @param obj The object
     * @param fieldName The field name
     * @return The field value
     */
    protected Object getValueForObjectField(Object obj, String fieldName)
    {
        if (obj != null)
        {
            Object paramFieldValue = null;
            if (om.getApiAdapter().isPersistable(obj))
            {
                StateManager paramSM = om.findStateManager(obj);
                AbstractClassMetaData paramCmd = om.getMetaDataManager().getMetaDataForClass(obj.getClass(), clr);
                AbstractMemberMetaData paramFieldMmd = paramCmd.getMetaDataForMember(fieldName);
                if (paramSM != null)
                {
                    om.getApiAdapter().isLoaded(paramSM, paramFieldMmd.getAbsoluteFieldNumber());
                    paramFieldValue = paramSM.provideField(paramFieldMmd.getAbsoluteFieldNumber());
                }
                else
                {
                    paramFieldValue = ClassUtils.getValueOfFieldByReflection(obj, fieldName);
                }
            }
            else
            {
                paramFieldValue = ClassUtils.getValueOfFieldByReflection(obj, fieldName);
            }

            return paramFieldValue;
        }
        else
        {
            return null;
        }
    }

    protected SQLTableMapping getSQLTableMappingForAlias(String alias)
    {
        if (caseInsensitive)
        {
            return sqlTableByPrimary.get(alias.toUpperCase());
        }
        else
        {
            return sqlTableByPrimary.get(alias);
        }
    }

    protected void setSQLTableMappingForAlias(String alias, SQLTableMapping mapping)
    {
        if (alias == null)
        {
            return;
        }
        if (caseInsensitive)
        {
            sqlTableByPrimary.put(alias.toUpperCase(), mapping);
        }
        else
        {
            sqlTableByPrimary.put(alias, mapping);
        }
    }

    protected boolean hasSQLTableMappingForAlias(String alias)
    {
        if (caseInsensitive)
        {
            return sqlTableByPrimary.containsKey(alias.toUpperCase());
        }
        else
        {
            return sqlTableByPrimary.containsKey(alias);
        }
    }

    /**
     * Method to bind the specified variable to the table and mapping.
     * @param varName Variable name
     * @param cmd Metadata for this variable type
     * @param sqlTbl Table for this variable
     * @param mapping The mapping of this variable in the table
     */
    public void bindVariable(String varName, AbstractClassMetaData cmd, SQLTable sqlTbl, JavaTypeMapping mapping)
    {
        SQLTableMapping m = getSQLTableMappingForAlias(varName);
        if (m != null)
        {
            throw new NucleusException("Variable " + varName + " is already bound to " + m.table +
                " yet attempting to bind to " + sqlTbl);
        }
        else
        {
            NucleusLogger.QUERY.debug(">> QueryToSQL.bindVariable variable " + varName + 
                " being bound to table=" + sqlTbl + " mapping=" + mapping);
        }
        m = new SQLTableMapping(sqlTbl, cmd, mapping);
        setSQLTableMappingForAlias(varName, m);
    }

    /**
     * Method to bind the specified parameter to the defined type.
     * If the parameter is already bound (declared in the query perhaps, or bound via an earlier usage) then 
     * does nothing.
     * @param paramName Name of the parameter
     * @param type The type (or subclass)
     */
    public void bindParameter(String paramName, Class type)
    {
        Symbol paramSym = compilation.getSymbolTable().getSymbol(paramName);
        if (paramSym != null && paramSym.getValueType() == null)
        {
            NucleusLogger.QUERY.debug(">> QueryToSQL.bindParameter parameter=" + paramName + " has had its type set to " + type);
            paramSym.setValueType(type);
        }
    }

    /**
     * Accessor for the type of a variable if already known (declared?).
     * @param varName Name of the variable
     * @return The type if it is known
     */
    public Class getTypeOfVariable(String varName)
    {
        Symbol sym = compilation.getSymbolTable().getSymbol(varName);
        if (sym != null && sym.getValueType() != null)
        {
            return sym.getValueType();
        }
        return null;
    }
}
