/**********************************************************************
Copyright (c) 2007 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.scostore;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ObjectManager;
import org.datanucleus.StateManager;
import org.datanucleus.exceptions.NucleusDataStoreException;
import org.datanucleus.store.connection.ManagedConnection;
import org.datanucleus.store.mapped.exceptions.MappedDatastoreException;
import org.datanucleus.store.mapped.scostore.AbstractCollectionStore;
import org.datanucleus.store.mapped.scostore.AbstractSetStore;
import org.datanucleus.store.mapped.scostore.AbstractSetStoreSpecialization;
import org.datanucleus.store.rdbms.RDBMSStoreManager;
import org.datanucleus.store.rdbms.SQLController;
import org.datanucleus.util.Localiser;
import org.datanucleus.util.NucleusLogger;

import java.sql.PreparedStatement;
import java.sql.SQLException;

/**
 * RDBMS-specific implementation of an {@link AbstractSetStoreSpecialization}.
 */
class RDBMSAbstractSetStoreSpecialization extends RDBMSAbstractCollectionStoreSpecialization
    implements AbstractSetStoreSpecialization
{
    RDBMSAbstractSetStoreSpecialization(Localiser localiser, ClassLoaderResolver clr, RDBMSStoreManager storeMgr)
    {
        super(localiser, clr, storeMgr);
    }

    /**
     * Method to process an "add" statement.
     * @param ownerSM StateManager for the owner
     * @param conn The connection
     * @param batched Whether we are batching it
     * @param element The element
     * @return Number of datastore records changed (always 0 if batch since nothing yet changed)
     * @throws MappedDatastoreException Thrown if an error occurs
     */
    public int[] internalAdd(StateManager ownerSM, ManagedConnection conn, boolean batched, Object element, boolean processNow,
                             AbstractSetStore setStore)
    throws MappedDatastoreException
    {
        ObjectManager om = ownerSM.getObjectManager();
        SQLController sqlControl = storeMgr.getSQLController();
        String addStmt = getAddStmt(setStore);
        try
        {
            PreparedStatement ps = sqlControl.getStatementForUpdate(conn, addStmt, batched);
            try
            {
                int jdbcPosition = 1;
                jdbcPosition = BackingStoreHelper.populateOwnerInStatement(ownerSM, om, ps, jdbcPosition, setStore);
                jdbcPosition = BackingStoreHelper.populateElementInStatement(om, ps, element, jdbcPosition, setStore.getElementMapping());
                if (setStore.getRelationDiscriminatorMapping() != null)
                {
                    jdbcPosition =
                        BackingStoreHelper.populateRelationDiscriminatorInStatement(om, ps, jdbcPosition, setStore);
                }

                // Execute the statement
                return sqlControl.executeStatementUpdate(conn, addStmt, ps, processNow);
            }
            finally
            {
                sqlControl.closeStatement(conn, ps);
            }
        }
        catch (SQLException e)
        {
            throw new MappedDatastoreException(getAddStmt(setStore), e);
        }
    }

    public boolean remove(StateManager sm, Object element, int size, AbstractSetStore setStore)
    {
        boolean modified = false;
        ObjectManager om = sm.getObjectManager();
        String removeStmt = getRemoveStmt(setStore);
        try
        {
            ManagedConnection mconn = storeMgr.getConnection(om);
            SQLController sqlControl = storeMgr.getSQLController();
            try
            {
                PreparedStatement ps = sqlControl.getStatementForUpdate(mconn, removeStmt, false);
                try
                {
                    int jdbcPosition = 1;
                    jdbcPosition = BackingStoreHelper.populateOwnerInStatement(sm, om, ps, jdbcPosition, setStore);
                    jdbcPosition = BackingStoreHelper.populateElementInStatement(om, ps, element, jdbcPosition, setStore.getElementMapping());
                    if (setStore.getRelationDiscriminatorMapping() != null)
                    {
                        jdbcPosition =
                            BackingStoreHelper.populateRelationDiscriminatorInStatement(om, ps, jdbcPosition, setStore);
                    }

                    int[] rowsDeleted = sqlControl.executeStatementUpdate(mconn, removeStmt, ps, true);
                    modified = (rowsDeleted[0] == 1);
                }
                finally
                {
                    sqlControl.closeStatement(mconn, ps);
                }
            }
            finally
            {
                mconn.release();
            }
        }
        catch (SQLException e)
        {
            NucleusLogger.DATASTORE.error(e);
            String msg = localiser.msg("056012",removeStmt);
            NucleusLogger.DATASTORE.error(msg);
            throw new NucleusDataStoreException(msg, e);
        }
        return modified;
    }

    public int[] internalRemove(StateManager ownerSM, ManagedConnection conn, boolean batched, Object element, boolean executeNow,
                                AbstractCollectionStore acs) throws MappedDatastoreException
    {
        ObjectManager om = ownerSM.getObjectManager();
        SQLController sqlControl = storeMgr.getSQLController();
        String removeStmt = getRemoveStmt(acs);
        try
        {
            PreparedStatement ps = sqlControl.getStatementForUpdate(conn, removeStmt, batched);
            try
            {
                int jdbcPosition = 1;
                jdbcPosition = BackingStoreHelper.populateOwnerInStatement(ownerSM, om, ps, jdbcPosition, acs);
                jdbcPosition = BackingStoreHelper.populateElementInStatement(om, ps, element, jdbcPosition, acs.getElementMapping());
                if (acs.getRelationDiscriminatorMapping() != null)
                {
                    jdbcPosition =
                        BackingStoreHelper.populateRelationDiscriminatorInStatement(om, ps, jdbcPosition, acs);
                }

                // Execute the statement
                return sqlControl.executeStatementUpdate(conn, removeStmt, ps, executeNow);
            }
            finally
            {
                sqlControl.closeStatement(conn, ps);
            }
        }
        catch (SQLException e)
        {
            throw new MappedDatastoreException("SQLException", e);
        }
    }

    public void preInternalRemove(ManagedConnection mconn) throws MappedDatastoreException
    {
        SQLController sqlControl = storeMgr.getSQLController();
         // Process all waiting batched statements before we start our work
        try
        {
            sqlControl.processStatementsForConnection(mconn);
        }
        catch (SQLException e)
        {
            throw new MappedDatastoreException("SQLException", e);
        }
    }
}
