/**********************************************************************
Copyright (c) 2008 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.sql.expression;

import org.datanucleus.exceptions.NucleusException;
import org.datanucleus.query.expression.Expression;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.mapping.DatastoreMapping;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.rdbms.sql.SQLStatement;

/**
 * Representation of a Boolean literal in a Query.
 */
public class BooleanLiteral extends BooleanExpression implements SQLLiteral
{
    private final Boolean value;

    /**
     * Creates a boolean literal with the specified value, using the provided mapping.
     * @param stmt The SQL statement
     * @param mapping the mapping
     * @param value the value
     * @param parameterName Parameter name (represented as JDBC "?")
     */
    public BooleanLiteral(SQLStatement stmt, JavaTypeMapping mapping, Object value, String parameterName)
    {
        super(stmt, null, mapping);
        this.parameterName = parameterName;

        if (value == null)
        {
            this.value = null;
        }
        else if (value instanceof Boolean)
        {
            this.value = (Boolean)value;
        }
        else
        {
            throw new NucleusException("Cannot create " + this.getClass().getName() + 
                " for value of type " + (value != null ? value.getClass().getName() : null));
        }

        if (parameterName != null)
        {
            st.appendParameter(parameterName, mapping, this.value);
        }
        else
        {
            setStatement();
        }
    }

    public Object getValue()
    {
        return new Boolean(value);
    }

    public BooleanExpression and(SQLExpression expr)
    {
        if (expr instanceof BooleanExpression)
        {
            return value ? (BooleanExpression)expr : this;
        }
        else
        {
            return super.and(expr);
        }
    }

    public BooleanExpression eor(SQLExpression expr)
    {
        if (expr instanceof BooleanExpression)
        {
            return value ? expr.not() : (BooleanExpression)expr;
        }
        else
        {
            return super.eor(expr);
        }
    }

    public BooleanExpression ior(SQLExpression expr)
    {
        if (expr instanceof BooleanExpression)
        {
            return value ? this : (BooleanExpression)expr;
        }
        else
        {
            return super.ior(expr);
        }
    }

    public BooleanExpression not()
    {
        return new BooleanLiteral(stmt, mapping, !value, null);
    }

    public BooleanExpression eq(SQLExpression expr)
    {
        if (isParameter() || expr.isParameter())
        {
            // Comparison with parameter, so just give boolean compare
            return new BooleanExpression(this, Expression.OP_EQ, expr);
        }
        else if (expr instanceof BooleanExpression)
        {
            return value ? (BooleanExpression)expr : expr.not();
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression ne(SQLExpression expr)
    {
        if (isParameter() || expr.isParameter())
        {
            // Comparison with parameter, so just give boolean compare
            return new BooleanExpression(this, Expression.OP_NOTEQ, expr);
        }
        else if (expr instanceof BooleanExpression)
        {
            return value ? expr.not() : (BooleanExpression)expr;
        }
        else
        {
            return super.ne(expr);
        }
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.rdbms.sql.expression.SQLLiteral#setNotParameter()
     */
    public void setNotParameter()
    {
        if (parameterName == null)
        {
            return;
        }
        parameterName = null;

        st.clearStatement();
        setStatement();
    }

    protected void setStatement()
    {
        DatastoreMapping datastoreMapping = mapping.getDatastoreMapping(0);
        if (datastoreMapping.isStringBased())
        {
            // Persisted using "Y", "N"
            st.append(this.value ? "'Y'" : "'N'");
        }
        else if ((datastoreMapping.isBitBased() || datastoreMapping.isIntegerBased()) &&
                !stmt.getDatabaseAdapter().supportsOption(DatastoreAdapter.BIT_IS_REALLY_BOOLEAN))
        {
            // Persisted using "1", "0"
            st.append(this.value ? "1" : "0");
        }
        else
        {
            st.append(this.value ? "TRUE" : "(1=0)");
        }
    }
}