/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.datasource;

import java.beans.PropertyVetoException;

import javax.sql.DataSource;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.rdbms.datasource.DatastorePoolException;
import org.datanucleus.store.rdbms.datasource.DataNucleusDataSourceFactory;
import org.datanucleus.util.ClassUtils;

/**
 * Plugin for the creation of a C3P0 connection pool.
 * Note that all C3P0 classes are named explicitly in the code to avoid loading
 * them at class initialisation.
 * See http://www.mchange.com/projects/c3p0/index.html
 * See http://www.sf.net/projects/c3p0
 */
public class C3P0DataSourceFactory extends AbstractDataSourceFactory implements DataNucleusDataSourceFactory
{
    /**
     * Method to make a C3P0 DataSource for use internally in DataNucleus.
     * @param storeMgr Context
     * @return The DataSource
     * @throws Exception Thrown if an error occurs during creation
     */
    public DataSource makePooledDataSource(StoreManager storeMgr)
    {
        String dbDriver = storeMgr.getConnectionDriverName();
        String dbURL = storeMgr.getConnectionURL();
        String dbUser = storeMgr.getConnectionUserName();
        String dbPassword = storeMgr.getConnectionPassword();

        // Load the database driver
        ClassLoaderResolver clr = storeMgr.getNucleusContext().getClassLoaderResolver(null);
        loadDriver(dbDriver, clr);

        // Check the existence of the necessary pooling classes
        ClassUtils.assertClassForJarExistsInClasspath(clr, 
            "com.mchange.v2.c3p0.ComboPooledDataSource", "c3p0.jar");

        // Create the actual pool of connections
        com.mchange.v2.c3p0.ComboPooledDataSource ds = new com.mchange.v2.c3p0.ComboPooledDataSource();

        // Apply any properties
        // Note that C3P0 will always look for c3p0.properties at the root of the CLASSPATH
        if (storeMgr.hasProperty("datanucleus.connectionPool.maxStatements"))
        {
            int size = storeMgr.getIntProperty("datanucleus.connectionPool.maxStatements");
            if (size >= 0)
            {
                ds.setMaxStatements(size);
            }
        }
        if (storeMgr.hasProperty("datanucleus.connectionPool.maxPoolSize"))
        {
            int size = storeMgr.getIntProperty("datanucleus.connectionPool.maxPoolSize");
            if (size >= 0)
            {
                ds.setMaxPoolSize(size);
            }
        }
        if (storeMgr.hasProperty("datanucleus.connectionPool.minPoolSize"))
        {
            int size = storeMgr.getIntProperty("datanucleus.connectionPool.minPoolSize");
            if (size >= 0)
            {
                ds.setMinPoolSize(size);
            }
        }
        if (storeMgr.hasProperty("datanucleus.connectionPool.initialPoolSize"))
        {
            int size = storeMgr.getIntProperty("datanucleus.connectionPool.initialPoolSize");
            if (size >= 0)
            {
                ds.setInitialPoolSize(size);
            }
        }

        // Load the database driver
        try
        {
            ds.setDriverClass(dbDriver);
        }
        catch (PropertyVetoException pve)
        {
            throw new DatastorePoolException("C3P0", dbDriver, dbURL, pve);
        }

        ds.setJdbcUrl(dbURL);
        ds.setUser(dbUser);
        ds.setPassword(dbPassword);

        // Users can specify the various properties via a c3p0.properties file
 
        return ds;
    }
}