/**********************************************************************
Copyright (c) 2005 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. 


Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.rdbms.scostore;

import java.util.Collection;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.store.ObjectProvider;
import org.datanucleus.store.StoreManager;
import org.datanucleus.store.mapped.mapping.EmbeddedElementPCMapping;
import org.datanucleus.store.mapped.mapping.JavaTypeMapping;
import org.datanucleus.store.scostore.CollectionStore;

/**
 * Abstract representation of a store of a Collection.
 * Contains all common parts of storing Sets and Lists.
 */
public abstract class AbstractCollectionStore extends ElementContainerStore implements CollectionStore
{
    /**
     * Constructor.
     * @param storeMgr Manager for the store
     * @param clr ClassLoader resolver
     */
    protected AbstractCollectionStore(StoreManager storeMgr, ClassLoaderResolver clr, 
            AbstractCollectionStoreSpecialization abstractCollectionStoreSpecialization)
    {
        super(storeMgr, clr, abstractCollectionStoreSpecialization);
    }

    /**
     * Method to update a field of an embedded element.
     * @param sm State Manager of the owner
     * @param element The element to update
     * @param fieldNumber The number of the field to update
     * @param value The value
     * @return true if the datastore was updated
     */
    public boolean updateEmbeddedElement(ObjectProvider sm, Object element, int fieldNumber, Object value)
    {
        boolean modified = false;
        if (elementMapping != null && elementMapping instanceof EmbeddedElementPCMapping)
        {
            String fieldName = emd.getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber).getName();
            if (fieldName == null)
            {
                // We have no mapping for this field so presumably is the owner field or a PK field
                return false;
            }
            JavaTypeMapping fieldMapping = ((EmbeddedElementPCMapping)elementMapping).getJavaTypeMapping(fieldName);
            if (fieldMapping == null)
            {
                // We have no mapping for this field so presumably is the owner field or a PK field
                return false;
            }
            modified = getSpecialization().updateEmbeddedElement(sm, element,
                fieldNumber, value, fieldMapping, this);
        }

        return modified;
    }

    private AbstractCollectionStoreSpecialization getSpecialization()
    {
        return (AbstractCollectionStoreSpecialization) specialization;
    }

  /**
     * Method to update the collection to be the supplied collection of elements.
     * @param sm StateManager of the object
     * @param coll The collection to use
     */
    public void update(ObjectProvider sm, Collection coll)
    {
        // Crude update - remove existing and add new!
        clear(sm);
        addAll(sm, coll, 0);
    }

    /**
     * Method to verify if the association owner vs elements contains
     * a specific element in the association 
     * @param sm The StateManager
     * @param element The element
     * @return Whether it contains the element 
     */
    public boolean contains(ObjectProvider sm, Object element)
    {
        if (!validateElementForReading(sm, element))
        {
            return false;
        }
        return getSpecialization().contains(sm, element, this);
    }
}