/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.backup;

import ch.qos.logback.classic.LoggerContext;
import java.io.File;
import java.io.IOException;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.NoSuchElementException;
import org.neo4j.backup.BackupExtensionService;
import org.neo4j.backup.BackupServer;
import org.neo4j.backup.BackupService;
import org.neo4j.com.ComException;
import org.neo4j.consistency.ConsistencyCheckSettings;
import org.neo4j.graphdb.TransactionFailureException;
import org.neo4j.graphdb.factory.GraphDatabaseSettings;
import org.neo4j.helpers.Args;
import org.neo4j.helpers.Service;
import org.neo4j.helpers.collection.MapUtil;
import org.neo4j.kernel.DefaultFileSystemAbstraction;
import org.neo4j.kernel.configuration.Config;
import org.neo4j.kernel.impl.nioneo.store.FileSystemAbstraction;
import org.neo4j.kernel.impl.nioneo.store.MismatchingStoreIdException;
import org.neo4j.kernel.impl.storemigration.LogFiles;
import org.neo4j.kernel.impl.storemigration.StoreFiles;
import org.neo4j.kernel.impl.storemigration.UpgradeNotAllowedByConfigurationException;
import org.neo4j.kernel.lifecycle.LifeSupport;
import org.neo4j.kernel.logging.LogbackService;
import org.neo4j.kernel.logging.Logging;
import org.neo4j.kernel.logging.SystemOutLogging;
import org.slf4j.impl.StaticLoggerBinder;

public class BackupTool {
    private static final String TO = "to";
    private static final String FROM = "from";
    private static final String VERIFY = "verify";
    private static final String CONFIG = "config";
    public static final String DEFAULT_SCHEME = "single";
    static final String MISMATCHED_STORE_ID = "You tried to perform a backup from database %s, but the target directory contained a backup from database %s. ";
    private final BackupService backupService;
    private final PrintStream systemOut;
    private final FileSystemAbstraction fs;

    public static void main(String[] args) {
        BackupTool tool = new BackupTool(new BackupService(), System.out);
        try {
            tool.run(args);
        }
        catch (ToolFailureException e) {
            e.haltJVM();
        }
    }

    BackupTool(BackupService backupService, PrintStream systemOut) {
        this.backupService = backupService;
        this.systemOut = systemOut;
        this.fs = new DefaultFileSystemAbstraction();
    }

    void run(String[] args) throws ToolFailureException {
        Args arguments = new Args(args);
        this.checkArguments(arguments);
        String from = arguments.get(FROM, null);
        String to = arguments.get(TO, null);
        boolean verify = arguments.getBoolean(VERIFY, Boolean.valueOf(true), Boolean.valueOf(true));
        Config tuningConfiguration = this.readTuningConfiguration(TO, arguments);
        URI backupURI = null;
        try {
            backupURI = new URI(from);
        }
        catch (URISyntaxException e) {
            throw new ToolFailureException("Please properly specify a location to backup as a valid URI in the form <scheme>://<host>[:port], where scheme is the target database's running mode, eg ha");
        }
        String module = backupURI.getScheme();
        BackupExtensionService service = null;
        if (module != null && !DEFAULT_SCHEME.equals(module)) {
            try {
                service = (BackupExtensionService)Service.load(BackupExtensionService.class, (String)module);
            }
            catch (NoSuchElementException e) {
                throw new ToolFailureException(String.format("%s was specified as a backup module but it was not found. Please make sure that the implementing service is on the classpath.", module));
            }
        }
        if (service != null) {
            SystemOutLogging logging;
            try {
                this.getClass().getClassLoader().loadClass("ch.qos.logback.classic.LoggerContext");
                LifeSupport life = new LifeSupport();
                LogbackService logbackService = (LogbackService)life.add((Object)new LogbackService(tuningConfiguration, (LoggerContext)StaticLoggerBinder.getSingleton().getLoggerFactory(), "neo4j-backup-logback.xml"));
                life.start();
                logging = logbackService;
            }
            catch (Throwable e) {
                logging = new SystemOutLogging();
            }
            try {
                backupURI = service.resolve(backupURI, arguments, (Logging)logging);
            }
            catch (Throwable e) {
                throw new ToolFailureException(e.getMessage());
            }
        }
        this.doBackup(backupURI, to, verify, tuningConfiguration);
    }

    private void checkArguments(Args arguments) throws ToolFailureException {
        if (arguments.get(FROM, null) == null) {
            throw new ToolFailureException("Please specify " + BackupTool.dash(FROM) + ", examples:\n" + "  " + BackupTool.dash(FROM) + " single://192.168.1.34\n" + "  " + BackupTool.dash(FROM) + " single://192.168.1.34:1234\n" + "  " + BackupTool.dash(FROM) + " ha://192.168.1.15:2181\n" + "  " + BackupTool.dash(FROM) + " ha://192.168.1.15:2181,192.168.1.16:2181");
        }
        if (arguments.get(TO, null) == null) {
            throw new ToolFailureException("Specify target location with " + BackupTool.dash(TO) + " <target-directory>");
        }
    }

    public Config readTuningConfiguration(String storeDir, Args arguments) throws ToolFailureException {
        Map specifiedProperties = MapUtil.stringMap((String[])new String[0]);
        String propertyFilePath = arguments.get(CONFIG, null);
        if (propertyFilePath != null) {
            File propertyFile = new File(propertyFilePath);
            try {
                specifiedProperties = MapUtil.load((File)propertyFile);
            }
            catch (IOException e) {
                throw new ToolFailureException(String.format("Could not read configuration properties file [%s]", propertyFilePath), e);
            }
        }
        specifiedProperties.put(GraphDatabaseSettings.store_dir.name(), storeDir);
        return new Config(specifiedProperties, new Class[]{GraphDatabaseSettings.class, ConsistencyCheckSettings.class});
    }

    private void doBackup(URI from, String to, boolean checkConsistency, Config tuningConfiguration) throws ToolFailureException {
        if (this.backupService.directoryContainsDb(to)) {
            try {
                this.doBackupIncremental(from, to, checkConsistency, tuningConfiguration);
                this.systemOut.println("Done");
            }
            catch (MismatchingStoreIdException e) {
                this.systemOut.println("Backup failed.");
                throw new ToolFailureException(String.format(MISMATCHED_STORE_ID, e.getExpected(), e.getEncountered()));
            }
        } else {
            this.doBackupFull(from, to, checkConsistency, tuningConfiguration);
            this.systemOut.println("Done");
        }
    }

    private void doBackupFull(URI from, String to, boolean checkConsistency, Config tuningConfiguration) throws ToolFailureException {
        this.systemOut.println("Performing full backup from '" + from + "'");
        try {
            this.backupService.doFullBackup(from.getHost(), this.extractPort(from), to, checkConsistency, tuningConfiguration);
        }
        catch (ComException e) {
            throw new ToolFailureException("Couldn't connect to '" + from + "'", e);
        }
    }

    private void doBackupIncremental(URI from, String to, boolean verify, Config tuningConfiguration) throws ToolFailureException {
        this.systemOut.println("Performing incremental backup from '" + from + "'");
        boolean failedBecauseOfStoreVersionMismatch = false;
        try {
            this.backupService.doIncrementalBackup(from.getHost(), this.extractPort(from), to, verify);
        }
        catch (TransactionFailureException e) {
            if (e.getCause() instanceof UpgradeNotAllowedByConfigurationException) {
                failedBecauseOfStoreVersionMismatch = true;
            }
            throw new ToolFailureException("TransactionFailureException from existing backup at '" + from + "'" + ".", e);
        }
        catch (ComException e) {
            throw new ToolFailureException("Couldn't connect to '" + from + "' ", e);
        }
        if (failedBecauseOfStoreVersionMismatch) {
            this.systemOut.println("The database present in the target directory is of an older version. Backing that up in target and performing a full backup from source");
            try {
                BackupTool.moveExistingDatabase(this.fs, to);
            }
            catch (IOException e) {
                throw new ToolFailureException("There was a problem moving the old database out of the way - cannot continue, aborting.", e);
            }
            this.doBackupFull(from, to, verify, tuningConfiguration);
        }
    }

    private int extractPort(URI from) {
        int port = from.getPort();
        if (port == -1) {
            port = BackupServer.DEFAULT_PORT;
        }
        return port;
    }

    private static void moveExistingDatabase(FileSystemAbstraction fs, String to) throws IOException {
        File toDir = new File(to);
        File backupDir = new File(toDir, "old-version");
        if (!fs.mkdir(backupDir)) {
            throw new IOException("Trouble making target backup directory " + backupDir.getAbsolutePath());
        }
        StoreFiles.move((FileSystemAbstraction)fs, (File)toDir, (File)backupDir);
        LogFiles.move((FileSystemAbstraction)fs, (File)toDir, (File)backupDir);
    }

    private static String dash(String name) {
        return "-" + name;
    }

    static class ToolFailureException
    extends Exception {
        ToolFailureException(String message) {
            super(message);
        }

        ToolFailureException(String message, Throwable cause) {
            super(message, cause);
        }

        void haltJVM() {
            System.out.println(this.getMessage());
            if (this.getCause() != null) {
                this.getCause().printStackTrace(System.out);
            }
            System.exit(1);
        }
    }
}

