/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.kernel.monitoring;

import java.lang.management.GarbageCollectorMXBean;
import java.lang.management.ManagementFactory;
import java.time.Duration;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import org.neo4j.logging.Log;
import org.neo4j.scheduler.Group;
import org.neo4j.scheduler.JobHandle;
import org.neo4j.scheduler.JobScheduler;
import org.neo4j.util.Preconditions;
import org.neo4j.util.VisibleForTesting;

public class VmPauseMonitor {
    private final long measurementDurationNs;
    private final long stallAlertThresholdNs;
    private final Log log;
    private final JobScheduler jobScheduler;
    private final Consumer<VmPauseInfo> listener;
    private JobHandle job;

    public VmPauseMonitor(Duration measureInterval, Duration stallAlertThreshold, Log log, JobScheduler jobScheduler, Consumer<VmPauseInfo> listener) {
        this.measurementDurationNs = Preconditions.requirePositive((long)measureInterval.toNanos());
        this.stallAlertThresholdNs = Preconditions.requireNonNegative((long)stallAlertThreshold.toNanos());
        this.log = Objects.requireNonNull(log);
        this.jobScheduler = Objects.requireNonNull(jobScheduler);
        this.listener = Objects.requireNonNull(listener);
    }

    public void start() {
        this.log.debug("Starting VM pause monitor");
        Preconditions.checkState((this.job == null ? 1 : 0) != 0, (String)"VM pause monitor is already started");
        this.job = Objects.requireNonNull(this.jobScheduler.schedule(Group.VM_PAUSE_MONITOR, this::run));
    }

    public void stop() {
        this.log.debug("Stopping VM pause monitor");
        Preconditions.checkState((this.job != null ? 1 : 0) != 0, (String)"VM pause monitor is not started");
        this.job.cancel(true);
        this.job = null;
    }

    private void run() {
        try {
            this.monitor();
        }
        catch (InterruptedException ignore) {
            this.log.debug("VM pause monitor stopped");
        }
        catch (RuntimeException e) {
            this.log.debug("VM pause monitor failed", (Throwable)e);
        }
    }

    @VisibleForTesting
    void monitor() throws InterruptedException {
        GcStats lastGcStats = VmPauseMonitor.getGcStats();
        long nextCheckPoint = System.nanoTime() + this.measurementDurationNs;
        while (!this.isStopped()) {
            TimeUnit.NANOSECONDS.sleep(this.measurementDurationNs);
            long now = System.nanoTime();
            long pauseNs = Math.max(0L, now - nextCheckPoint);
            nextCheckPoint = now + this.measurementDurationNs;
            GcStats gcStats = VmPauseMonitor.getGcStats();
            if (pauseNs >= this.stallAlertThresholdNs) {
                VmPauseInfo pauseInfo = new VmPauseInfo(TimeUnit.NANOSECONDS.toMillis(pauseNs), gcStats.time - lastGcStats.time, gcStats.count - lastGcStats.count);
                this.listener.accept(pauseInfo);
            }
            lastGcStats = gcStats;
        }
    }

    @VisibleForTesting
    boolean isStopped() {
        return Thread.currentThread().isInterrupted();
    }

    private static GcStats getGcStats() {
        long time = 0L;
        long count = 0L;
        for (GarbageCollectorMXBean gcBean : ManagementFactory.getGarbageCollectorMXBeans()) {
            time += gcBean.getCollectionTime();
            count += gcBean.getCollectionCount();
        }
        return new GcStats(time, count);
    }

    private static class GcStats {
        private final long time;
        private final long count;

        private GcStats(long time, long count) {
            this.time = time;
            this.count = count;
        }
    }

    public static class VmPauseInfo {
        private final long pauseTime;
        private final long gcTime;
        private final long gcCount;

        VmPauseInfo(long pauseTime, long gcTime, long gcCount) {
            this.pauseTime = pauseTime;
            this.gcTime = gcTime;
            this.gcCount = gcCount;
        }

        public long getPauseTime() {
            return this.pauseTime;
        }

        public String toString() {
            return String.format("{pauseTime=%d, gcTime=%d, gcCount=%d}", this.pauseTime, this.gcTime, this.gcCount);
        }
    }
}

