/*
 * Copyright (c) 2001-2006, John Mettraux, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: OwfeDataSource.java 3251 2006-09-10 12:36:49Z jmettraux $
 */

//
// OwfeDataSource.java
//
// jmettraux@openwfe.org
//
// generated with 
// jtmpl 1.0.04 31.10.2002 John Mettraux (jmettraux@openwfe.org)
//

package openwfe.org.sql.ds;

import openwfe.org.MapUtils;
import openwfe.org.AbstractService;
import openwfe.org.ServiceException;
import openwfe.org.ApplicationContext;
import openwfe.org.sql.SqlUtils;


/**
 * A dataSource  la OpenWFE
 *
 * <p><font size=2>CVS Info :
 * <br>$Author: jmettraux $
 * <br>$Date: 2006-09-10 14:36:49 +0200 (Sun, 10 Sep 2006) $
 * <br>$Id: OwfeDataSource.java 3251 2006-09-10 12:36:49Z jmettraux $ </font>
 *
 * @author john.mettraux@openwfe.org
 * @author jay.lawrence@openwfe.org
 */
public abstract class OwfeDataSource

    extends AbstractService

{

    private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
        .getLogger(OwfeDataSource.class.getName());

    //
    // CONSTANTS (definitions)

    /**
     * All DataSources generally require a 'url' parameter/attribute.
     */
    public final static String P_URL 
        = "url";

    /**
     * You can set a 'loginTimeout' for the underlying datasource.
     */
    public final static String P_LOGIN_TIMEOUT
        = "loginTimeout";

    /**
     * If you set the parameter 'doNotCheckConnection' to 'true' or 'yes', 
     * the datasource will never check the connection before returning it
     * in getConnection().
     * By default, connections will get checked.
     */
    public final static String P_DO_NOT_CHECK_CONNECTION
        = "doNotCheckConnection";

    /**
     * You can use this constant in your services to grab a reference to 
     * a datasource.
     */
    public final static String P_DATA_SOURCE = "dataSource";

    //
    // FIELDS

    private javax.sql.DataSource dataSource = null;

    //
    // the connection shared by every sql service

    private long lastConnectionRelease = System.currentTimeMillis();
    private java.sql.Connection sqlConnection = null;

    private boolean doNotCheckConnection = false;

    //
    // CONSTRUCTORS

    public void init 
        (final String serviceName, 
         final ApplicationContext context, 
         final java.util.Map serviceParams)
    throws 
        ServiceException
    {
        super.init(serviceName, context, serviceParams);

        this.doNotCheckConnection = MapUtils
            .getAsBoolean(serviceParams, P_DO_NOT_CHECK_CONNECTION, false);

        log.info
            ("init() "+
             "doNotCheckConnection is set to '"+this.doNotCheckConnection+"'");
    }

    //
    // METHODS

    /**
     * Setting the data source
     */
    protected void setDataSource (final javax.sql.DataSource ds)
        throws ServiceException
    {
        this.dataSource = ds;

        //
        // set the login timeout

        final long lt = MapUtils.getAsTime(getParams(), P_LOGIN_TIMEOUT, -1);

        if (lt < 0) return; // no timeout to set

        final int timeout = (int)(lt / 1000L);

        try
        {
            this.dataSource.setLoginTimeout(timeout);
        }
        catch (final java.sql.SQLException se)
        {
            throw new ServiceException
                ("Failed to set login timeout on wrapped datasource", se);
        }
    }

    /**
     * TODO : comment me.
     */
    protected java.sql.Connection establishConnection ()
        throws java.sql.SQLException
    {
        return this.dataSource.getConnection();
    }

    /**
     * Returns a connection, if a connection has already been established
     * simply returns it.
     */
    public java.sql.Connection getConnection ()
        throws java.sql.SQLException
    {
        synchronized (this.getClass())
        {
            if (log.isDebugEnabled())
                log.debug("getConnection() ("+this.getClass().getName()+")");

            long start = System.currentTimeMillis();

            long delta = start - this.lastConnectionRelease;

            if (this.sqlConnection == null || 
                delta > (long)3600 * 1000 ||
                isConnectionInvalid())
            {
                if (this.sqlConnection != null)
                {
                    log.debug("getConnection() closing old connection");
                    try
                    {
                        this.sqlConnection.close();
                    }
                    catch (final Throwable t)
                    {
                        if (log.isDebugEnabled())
                        {
                            log.debug
                                ("getConnection() "+
                                 "failed to close old connection "+
                                 t.toString());
                        }
                    }
                }

                log.debug("getConnection() setting up new connection");

                this.sqlConnection = establishConnection();
                this.lastConnectionRelease = System.currentTimeMillis();
            }

            long elapsedTime = System.currentTimeMillis() - start;

            if (log.isDebugEnabled())
                log.debug("getConnection() took "+elapsedTime+" ms");

            return this.sqlConnection;
        }
    }

    /**
     * release connection : in fact, it doesn't close it, it simply
     * tags it as unused
     */
    public void releaseConnection ()
    {
        log.debug("releaseConnection()");

        this.lastConnectionRelease = System.currentTimeMillis();
    }

    /**
     * Returns a brand new connection, the code calling this method
     * has the responsability of properly closing this new connection.
     */
    public java.sql.Connection newConnection ()
        throws java.sql.SQLException
    {
        //log.debug("newConnection() returning brand new connection");

        return this.dataSource.getConnection();
    }

    /**
     * This method may be overriden by datasource implementations, at 
     * this level, it does nothing, but it's called by getConnection().
     * It returns true when the connection seems to be useless.
     */
    protected boolean isConnectionInvalid ()
    {
        if (getConnectionValiditySqlCode() == null) return false;

        if (this.doNotCheckConnection) return false;

        java.sql.Statement st = null;
        java.sql.ResultSet rs = null;
        try
        {
            st = this.sqlConnection.createStatement();
            rs = st.executeQuery(getConnectionValiditySqlCode());
        }
        catch (final Throwable t)
        {
            //log.debug("isConnectionInvalid() Yes :", t);
            log.info("isConnectionInvalid() Yes : "+t.toString());
            return true;
        }
        finally
        {
            SqlUtils.closeStatement(st, rs);
        }

        log.debug("isConnectionInvalid() seems that con is valid.");

        return false;
    }

    //
    // ABSTRACT METHODS

    /**
     * This method when implemented has to return a piece of SQL code
     * that will be run against any database for connection
     * validity checking.
     * For example, for MySQL (and some others), "select now()" could
     * be returned.<br>
     * If this method returns null, connection validity will not be
     * tested.
     */
    public abstract String getConnectionValiditySqlCode ();

    //
    // METHODS from Service

    public void stop ()
        throws ServiceException
    {
        log.debug("stop()");

        super.stop();

        this.lastConnectionRelease = -1;

        if (this.sqlConnection != null)
        {
            try
            {
                this.sqlConnection.close();
                this.sqlConnection = null;
            }
            catch (final java.sql.SQLException se)
            {
                // ignore
            }
        }
    }

    /**
     * A method by Jay Lawrence, DataBase specialized exception logging :
     * will iterate through the SQLException by calling getNextException()
     */
    public void logSQLException
        (final String funcName, 
         final org.apache.log4j.Logger logger, 
         java.sql.SQLException e) 
    {
        // TODO : wouldn'it be better to have full stack traces ?

        while (e != null) 
        {
            logger.error(funcName+"() SQLException : "+e.toString());
            e = (java.sql.SQLException)e.getNextException();
        }
    }

    //
    // STATIC METHODS

    /**
     * Looks in the serviceParams for the value of the parameter named
     * 'dataSource' (openwfe.org.sql.ds.OwfeDataSource.P_DATA_SOURCE), and
     * then given this value (a service name) returns the DataSource 
     * stored in the application context under this name.
     */
    public static OwfeDataSource lookupDataSource
        (final ApplicationContext context,
         final java.util.Map serviceParams)
    throws
        ServiceException
    {
        String dsKey = (String)serviceParams.get(P_DATA_SOURCE);

        /*
        if (dsKey == null)
        {
            throw new ServiceException
                ("No dataSource indicated, parameter '"+
                 OwfeDataSource.DATA_SOURCE+"' is not set");
        }
        */

        if (dsKey == null) dsKey = P_DATA_SOURCE;

        if (log.isDebugEnabled())
        {
            log.debug
                ("lookupDataSource() "+
                 "looking for data source named '"+dsKey+"'");
        }

        final OwfeDataSource ds = (OwfeDataSource)context.lookup(dsKey);

        if (ds == null)
        {
            throw new ServiceException
                ("No OwfeDataSource stored as service named '"+dsKey+"'");
        }

        return ds;
    }

}
