/*
 * Copyright (c) 2005, Christelle Heritier, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: DynamicInvoker.java 2861 2006-06-23 13:18:35Z lukas_eder $
 */
package openwfe.org.ws.invoker;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Operation;
import javax.wsdl.Port;
import javax.wsdl.Service;
import javax.xml.namespace.QName;
import javax.xml.rpc.encoding.DeserializerFactory;

import openwfe.org.engine.dispatch.DispatchingException;

import org.apache.axis.Constants;
import org.apache.axis.client.Call;
import org.apache.axis.encoding.TypeMapping;
import org.apache.axis.encoding.ser.ArrayDeserializerFactory;
import org.apache.axis.encoding.ser.ArraySerializerFactory;
import org.apache.axis.encoding.ser.ElementDeserializer;
import org.apache.axis.encoding.ser.ElementDeserializerFactory;
import org.apache.axis.encoding.ser.ElementSerializerFactory;
import org.apache.axis.encoding.ser.SimpleDeserializer;
import org.apache.axis.wsdl.gen.Parser;
import org.apache.axis.wsdl.symbolTable.BindingEntry;
import org.apache.axis.wsdl.symbolTable.Parameter;
import org.apache.axis.wsdl.symbolTable.Parameters;
import org.apache.axis.wsdl.symbolTable.SymbolTable;
import org.apache.axis.wsdl.symbolTable.TypeEntry;
import org.apache.axis.wsdl.toJava.Utils;

/**
 * A DynamicInvoker by Christelle Hritier.
 * 
 * <p>
 * <font size=2>CVS Info :<br>
 * $Id: DynamicInvoker.java 2861 2006-06-23 13:18:35Z lukas_eder $ </font>
 * 
 * @author Christelle Heritier
 * @author john.mettraux@openwfe.org
 * @author Lukas Eder
 */
public class DynamicInvoker {

	private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
			.getLogger(DynamicInvoker.class.getName());

	private Parser wsdlParser = null;

	/**
	 * Initialize the DynamicInvoker for a given URL that locates a WSDL file.
	 * 
	 * @param url
	 *            The URL locating the WSDL file.
	 * @throws Exception
	 */
	public DynamicInvoker(String url) throws Exception {
		wsdlParser = new Parser();

		if (log.isDebugEnabled())
			log.debug("DynamicInvoker() reading WSDL from " + url);

		wsdlParser.run(url);
	}

	/**
	 * Invoke a method on the underlying service.
	 * 
	 * @param operationName
	 *            The operation that is invoked.
	 * @param params
	 *            The operation's parameters.
	 * @return A map of String=&gt;Object having as key parameter names (OUT,
	 *         INOUT or return parameters) and as values their corresponding
	 *         invokation results.
	 * @throws Exception
	 */
	public HashMap invokeMethod(String operationName, String[] args)
			throws DispatchingException {
		OperationData data = getOperationData(operationName);
		Call call = createCall(data);

		// invoke the service
		Object[] arguments = deserializeArguments(call, data.inParams, args);
		Object response = null;
		try {
			response = call.invoke(arguments);
		} catch (final Throwable t) {
			throw new DispatchingException("Failed to invoke service", t);
		}

		HashMap result = new HashMap();
		Map outValues = call.getOutputParams();
		for (int i = 0; i < data.outParams.length; i++) {
			String name = data.outParams[i].getName();
			Object value = outValues.get(name);

			if (value == null)
				result.put(name, response);
			else
				result.put(name, value);
		}

		return result;
	}

	/**
	 * Applies installed deserializers for each one of the arguments.
	 * <p>
	 * This method assumes that the parameters and args arguments have the same
	 * size and that the parameters/args follow the exact same ordering within
	 * their corresponding array. A deserializer is then retrieved from the RPC
	 * call object's type mapping in order to deserialize all args.
	 * 
	 * @param call
	 *            The RPC object hosting the type mapping.
	 * @param parameters
	 *            The call's parameters.
	 * @param args
	 *            The original arguments (as passed by OpenWFE to the invoker).
	 * @return An array of deserialized parameters.
	 * @throws DispatchingException
	 */
	private Object[] deserializeArguments(final Call call,
			final Parameter[] parameters, final String[] args)
			throws DispatchingException {

		Object[] result = new Object[parameters.length];

		for (int i = 0; i < parameters.length; i++) {
			QName qName = Utils.getXSIType(parameters[i]);
			TypeMapping mapping = call.getTypeMapping();
			DeserializerFactory factory = mapping.getDeserializer(qName);
			Object obj = factory.getDeserializerAs(Constants.AXIS_SAX);

			if (obj instanceof SimpleDeserializer) {
				SimpleDeserializer deserializer = (SimpleDeserializer) obj;
				try {
					result[i] = deserializer.makeValue(args[i]);
				} catch (final Throwable t) {
					throw new DispatchingException(
							"Error while trying to deserialize a parameter : "
									+ args[i], t);
				}
			} else {
				throw new DispatchingException(
						"Unsupported deserializer type : " + obj.getClass());
			}
		}

		return result;
	}

	// -------------------------------------------------------------------------
	// HELPER METHODS
	// -------------------------------------------------------------------------

	/**
	 * Retrieve a some data from the symbol table that hosts a given operation.
	 * 
	 * @param operationName
	 *            The operation's name.
	 * @return Some data describing the operation
	 * @throws DispatchingException
	 */
	private OperationData getOperationData(String operationName)
			throws DispatchingException {

		OperationData data = new OperationData();

		// Check all services for their ports
		Map services = getSymbolTable().getDefinition().getServices();
		Iterator it1 = services.values().iterator();
		while (it1.hasNext()) {

			// Check all ports for their operations
			Service service = (Service) it1.next();
			Map ports = service.getPorts();
			Iterator it2 = ports.values().iterator();
			while (it2.hasNext()) {

				// Try to retrieve the operation with the requested name.
				Port port = (Port) it2.next();
				Binding binding = port.getBinding();
				BindingOperation operation = binding.getBindingOperation(
						operationName, null, null);

				// If any operation was found, return the service.
				if (operation != null) {

					// Store javax.wsdl.* objects
					data.service = service;
					data.port = port;
					data.binding = binding;
					data.operation = operation.getOperation();

					// Store org.apache.axis.wsdl.* objects
					BindingEntry bindingEntry = getSymbolTable()
							.getBindingEntry(binding.getQName());
					data.parameters = bindingEntry
							.getParameters(data.operation);

					List inParams = new ArrayList();
					List outParams = new ArrayList();

					for (int i = 0; i < data.parameters.list.size(); i++) {
						Parameter p = (Parameter) data.parameters.list.get(i);

						if (p.getMode() == Parameter.IN
								|| p.getMode() == Parameter.INOUT)
							inParams.add(p);

						if (p.getMode() == Parameter.OUT
								|| p.getMode() == Parameter.INOUT)
							outParams.add(p);
					}

					if (data.parameters.returnParam != null) {
						outParams.add(data.parameters.returnParam);
					}

					data.inParams = (Parameter[]) inParams
							.toArray(new Parameter[inParams.size()]);
					data.outParams = (Parameter[]) outParams
							.toArray(new Parameter[outParams.size()]);

					return data;
				}
			}
		}

		throw new DispatchingException("Operation was not found : "
				+ operationName);
	}

	/**
	 * Create an RPC object for an operation described by an operation data
	 * object.
	 * 
	 * @param data
	 *            The operation's description object.
	 * @return An RPC object.
	 * @throws DispatchingException
	 */
	private Call createCall(OperationData data) throws DispatchingException {

		// Create an axis service object for the given operation data's service.
		org.apache.axis.client.Service axisService = null;
		try {
			axisService = new org.apache.axis.client.Service(wsdlParser,
					data.service.getQName());
		} catch (final Throwable t) {
			throw new DispatchingException("Failed to initialize service : "
					+ data.service.getQName(), t);
		}

		// Create an RPC object for the operation data's port and operation.
		QName portName = QName.valueOf(data.port.getName());
		QName operationName = QName.valueOf(data.operation.getName());
		
		Call call = null;
		try {
			call = (Call) axisService.createCall(portName, operationName);
		} catch (final Throwable t) {
			throw new DispatchingException("Failed to initialize call "
					+ portName + " / " + operationName, t);
		}

		initializeCall(call);
		initializeTypeMapping(call, data);

		return call;
	}

	/**
	 * Set various default values (like timeout) for the RPC object.
	 * 
	 * @param call
	 *            The RPC object
	 */
	private void initializeCall(Call call) {
		call.setTimeout(new Integer(15 * 1000));
		call.setProperty(ElementDeserializer.DESERIALIZE_CURRENT_ELEMENT,
				Boolean.TRUE);
	}

	/**
	 * Installs type mappings for parameters (IN, OUT, INOUT) that are found in
	 * the operation data.
	 * 
	 * @param call
	 *            The RPC object for which to register type mappings.
	 * @param data
	 *            The operation data containing parameters whose types need to
	 *            be mapped.
	 */
	private void initializeTypeMapping(Call call, OperationData data) {
		Set types = new HashSet();

		for (int i = 0; i < data.parameters.list.size(); i++) {
			Parameter parameter = (Parameter) data.parameters.list.get(i);
			TypeEntry type = parameter.getType();
			types.addAll(type.getNestedTypes(getSymbolTable(), true));
			types.add(type);
		}
		
		if (data.parameters.returnParam != null) {
			TypeEntry type = data.parameters.returnParam.getType();
			types.addAll(type.getNestedTypes(getSymbolTable(), true));
			types.add(type);
		}

		initializeTypeMapping(call, types);
	}

	/**
	 * Installs type mappings for parameters (IN, OUT, INOUT) that are found in
	 * the operation data.
	 * 
	 * @param call
	 *            The RPC object for which to register type mappings.
	 * @param types
	 *            A set of types that will be registered for the RPC object.
	 */
	private void initializeTypeMapping(Call call, Collection types) {
		Iterator it = types.iterator();

		while (it.hasNext()) {
			TypeEntry type = (TypeEntry) it.next();

			// deserializer for array
			if (type.getDimensions().indexOf("[]") > -1) {
				call.registerTypeMapping(org.w3c.dom.Element.class, type
						.getQName(), new ArraySerializerFactory(),
						new ArrayDeserializerFactory());

			} else if (!type.isBaseType()) {
				call.registerTypeMapping(org.w3c.dom.Element.class, type
						.getQName(), new ElementSerializerFactory(),
						new ElementDeserializerFactory());

			}
		}
	}

	// -------------------------------------------------------------------------
	// DELEGATE METHODS
	// -------------------------------------------------------------------------

	/**
	 * @return The Parser's SymbolTable
	 */
	private SymbolTable getSymbolTable() {
		return wsdlParser.getSymbolTable();
	}

	// -------------------------------------------------------------------------
	// INNER TYPES
	// -------------------------------------------------------------------------

	/**
	 * A simple bean for referencing various kinds of <code>javax.wsdl.*</code>
	 * and <code>org.apache.axis.wsdl.*</code> objects that are related to a
	 * given operation.
	 */
	private class OperationData {
		protected Service service = null;

		protected Port port = null;

		protected Binding binding = null;

		protected Operation operation = null;

		protected Parameters parameters = null;

		protected Parameter[] inParams = null;

		protected Parameter[] outParams = null;
	}
}
