/*
 * Copyright (c) 2005, John Mettraux, OpenWFE.org and Christelle Heritier
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WsInvoker.java 3334 2006-09-17 06:19:18Z jmettraux $
 */

//
// WsInvoker.java
//
// john.mettraux@openwfe.org
// Christelle Heritier
//
// generated with 
// jtmpl 1.1.01 2004/05/19 (john.mettraux@openwfe.org)
//
package openwfe.org.ws.invoker;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Vector;

import openwfe.org.ApplicationContext;
import openwfe.org.MapUtils;
import openwfe.org.ServiceException;
import openwfe.org.engine.Definitions;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.dispatch.WorkItemDispatcher;
import openwfe.org.engine.expool.ExpressionPool;
import openwfe.org.engine.workitem.Attribute;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.ListAttribute;
import openwfe.org.engine.workitem.StringAttribute;
import openwfe.org.engine.workitem.WorkItem;

import org.apache.axis.utils.XMLUtils;
import org.uddi4j.client.UDDIProxy;
import org.uddi4j.datatype.tmodel.TModel;
import org.uddi4j.response.TModelDetail;
import org.w3c.dom.Element;

/**
 * A WorkItemDispacther implementation that triggers a webservice (the
 * webservice is supposed to reply immediately, with a return value).
 * 
 * <p>
 * <font size=2>CVS Info :<br>
 * $Author: jmettraux $<br>
 * $Id: WsInvoker.java 3334 2006-09-17 06:19:18Z jmettraux $ </font>
 * 
 * @author Christelle Heritier
 * @author john.mettraux@openwfe.org
 * @author Lukas Eder
 */
public class WsInvoker

implements WorkItemDispatcher

{

	private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
			.getLogger(WsInvoker.class.getName());

	//
	// CONSTANTS & co

	/**
	 * 'wsdl' should contain the URL of the WSDL description of the webservice.
	 * If this URL is not defined or if it is invalid, the participant will try
	 * to look it up in a UDDI registry.
	 * 
	 * @see openwfe.org.ws.invoker.WsInvoker.P_UDDI_INQUIRY_URL
	 * @see openwfe.org.ws.invoker.WsInvoker.P_UDDI_T_MODEL_KEY
	 */
	public final static String P_WSDL = "wsdl";

	/**
	 * 'uddiInquiryURL' defines the URL of the registry that is to be inquired
	 * if the 'wsdl' parameter is undefined or invalid. This parameter only
	 * works together with the 'uddiTModelKey' parameter.
	 * 
	 * @see openwfe.org.ws.invoker.WsInvoker.P_UDDI_T_MODEL_KEY
	 */
	public final static String P_UDDI_INQUIRY_URL = "uddiInquiryURL";

	/**
	 * 'uddiTModelKey' defines the unique identifier of the service interface
	 * (containing a 'wsdl' description) within the associated registry.
	 * 
	 * @see openwfe.org.ws.invoker.WsInvoker.P_UDDI_INQUIRY_URL
	 */
	public final static String P_UDDI_T_MODEL_KEY = "uddiTModelKey";

	/**
	 * 'operationName' : the name of the operation of the webservice that will
	 * get invoked.
	 */
	public final static String P_OP_NAME = "operationName";

	/**
	 * The 'signature' for the call : a comma separated list of field names that
	 * will be used as parameters for the webservice call.
	 */
	public final static String P_SIGNATURE = "signature";

	/**
	 * The param 'return_to_field' represents the field into which the return
	 * value of the web service invocation is to be stored.
	 */
	public final static String P_RETURN = "return_to_field";

	/**
	 * The time in milliseconds during which a wsdl URL discovered from a UDDI
	 * registry is considered valid.
	 * <p>
	 * By default, this is five minutes.
	 */
	private static final long WSDL_CACHE_TIMEOUT = 5 * 60 * 1000;

	//
	// FIELDS

	private ApplicationContext context = null;

	private int wsdlTimeStamp = -1;

	private String wsdl = null;

	private String uddiInquiryURL = null;

	private String uddiTModelKey = null;

	private String operationName = null;

	private String signature = null;

	private String returnValue = null;

	//
	// CONSTRUCTORS

	public void init(final String name, final ApplicationContext context,
			final java.util.Map params) throws ServiceException {
		this.context = context;

		this.wsdl = MapUtils.getAsString(params, P_WSDL);

		log.info("init() wsdl set to '" + this.wsdl + "'");

		this.uddiInquiryURL = MapUtils.getAsString(params, P_UDDI_INQUIRY_URL);

		log.info("init() uddiInquiryURL set to '" + this.uddiInquiryURL + "'");

		this.uddiTModelKey = MapUtils.getAsString(params, P_UDDI_T_MODEL_KEY);

		log.info("init() uddiTModelKey set to '" + this.uddiTModelKey + "'");

		this.operationName = MapUtils.getMandatoryString(params, P_OP_NAME);

		log.info("init() operationName set to '" + this.operationName + "'");

		this.signature = MapUtils.getMandatoryString(params, P_SIGNATURE);

		log.info("init() signature set to '" + this.signature + "'");

		this.returnValue = MapUtils.getAsString(params, P_RETURN);

		log.info("init() returnValue set to '" + this.returnValue + "'");

		// the wsdl parameter is optional only if both uddi parameters are
		// defined, i.e. if a wsdl URL can be discovered.
		if (wsdl == null)
			if (uddiInquiryURL == null || uddiTModelKey == null)
				throw new IllegalArgumentException(
						"wsdl and uddi parameters cannot both be undefined");

		// if any one of the uddi parameters is defined, the other one has
		// to be defined as well.
		if (uddiInquiryURL == null && uddiTModelKey != null)
			throw new IllegalArgumentException(
					"uddiTModelKey cannot be set without uddiInquiryURL");

		if (uddiInquiryURL != null && uddiTModelKey == null)
			throw new IllegalArgumentException(
					"uddiInquiryURL cannot be set without uddiTModelKey");
	}

	//
	// METHODS from WorkItemDispatcher

	public Object dispatch(final WorkItem wi) throws DispatchingException {
		final InFlowWorkItem ifwi = (InFlowWorkItem) ((InFlowWorkItem) wi)
				.clone();

		// log.debug
		// ("dispatch() lastExpressionId : "+ifwi.getLastExpressionId());

		try {
			// create an instance with the address of the service
			// NOTE: the wsdl member is not accessed directly but resolved
			// through a UDDI registry if applicable.
			final DynamicInvoker invoker = new DynamicInvoker(getWSDL());

			String[] paramValues = null;

			// params in the participant-map.xml file

			if (this.signature != null && !this.signature.equals("")) {
				final Object[] params = this.signature.split(",");

				// the value of this parameters

				paramValues = new String[params.length];

				for (int i = 0; i < params.length; i++) {
					final String paramName = ((String) params[i]).trim();

					final Object oValue = wi.getAttributes().get(paramName);

					if (oValue == null) {
						throw new DispatchingException("Missing parameter '"
								+ paramName + "' for webservice");
					}

					final String value = oValue.toString();

					// log.debug("dispatch() params : "+ value);

					paramValues[i] = value;
				}
			}

			// invoke the service
			HashMap map = invoker.invokeMethod(operationName, paramValues);

			final String toField = this.returnValue;

			final Iterator it = map.values().iterator();
			while (it.hasNext()) {
				Object value = it.next();

				// store the result
				if (value instanceof Element) {
					ifwi.getAttributes().puts(toField,
							XMLUtils.getInnerXMLString((Element) value));
				} else if (value instanceof Element[]) {
					Element[] elements = (Element[]) value;
					ListAttribute list = new ListAttribute();
					for (int i = 0; i < elements.length; i++) {
						String string = XMLUtils.getInnerXMLString(elements[i]);
						list.ladd(i, new StringAttribute(string));
					}
					
					ifwi.getAttributes().put(toField, list);
				} else if (value instanceof Attribute) {
					ifwi.getAttributes().put(toField, (Attribute) value);
				} else {
					ifwi.getAttributes().puts(toField, "" + value);
				}
			}
			log.debug("dispatch() DONE!");
		} catch (final Throwable t) {
			log.debug("dispatch() WS invocation problem", t);

			throw new DispatchingException("WS invocation problem", t);

			// return new FatalReply
			// ("WS invocation failed", t);
		}

		//
		// reply (in own thread to let main thread reply immediately)

		(new Thread() {
			public void run() {
				// this.yield();
				// making sure the parent thread resumes a bit
				// before this thread gets into full play

				final ExpressionPool pool = Definitions
						.getExpressionPool(WsInvoker.this.context);

				try {
					pool.reply(ifwi.getLastExpressionId(), ifwi);
				} catch (final Throwable t) {
					log.warn("dispatch() reply failure after WS invocation", t);
				}
			}
		}).start();

		//
		// return something

		// return new OkReply("");
		return null;
	}

	//
	// METHODS

	/**
	 * Retrieve the correct wsdl parameter. If the WSDL_CACHE_TIMEOUT was
	 * surpassed, the invoker's wsdl attribute is considered invalid and a new
	 * query to the UDDI registry needs to be done, if such a registry is
	 * defined.
	 * 
	 * @return The updated wsdl attribute.
	 * @throws DispatchingException
	 */
	private String getWSDL() throws DispatchingException {
		if (uddiInquiryURL != null) {
			final long now = System.currentTimeMillis();
			final long time = now - wsdlTimeStamp;

			if (wsdl == null || time > WSDL_CACHE_TIMEOUT) {
				wsdl = getUDDIServiceInterface();
			}
		}

		return wsdl;
	}

	/**
	 * Retrieves the wsdl URL from a UDDI registry.
	 * 
	 * @return The wsdl URL.
	 * @throws DispatchingException
	 */
	private String getUDDIServiceInterface() throws DispatchingException {
		try {
			final UDDIProxy proxy = new UDDIProxy();
			proxy.setInquiryURL(uddiInquiryURL);

			final TModelDetail modelDetail = proxy
					.get_tModelDetail(uddiTModelKey);

			final Vector modelVector = modelDetail.getTModelVector();

			if (modelVector.size() > 0) {
				final TModel model = (TModel) modelVector.get(0);
				return model.getOverviewDoc().getOverviewURLString();
			}
		} catch (final Throwable t) {
			throw new DispatchingException(
					"uddi discovery failed for registry " + uddiInquiryURL
							+ " and key " + uddiTModelKey, t);
		}

		return null;
	}

	//
	// STATIC METHODS

}
