/**
 * Copyright (c) 2006, Lukas Eder, OpenWFE.org, EPFL
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.
 *
 * . Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 *   
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: WsInvokerTest.java 2859 2006-06-23 12:25:24Z lukas_eder $
 */
package openwfe.org.ws.invoker._test;

import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import openwfe.org.ApplicationContext;
import openwfe.org.ServiceException;
import openwfe.org.engine.dispatch.DispatchingException;
import openwfe.org.engine.dispatch.WorkItemDispatcher;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.workitem.StringMapAttribute;
import openwfe.org.ws.invoker.WsInvoker;

/**
 * A JUnit test case for testing the
 * <code>openwfe.org.ws.invoker.WsInvoker</code> and
 * <code>openwfe.org.ws.invoker.DynamicInvoker</code> classes against UDDI
 * registry inquiry and webservice invokation given a WSDL file.
 * <p>
 * These tests use the <link>http://uddi.xmethods.net/inquire</link> public
 * UDDI registry as it seems to be the only one that provides easy access to
 * WSDL files. IBM has shut down their UDDI registry, whereas SAP and Microsoft
 * do not seem to provide access WSDL files in their service interface
 * descriptions.
 * <p>
 * It is important to understand that XMethods is not a very reliable provider
 * of a UDDI registry. Hence, tests that are NOT supposed to fail may fail
 * because of a timeout. This does not mean, however, that the code is broken.
 * 
 * @author Lukas Eder
 */
public class WsInvokerTest extends TestCase {

	public static final boolean LOOKUP_WSDL_FILE = false;

	public static void main(String[] args) {
		junit.textui.TestRunner.run(WsInvokerTest.class);
	}
	
	public WsInvokerTest(String arg0) {
		super(arg0);
	}

	protected void setUp() throws Exception {
		super.setUp();
	}

	protected void tearDown() throws Exception {
		super.tearDown();
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a simple webservice with two parameters of xsd:string type and a
	 * result of xsd:string type.
	 */
	public final void testDispatchWSDL1() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.webservicex.net/globalweather.asmx?WSDL
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		map.put(WsInvoker.P_WSDL,
				"http://www.webservicex.net/globalweather.asmx?WSDL");
		map.put(WsInvoker.P_OP_NAME, "GetWeather");
		map.put(WsInvoker.P_SIGNATURE, "CityName, CountryName");
		map.put(WsInvoker.P_RETURN, "results");

		StringMapAttribute attributes = new StringMapAttribute();
		attributes.puts("CityName", "Sion");
		attributes.puts("CountryName", "Switzerland");
		dispatch(map, attributes, false);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a simple webservice with no parameters and a an array of
	 * xsd:complexType as result.
	 */
	public final void testDispatchWSDL2() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.xmethods.net/wsdl/query.wsdl#XMethodsQuerySoap
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		map.put(WsInvoker.P_WSDL,
				"http://www.xmethods.net/wsdl/query.wsdl#XMethodsQuerySoap");
		map.put(WsInvoker.P_OP_NAME, "getAllServiceNames");
		map.put(WsInvoker.P_SIGNATURE, "");
		map.put(WsInvoker.P_RETURN, "results");

		StringMapAttribute attributes = new StringMapAttribute();
		dispatch(map, attributes, false);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a simple webservice with one parameter and a an array of
	 * xsd:complexType as result.
	 */
	public final void testDispatchWSDL3() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.xmethods.net/wsdl/query.wsdl#XMethodsQuerySoap
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		map.put(WsInvoker.P_WSDL,
				"http://www.ghettodriveby.com/soap/?wsdl#RandomGoogleSearchBinding");
		map.put(WsInvoker.P_OP_NAME, "getRandomGoogleSearch");
		map.put(WsInvoker.P_SIGNATURE, "word");
		map.put(WsInvoker.P_RETURN, "results");

		StringMapAttribute attributes = new StringMapAttribute();
		attributes.puts("word", "youpi");
		dispatch(map, attributes, false);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a simple webservice with two parameters of xsd:string type and a
	 * result of xsd:string type.
	 */
	public final void testDispatchUDDI1() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.xmethods.net/sd/2001/BabelFishService.wsdl#BabelFishBinding
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		if (LOOKUP_WSDL_FILE) {
			map.put(WsInvoker.P_UDDI_INQUIRY_URL,
					"http://uddi.xmethods.net/inquire");
			map.put(WsInvoker.P_UDDI_T_MODEL_KEY,
					"uuid:E00104D5-2AC8-9DEA-EF4C-8BD920E1B4DD");
		} else {
			map
					.put(WsInvoker.P_WSDL,
							"http://www.xmethods.net/sd/2001/BabelFishService.wsdl#BabelFishBinding");
		}
		map.put(WsInvoker.P_OP_NAME, "BabelFish");
		map.put(WsInvoker.P_SIGNATURE, "translationmode, sourcedata");
		map.put(WsInvoker.P_RETURN, "results");

		StringMapAttribute attributes = new StringMapAttribute();
		attributes.puts("translationmode", "pt_en");
		attributes.puts("sourcedata", "quero falar muito bem portugues");

		dispatch(map, attributes, false);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a simple webservice with two parameters of xsd:string type and a
	 * result of xsd:string type.
	 * 
	 * It is equivalent to testDispatchUDDI1() except that the WS invokation is
	 * supposed to fail because there is a missing parameter.
	 */
	public final void testDispatchUDDI1fail1() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.xmethods.net/sd/2001/BabelFishService.wsdl#BabelFishBinding
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		if (LOOKUP_WSDL_FILE) {
			map.put(WsInvoker.P_UDDI_INQUIRY_URL,
					"http://uddi.xmethods.net/inquire");
			map.put(WsInvoker.P_UDDI_T_MODEL_KEY,
					"uuid:E00104D5-2AC8-9DEA-EF4C-8BD920E1B4DD");
		} else {
			map
					.put(WsInvoker.P_WSDL,
							"http://www.xmethods.net/sd/2001/BabelFishService.wsdl#BabelFishBinding");
		}
		map.put(WsInvoker.P_OP_NAME, "BabelFish");
		map.put(WsInvoker.P_SIGNATURE, "translationmode");
		map.put(WsInvoker.P_RETURN, "results");

		StringMapAttribute attributes = new StringMapAttribute();
		attributes.puts("translationmode", "pt_en");

		dispatch(map, attributes, true);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a simple webservice with two parameters of xsd:string type and a
	 * result of xsd:string type.
	 * 
	 * It is equivalent to testDispatchUDDI1() except that the WS invokation is
	 * supposed to fail because the operation name is invalid.
	 */
	public final void testDispatchUDDI1fail2() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.xmethods.net/sd/2001/BabelFishService.wsdl#BabelFishBinding
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		if (LOOKUP_WSDL_FILE) {
			map.put(WsInvoker.P_UDDI_INQUIRY_URL,
					"http://uddi.xmethods.net/inquire");
			map.put(WsInvoker.P_UDDI_T_MODEL_KEY,
					"uuid:E00104D5-2AC8-9DEA-EF4C-8BD920E1B4DD");
		} else {
			map
					.put(WsInvoker.P_WSDL,
							"http://www.xmethods.net/sd/2001/BabelFishService.wsdl#BabelFishBinding");
		}
		map.put(WsInvoker.P_OP_NAME, "BabelFishForPresident");
		map.put(WsInvoker.P_SIGNATURE, "translationmode, sourcedata");
		map.put(WsInvoker.P_RETURN, "results");

		StringMapAttribute attributes = new StringMapAttribute();
		attributes.puts("translationmode", "pt_en");
		attributes.puts("sourcedata", "quero falar muito bem portugues");

		dispatch(map, attributes, true);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a more advanced webservice with no parameters but a result of
	 * type array of xsd:complextype.
	 */
	public final void testDispatchUDDI2() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// http://www.xmethods.net/wsdl/query.wsdl#XMethodsQuerySoap
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		if (LOOKUP_WSDL_FILE) {
			map.put(WsInvoker.P_UDDI_INQUIRY_URL,
					"http://uddi.xmethods.net/inquire");
			map.put(WsInvoker.P_UDDI_T_MODEL_KEY,
					"uuid:A29C0D6C-5529-0D27-A91A-8E02D343532B");
		} else {
			map
					.put(WsInvoker.P_WSDL,
							"http://www.xmethods.net/wsdl/query.wsdl#XMethodsQuerySoap");
		}
		map.put(WsInvoker.P_OP_NAME, "getAllServiceNames");
		map.put(WsInvoker.P_SIGNATURE, "");
		map.put(WsInvoker.P_RETURN, "results");

		dispatch(map, new StringMapAttribute(), false);
	}

	/**
	 * Test method for 'openwfe.org.ws.invoker.WsInvoker.dispatch(WorkItem)'
	 * 
	 * It uses a key that does not exist, hence the test should fail.
	 */
	public final void testDispatchUDDI3fail() throws ServiceException,
			DispatchingException {

		// ---------------------------------------------------------------------
		// ???
		// ---------------------------------------------------------------------

		Map map = new HashMap();
		map.put(WsInvoker.P_UDDI_INQUIRY_URL,
				"http://uddi.xmethods.net/inquire");
		map.put(WsInvoker.P_UDDI_T_MODEL_KEY,
				"uuid:00000000-0000-0000-0000-000000000000");
		map.put(WsInvoker.P_OP_NAME, "");
		map.put(WsInvoker.P_SIGNATURE, "");
		map.put(WsInvoker.P_RETURN, "");

		dispatch(map, new StringMapAttribute(), true);
	}

	/**
	 * Execute the dispatching.
	 * 
	 * @param participantParameters
	 *            The parameters of the participant.
	 * @param workItemAttributes
	 *            The attributes contained in the work item that is dispatched.
	 * @param shouldFail
	 *            Whether an exception should be raised upon dispatching.
	 * @throws ServiceException
	 * @throws DispatchingException
	 */
	private void dispatch(Map participantParameters,
			StringMapAttribute workItemAttributes, boolean shouldFail)
			throws ServiceException, DispatchingException {

		InFlowWorkItem wi = new InFlowWorkItem(workItemAttributes);
		WorkItemDispatcher invoker = new WsInvoker();
		ApplicationContext context = null;

		invoker.init("", context, participantParameters);

		try {
			invoker.dispatch(wi);
			if (shouldFail)
				fail("There should have been a DispatchingException upon ws invokation");
		} catch (DispatchingException e) {
			if (!shouldFail)
				throw e;
		}
	}
}