/*
 * Copyright (c) 2005-2006, Christelle Heritier, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * * $Id: SoapWorkSession.java 3222 2006-09-04 09:22:07Z jmettraux $
 */
package openwfe.org.ws.worklist;

import java.rmi.RemoteException;
import java.util.ArrayList;

import openwfe.org.OpenWfeException;
import openwfe.org.ServiceException;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.worklist.WorkListException;
import openwfe.org.ws.worklist.impl.NotAuthentifiedException;


public interface SoapWorkSession 
{

    /**
     * Return true if the user is authentified correctly
     */
    public boolean authentify 
        (final String username, final String password)
    throws 
        ServiceException;


    /**
     * Lists the store names this worklist manages 
     */
    public java.util.ArrayList getStoreNames () 
        throws RemoteException, NotAuthentifiedException;

    /**
     * Returns the count of workitems in the given store
     */
    public int countWorkItems (String storeName) 
        throws RemoteException, WorkListException;

    /**
     * Returns the flow expression ids of all the workitems belonging to the
     * given workflow instance. This workflowInstanceId is the id given as
     * the return value of the launch() method
     */
    public java.util.ArrayList findFlowInstance
        (String storeName, String workflowInstanceId) 
    throws 
        RemoteException, WorkListException;

    /**
     * Returns the headers of the workitems in the store. You can set a
     * limit if you don't want to much headers to come back.
     */
    public java.util.ArrayList getHeaders (String storeName, int limit)
        throws RemoteException, WorkListException;

    /**
     * Retrieves a workitem from a store
     */
    public InFlowWorkItem get (String storeName, FlowExpressionId fei)
        throws RemoteException, WorkListException;

    /**
     * Retrieves a workitem from a store
     */
    public InFlowWorkItem get (String storeName, String id)
        throws RemoteException, WorkListException;

    /**
     * Retrieves a workitem from a store and ensure that you will be the
     * only able to tinker with it.
     */
    public InFlowWorkItem getAndLock (String storeName, FlowExpressionId fei)
        throws RemoteException, WorkListException;

    /**
     * Retrieves a workitem from a store and ensure that you will be the
     * only able to tinker with it.
     */
    public InFlowWorkItem getAndLock (String storeName, String id)
        throws RemoteException, WorkListException;

    /**
     * Unlocks a workitem
     */
    public void release (String storeName, InFlowWorkItem item)
        throws RemoteException, WorkListException;

    /**
     * Saves a workitem (the modifications you made to it)
     */
    public void save (String storeName, InFlowWorkItem item)
        throws RemoteException, WorkListException;

    /**
     * Gives the workitem back to the worklist you feeds it back to workflow
     * engine
     */
    public void forward (String storeName, InFlowWorkItem item)
        throws RemoteException, OpenWfeException, NotAuthentifiedException;

    /**
     * Returns a list of flows the user is entitled to launch
     */
    public ArrayList getLaunchables () 
        throws RemoteException, WorkListException;

    /**
     * Launches a flow in a given engine, with a launchitem containing info
     * about which flow to launch
     * 
     * @return the workflow instance id of the newly created flow
     */
    public String launch (String engineId, LaunchItem item)
        throws RemoteException, OpenWfeException, NotAuthentifiedException;

    /**
     * Delegates a workitem from one store to the other in the same
     * worklist.
     */
    public void delegate 
        (String storeName, InFlowWorkItem item, String targetStoreName) 
    throws 
        RemoteException, WorkListException; 

    /**
     * Another type of delegation : when you delegate directly to a
     * participant. This method lists the participant the user/client has
     * the right to delegate to (with the next method).
     */
    public java.util.ArrayList getParticipantsForDelegation ()
        throws RemoteException, WorkListException;

    /**
     * The method to call for direct delegation.
     */
    public void delegateToParticipant
        (String storeName, InFlowWorkItem item, String participantName) 
    throws 
        RemoteException, WorkListException;

    //
    // info methods

    /**
     * Checks the right you currently have (as owner of a worksession) on a
     * given storeName. Action names may be read, write, delete or delegate.
     */
    public boolean checkPermission (String storeName, String action)
        throws RemoteException, WorkListException;

}

