/*
 * Copyright (c) 2005-2006, Christelle Heritier, OpenWFE.org
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 * . Redistributions of source code must retain the above copyright notice, this
 *   list of conditions and the following disclaimer.  
 * 
 * . Redistributions in binary form must reproduce the above copyright notice, 
 *   this list of conditions and the following disclaimer in the documentation 
 *   and/or other materials provided with the distribution.
 * 
 * . Neither the name of the "OpenWFE" nor the names of its contributors may be
 *   used to endorse or promote products derived from this software without
 *   specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: RmiSoapWorkSession.java 3118 2006-08-30 14:38:36Z jmettraux $
 */

package openwfe.org.ws.worklist.impl;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import javax.security.auth.Subject;
import org.apache.axis.MessageContext;
import org.apache.axis.session.Session;

import openwfe.org.Service;
import openwfe.org.OpenWfeException;
import openwfe.org.ApplicationContext;
import openwfe.org.ServiceException;
import openwfe.org.engine.workitem.LaunchItem;
import openwfe.org.engine.workitem.InFlowWorkItem;
import openwfe.org.engine.expressions.FlowExpressionId;
import openwfe.org.rmi.session.WorkSessionServer;
import openwfe.org.rmi.session.WorkSessionServerImpl;
import openwfe.org.worklist.Header;
import openwfe.org.worklist.Launchable;
import openwfe.org.worklist.WorkSession;
import openwfe.org.worklist.WorkListException;
import openwfe.org.ws.worklist.SoapWorkSession;


public class RmiSoapWorkSession implements SoapWorkSession
{
	private final static org.apache.log4j.Logger log = org.apache.log4j.Logger
			.getLogger(RmiSoapWorkSession.class.getName());

	//
	// FIELDS

	private ApplicationContext applicationContext =  null;
	private WorkSession session;
	private java.util.Map serviceParams = null;
	private final static String SOAP_WORKSESSION_SERVER = "local.workSessionServer";
	private final static String SESSION_ID = "sessionID";
	private Subject subject = null;
	private java.util.Map ids = new java.util.HashMap(77);
	private Session axisSession;
	
	// CONSTRUCTOR
	public RmiSoapWorkSession() throws RemoteException, ServiceException {
		super();
		this.applicationContext = ApplicationContext.lookupContext("openwfe-worklist");
	}
	

	//
	// METHODS from WorkSession

	//
	// non remote methods

	public void init(final ApplicationContext context,
			final java.util.Map params, final Subject subject)
			throws RemoteException {
		this.applicationContext = context;
		this.serviceParams = params;
		this.subject = subject;
	}
	
	public Subject getSubject()throws RemoteException{
		return this.subject;
	}

	public ApplicationContext getApplicationContext() {
		return this.applicationContext;
	}
     
	
	
	//Test there is only one Session
	/*
	public String Hello() throws Exception {
		axisSession = MessageContext.getCurrentContext().getSession();
		if(axisSession==null){
			log.debug("\n axisSession is null \n");
		}
		String temp = (String)axisSession.get("counter");
		if(temp == null){
			temp="bonjour ";
		}
		temp = temp+"a";
		axisSession.set("counter", temp);
		return temp;
	}
	*/
	
	
	/**
	 * Returns the WorkSessionServer instance
	 */
	 protected WorkSessionServer getWorkSessionServer ()
     throws Exception
 {
     WorkSessionServer result = (WorkSessionServer)this.applicationContext.lookup(SOAP_WORKSESSION_SERVER);

     if (result != null) return result;

     // instantiating the workSessionServer
 
     result = new WorkSessionServerImpl();

     ((Service)result).init
         (SOAP_WORKSESSION_SERVER,
          this.applicationContext,
          this.serviceParams);

     this.applicationContext
         .put(SOAP_WORKSESSION_SERVER, result);

     return result;
 }
	 /**
	  * Return true if the user is authentified correctly
	 */
	 public boolean authentify 
             (final String username, final String password)
         throws 
             ServiceException 
         {
		try
		{
			this.axisSession = MessageContext.getCurrentContext().getSession();
			this.session = (WorkSession) getWorkSessionServer().login(username, password);
		}
		catch (final Exception e){
			throw new ServiceException("login failed", e);
		}
		this.axisSession.set("SESSION_ID",this.session);
		return true;
	}
	
	 /**
	  * Lists the store names this worklist manages 
	  */
	public java.util.ArrayList getStoreNames() throws RemoteException,
			NotAuthentifiedException {
		this.session = (WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		return (ArrayList) this.session.getStoreNames();
	}
	
	/**
	 * Returns the count of workitems in the given store
	 */
	public int countWorkItems(String storeName) throws RemoteException, WorkListException {
		//first get the good session 
		this.session = (WorkSession)axisSession.get("SESSION_ID");

		if(session == null)
			throw new NotAuthentifiedException();

		//method countWorkItems from WorkSession
		return this.session.countWorkItems(storeName);
		}


	/**
	 * Returns the flow expression ids of all the workitems belonging to the
	 * given workflow instance. This workflowInstanceId is the id given as
	 * the return value of the launch() method
	 */
	public java.util.ArrayList findFlowInstance(final String storeName,
			final String workflowInstanceId) throws RemoteException,
			WorkListException{
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		final java.util.ArrayList feis = (java.util.ArrayList)this.session.findFlowInstance(storeName, workflowInstanceId) ;
		registerIds(feis);	
		return (java.util.ArrayList)this.session.findFlowInstance(storeName, workflowInstanceId);
	}

	/**
	 * Returns the headers of the workitems in the store. You can set a
	 * limit if you don't want to much headers to come back.
	 */
	public java.util.ArrayList getHeaders(final String storeName,
			final int limit) throws RemoteException, WorkListException{
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		final java.util.List headers = (java.util.List)this.session.getHeaders(storeName, limit);
		registerIds(headers);
		ArrayList result = new ArrayList(headers.size());
		//transform each Header into a String
		for(int i=0;i<headers.size() ;i++){
			result.add(i, ((Header)headers.get(i)).toString());
		}
		return result;
	}

	/**
	 * Retrieves a workitem from a store
	 */
	public InFlowWorkItem get(final String storeName, final FlowExpressionId fei)
			throws RemoteException, WorkListException {
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		return (InFlowWorkItem)this.session.get(storeName, fei);
	}

	/**
	 * Retrieves a workitem from a store
	 */
	public InFlowWorkItem get(final String storeName, final String id) throws RemoteException, WorkListException {
		final FlowExpressionId fei = (FlowExpressionId)this.ids.get(id);
				return get(storeName, fei);
	}

	/**
	 * Retrieves a workitem from a store and ensure that you will be the
	 * only able to tinker with it.
	 */	
	public InFlowWorkItem getAndLock(final String storeName,
			final FlowExpressionId fei) throws RemoteException,
			WorkListException{
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session ==null)
			throw new NotAuthentifiedException();

		return this.session.getAndLock(storeName, fei);
	}

	/**
	 * Retrieves a workitem from a store and ensure that you will be the
	 * only able to tinker with it.
	 */
	public InFlowWorkItem getAndLock(final String storeName, final String id)
			throws RemoteException, WorkListException {
		final FlowExpressionId fei = (FlowExpressionId) this.ids.get(id);
		return getAndLock(storeName, fei);
	}

	/**
	 * unlocks a workitem
	 */
	public void release(final String storeName, final InFlowWorkItem item)
			throws RemoteException, WorkListException {
		this.session=(WorkSession) axisSession.get("SESSION_ID");
		if (session == null)
			throw new NotAuthentifiedException();

		this.session.release(storeName, item);
	}

	/**
	 * saves a workitem (the modifications you made to it)
	 */
	public void save(final String storeName, final InFlowWorkItem item)
			throws RemoteException, WorkListException{
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		this.session.save(storeName, item);
	}

	
	/**
	 * gives the workitem back to the worklist you feeds it back to workflow
	 * engine
	 */
	public void forward(final String storeName, final InFlowWorkItem item)
			throws RemoteException, OpenWfeException, NotAuthentifiedException {
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		this.session.forward(storeName, item);
	}

	/**
	 * This method returns a list of Launchable instances, ie objects
	 * pointing to workflow definitions the user is authorized to launch.
	 */
	public ArrayList getLaunchables() throws RemoteException,
			WorkListException {
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		List list = this.session.getLaunchables();
		ArrayList result = new ArrayList(list.size());
		//transform each Launchable into a String
		for(int i=0;i<list.size() ;i++){
			result.add(i,((Launchable)list.get(i)).toString());
		}
		return result;
	}

	/**
	 * Launches a flow in a given engine, with a launchitem containing info
	 * about which flow to launch
	 * 
	 * @return the workflow instance id of the newly created flow
	 */
	public String launch(final String engineId, final LaunchItem item)
			throws RemoteException, OpenWfeException, NotAuthentifiedException  {
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		return this.session.launch(engineId, item);		
	}
	
	/**
	 * Delegates a workitem from one store to the other in the same
	 * worklist.
	 */
	public void delegate(String storeName, InFlowWorkItem item,
			String targetStoreName) throws RemoteException, WorkListException{
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		this.session.delegate(storeName, item, targetStoreName);
	}

	/**
	 * Another type of delegation : when you delegate directly to a
	 * participant. This method lists the participant the user/client has
	 * the right to delegate to (with the next method).
	 */
	public java.util.ArrayList getParticipantsForDelegation()
			throws RemoteException, WorkListException{
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		return (java.util.ArrayList)this.session.getParticipantsForDelegation();
	}

	/**
	 * The method to call for direct delegation.
	 */
	public void delegateToParticipant(String storeName, InFlowWorkItem item,
			String participantName) throws RemoteException, WorkListException {	
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		this.session.delegateToParticipant(storeName, item,participantName);
	}

	/**
	 * Checks the right you currently have (as owner of a worksession) on a
	 * given storeName. Action names may be read, write, delete or delegate.
	 */
	public boolean checkPermission(String storeName, String action)
			throws RemoteException, WorkListException {
		this.session=(WorkSession) axisSession.get("SESSION_ID");

		if (session == null)
			throw new NotAuthentifiedException();

		return this.session.checkPermission(storeName, action);
	}

	//
	// OTHER METHODS

	/**
	 * Keeps a map of the ' flow expression ids with the 
	 * string id as key and with the workitem as values.
	 */
	private void registerIds(final java.util.List l) {
		final java.util.Iterator it = l.iterator();
		while (it.hasNext()) {
			final Object o = it.next();

			FlowExpressionId fei = null;

			if (o instanceof Header) {
				fei = ((Header) o).getExpressionId();
			} else if (o instanceof FlowExpressionId) {
				fei = (FlowExpressionId) o;
			} else {
				throw new IllegalArgumentException(
						"Couldn't register flow expression id "
								+ "out of object of class "
								+ o.getClass().getName());
			}

			this.ids.put(fei.getWorkflowInstanceId(), fei);
		}
	}

}

