package org.codehaus.plexus.formica.util;

/*
 * Copyright 2004-2005 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.InputStream;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.PasswordAuthentication;
import java.net.URL;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

/**
 * This https implementation ignores certificates and processes urls of the format
 * 
 * https://[username:password@]host.com/
 * 
 * @author <a href="mailto:jesse.mcconnell@gmail.com">Jesse McConnell</a>
 * @version $Id: MungedHttpsURL.java 2442 2005-09-07 09:38:25Z evenisse $
 */
public class MungedHttpsURL
{
    // ----------------------------------------------------------------------
    // urlString to make the Https connection to
    // ----------------------------------------------------------------------

    private String urlString;

    private String username;

    private String password;

    // ----------------------------------------------------------------------
    // Constructor
    // ----------------------------------------------------------------------

    public MungedHttpsURL( String mungedUrl )
        throws MalformedURLException
    {
       username = scrapeUsername( mungedUrl );

       password = scrapePassword( mungedUrl );

       urlString = scrapeUrl( mungedUrl );

       if ( urlString == null )
       {
           throw new MalformedURLException( "Unable to generate clean url from url string: " + mungedUrl );
       }
       
    }

    public MungedHttpsURL( String urlString, String username, String password )
        throws MalformedURLException
    {
       this.username = username;

       this.password = password;

       this.urlString = urlString;

       if ( !isValid() )
       {
           throw new MalformedURLException( "Unable to validate URL" );
       }
    }
    
    // ----------------------------------------------------------------------
    // HttpsURL Implementation
    // ----------------------------------------------------------------------

    /**
     * this bit attempts to ignore certificates that might need to be accepted and also tries to 
     * conform to the format https://[<username>:<password>@]host
     *  
     * adapted from code snippets from http://javaalmanac.com/egs/javax.net.ssl/TrustAll.html
     */
    public boolean isValid()
    {
        try
        {
            URL url = getURL();

            InputStream is = url.openStream();

            is.close();
        }
        catch ( Exception e )
        {
           return false;
        }

        return true;
    }

    public URL getURL()
        throws MalformedURLException
    {
        try
        {
            if ( urlString.startsWith( "https" ) )
            {
                return getHttpsUrl();
            }
            else
            {
                return getHttpUrl();
            }

        }
        catch ( Exception e )
        {
            throw new MalformedURLException( "unable to create munged url" );
        }
    }

    private URL getHttpUrl() throws MalformedURLException
    {
        if ( username != null && password != null )
        {
            Authenticator.setDefault( new Authenticator()
            {
                protected PasswordAuthentication getPasswordAuthentication()
                {
                    PasswordAuthentication passwordAuthentication = new PasswordAuthentication( username, password.toCharArray() );

                    return passwordAuthentication;
                }
            } );
        }

        return new URL( urlString );
    }

    private URL getHttpsUrl()
            throws Exception
    {
        ignoreCertificates();

        String authString = username + ":" + password;

        URL url = new URL( urlString );

        HttpURLConnection urlc = (HttpURLConnection) url.openConnection();

        urlc.setDoInput( true );

        urlc.setUseCaches( false );

        urlc.setRequestProperty( "Content-Type", "application/octet-stream" );

        if ( username != null && password != null )
        {
            urlc.setRequestProperty( "Authorization", "Basic " +
                new sun.misc.BASE64Encoder().encode( authString.getBytes() ) );
        }

        return url;
    }

    public String getUsername()
    {
        return username;
    }

    public String getPassword()
    {
        return password;
    }

    public String getUrlString()
    {
        return urlString;
    }

    // ----------------------------------------------------------------------
    // helper methods
    // ----------------------------------------------------------------------

    /**
     * setup the environment to ignore all certificates for the connection
     */
    private void ignoreCertificates()
        throws Exception
    {
        // Create a trust manager that does not validate certificate
        // chains
        TrustManager[] trustAllCerts = new TrustManager[]
        {
            new X509TrustManager()
            {
                public java.security.cert.X509Certificate[] getAcceptedIssuers()
                {
                    return null;
                }

                public void checkClientTrusted( java.security.cert.X509Certificate[] certs, String authType )
                {
                }

                public void checkServerTrusted( java.security.cert.X509Certificate[] certs, String authType )
                {
                }
            }
        };

        // Install the all-trusting trust manager

        SSLContext sslContext = SSLContext.getInstance( "SSL" );

        sslContext.init( null, trustAllCerts, new java.security.SecureRandom() );

        HttpsURLConnection.setDefaultSSLSocketFactory( sslContext.getSocketFactory() );
    }

    /**
     * return the username from https://<name>:<password>@<url>
     */
    private String scrapeUsername( String url )
    {
        String t = url.substring( url.lastIndexOf('/')+1, url.length() );

        if ( t.indexOf( ":" ) < 0 && t.indexOf( "@" ) < 0 )
        {
            return null;
        }

        return t.substring( 0, t.indexOf( ":" ) );
    }

    /**
     * return to password from https://<name>:<password>@<url>
     */
    private String scrapePassword( String url )
    {
        String t = url.substring( url.lastIndexOf('/')+1, url.length() );

        if ( t.indexOf( ":" ) < 0 && t.indexOf( "@" ) < 0 )
        {
            return null;
        }

       return t.substring( t.indexOf( ":" ) + 1, t.indexOf( "@" ) );
    }

    /**
     * return the url from https://<name>:<password>@<url>
     */
    private String scrapeUrl( String url )
    {
        if ( url.indexOf( "@" ) != -1 )
        {
            String s = url.substring( url.indexOf( "@" ) + 1, url.length() );
            if ( url.startsWith( "http://" ) )
            {
                return "http://" + s;
            }
            else
            {
                return "https://" + s;
            }
        }
        else
        {
            return url;
        }
    }
}
