package com.aliyun.openservices.iot.api;

import com.google.common.collect.Maps;
import lombok.AccessLevel;
import lombok.Data;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Map;
import java.util.Optional;

/**
 * @author brhao
 * @date 02/04/2018
 */
@Data
@Slf4j
public class Profile {
    public static final boolean ENABLE_SSL = true;

    public static final int DEFAULT_PORT = 443;

    private String endPoint;
    @Setter(AccessLevel.NONE)
    private String host;
    @Setter(AccessLevel.NONE)
    private int port;
    private boolean cleanSession;
    private int heartBeatInterval = 30 * 1000;
    private int heartBeatTimeOut = 60 * 1000;
    private boolean multiConnection = false;

    private int callbackThreadCorePoolSize = Runtime.getRuntime().availableProcessors();
    private int callbackThreadMaximumPoolSize = Runtime.getRuntime().availableProcessors() * 4;
    private int callbackThreadBlockingQueueSize = 1024;

    private Map<String, String> authParams;

    public String getHost() {
        return url().map(URL::getHost).orElse(null);
    }

    public int getPort() {
        return url().map(URL::getPort).filter(p -> p!=-1).orElse(DEFAULT_PORT);
    }

    private Optional<URL> url() {
        try {
            URL url = new URL(endPoint);
            return Optional.of(url);
        } catch (MalformedURLException e) {
            log.error("fail to parse url {},{}", endPoint, e.getMessage());
        }
        return Optional.empty();
    }

    private Profile(String endPoint) {
        this.endPoint = endPoint;
        authParams = Maps.newHashMap();
    }

    public static Profile getDeviceProfile(String endPoint, String productKey, String deviceName, String deviceSecret, String clientId) {
        Profile profile = new Profile(endPoint);
        profile.authParams.put(Constraint.AUTH_TYPE, Constraint.AUTH_TYPE_DEVICE_NAME);
        profile.authParams.put(Constraint.PARAM_PRODUCT_KEY, productKey);
        profile.authParams.put(Constraint.PARAM_DEVICE_NAME, deviceName);
        profile.authParams.put(Constraint.PARAM_DEVICE_SECRET, deviceSecret);
        profile.authParams.put(Constraint.PARAM_CLIENT_ID, clientId);
        profile.multiConnection = false;
        return profile;
    }

    public static Profile getAppKeyProfile(String endPoint, String appKey, String appSecret) {
        Profile profile = new Profile(endPoint);
        profile.authParams.put(Constraint.AUTH_TYPE, Constraint.AUTH_TYPE_APP_KEY);
        profile.authParams.put(Constraint.PARAM_APP_KEY, appKey);
        profile.authParams.put(Constraint.PARAM_APP_SECRET, appSecret);
        profile.multiConnection = true;
        return profile;
    }

    public static Profile getAccessKeyProfile(String endPoint, String regionId, String accessKey, String accessSecret) {
        return getAccessKeyProfile(endPoint, regionId, accessKey, accessSecret, null);
    }

    public static Profile getAccessKeyProfile(String endPoint, String regionId, String accessKey, String accessSecret, String stsToken) {
        Profile profile = new Profile(endPoint);
        profile.authParams.put(Constraint.AUTH_TYPE, Constraint.AUTH_TYPE_ACCESS_KEY);
        profile.authParams.put(Constraint.PARAM_ACCESS_KEY, accessKey);
        profile.authParams.put(Constraint.PARAM_ACCESS_SECRET, accessSecret);
        profile.authParams.put(Constraint.REGION_ID, regionId);
        profile.authParams.put(Constraint.PARAM_STS_TOKEN, stsToken);
        profile.multiConnection = true;
        return profile;
    }
}