package com.aliyun.openservices.iot.api.auth;

import com.aliyun.openservices.iot.api.Profile;
import com.aliyun.openservices.iot.api.auth.handler.AppKeyAuthHandler;
import com.aliyun.openservices.iot.api.auth.handler.DeviceAuthHandler;
import com.aliyun.openservices.iot.api.auth.handler.accesskey.AccessKeyAuthHandler;
import com.aliyun.openservices.iot.api.exception.IotClientException;
import com.google.common.collect.Maps;

import java.util.Map;

import static com.aliyun.openservices.iot.api.Constraint.*;

/**
 * Authentication handler factory
 * support deviceName, appkey and aliyunAK
 *
 * @author brhao
 * @date 13/06/2018
 */
public class AuthenticationFactory {
    private static Map<String, AuthHandler> authHandlers = Maps.newHashMap();

    public static AuthHandler getAuthHandler(Profile profile) {
        Map<String, String> params = profile.getAuthParams();
        if (params == null || params.isEmpty()) {
            throw new IotClientException("authorization parameter is empty");
        }

        String authType = profile.getAuthParams().get(AUTH_TYPE);
        if (AUTH_TYPE_ACCESS_KEY.equals(authType)) {
            return new AccessKeyAuthHandler(params);
        } else if (AUTH_TYPE_DEVICE_NAME.equals(authType)) {
            return new DeviceAuthHandler(params);
        } else if (AUTH_TYPE_APP_KEY.equals(authType)) {
            return new AppKeyAuthHandler(params);
        } else if (authHandlers.containsKey(authType)) {
            return authHandlers.get(authType);
        } else {
            throw new IotClientException("unsupported auth type: " + authType);
        }
    }

    public static void registerAuthHandler(String authType, AuthHandler authHandler) {
        authHandlers.put(authType, authHandler);
    }

    public static void unregisterAuthHandler(String authType) {
        authHandlers.remove(authType);
    }
}
