package com.aliyun.openservices.iot.api.http2;

import com.aliyun.openservices.iot.api.Profile;
import com.aliyun.openservices.iot.api.auth.AuthHandler;
import com.aliyun.openservices.iot.api.auth.AuthenticationFactory;
import com.aliyun.openservices.iot.api.http2.callback.AbstractHttp2StreamDataReceiver;
import com.aliyun.openservices.iot.api.http2.connection.*;
import com.aliyun.openservices.iot.api.http2.connection.impl.ConnectionManagerImpl;
import com.aliyun.openservices.iot.api.http2.entity.Http2Response;
import com.aliyun.openservices.iot.api.http2.entity.StreamData;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.netty.handler.codec.http2.DefaultHttp2Headers;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.handler.codec.http2.Http2Stream;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * iot client
 * communicate with http2 server
 *
 * @author brhao
 * @date 16/03/2018
 */
@Slf4j
public class IotHttp2Client {
    public static final int CONNECTION_COUNT_UNLIMITED = -1;
    private final SocketAddress SOCKET_ADDRESS;
    private static final char SETTINGS_SUBSCRIBE_CONNECTION_COUNT = 'c';
    private static long[] reconnectInterval = new long[]{0, 1000, 10 * 1000, 60 * 1000, 60 * 10 * 1000};
    private int reconnectFailedCount = 0;

    private AtomicInteger connectionCount = new AtomicInteger(1);
    private int maxConnectionCount = CONNECTION_COUNT_UNLIMITED;

    private ScheduledExecutorService executorService;
    private AtomicBoolean started = new AtomicBoolean(false);

    private ConnectionManager connectionManager;
    private AuthHandler authHandler;

    public IotHttp2Client(Profile profile, int maxConnectionCount) {
        SOCKET_ADDRESS = new InetSocketAddress(profile.getHost(), profile.getPort());
        connectionManager = new ConnectionManagerImpl(Profile.ENABLE_SSL, profile.getHeartBeatInterval(),
                profile.getHeartBeatTimeOut());
        connectionManager.addConnectionListener(connectionSettingListener);
        this.maxConnectionCount = maxConnectionCount;
        executorService = new ScheduledThreadPoolExecutor(1,
                new ThreadFactoryBuilder().setDaemon(true).setNameFormat("iot-client-thread-%d").build());
        authHandler = AuthenticationFactory.getAuthHandler(profile);
    }

    private ConnectionListener connectionSettingListener = new ConnectionListener() {
        @Override
        public void onSettingReceive(Connection connection, Http2Settings settings) {
            if (!settings.containsKey(SETTINGS_SUBSCRIBE_CONNECTION_COUNT)) {
                return;
            }
            int value = settings.getIntValue(SETTINGS_SUBSCRIBE_CONNECTION_COUNT);
            if (maxConnectionCount != CONNECTION_COUNT_UNLIMITED && maxConnectionCount >= 0) {
                value = Math.min(maxConnectionCount, value);
                log.info("maxConnectionCount: {}, server setting: {}", maxConnectionCount, value);
            }
            connectionCount.set(value);
            log.info("receive setting, connection: {}, subscription count : {} ", connection,
                    connectionCount);
        }

        @Override
        public void onStatusChange(ConnectionStatus status, Connection connection) {
            log.info("connection status changed, connection: {}, status: {}", connection, status);

            // when connection is authorized, add connection monitor
            if (status == ConnectionStatus.AUTHORIZED && started.compareAndSet(false, true)) {
                if (executorService != null) {
                    executorService.scheduleWithFixedDelay(IotHttp2Client.this::updateConnectionCount, 10000, 10000,
                            TimeUnit.MILLISECONDS);
                }
            }

            // when authorized connection lost, reconnect immediately
            if (connection.getStatus() == ConnectionStatus.AUTHORIZED && status == ConnectionStatus.CLOSED) {
                executorService.submit(IotHttp2Client.this::updateConnectionCount);
            }
        }
    };

    private void updateConnectionCount() {
        long current = allConnections().size();
        int expected = connectionCount.get();

        if (maxConnectionCount >= 0) {
            expected = Math.min(maxConnectionCount, expected);
        }

        if (current >= expected) {
            return;
        }

        log.info("update connection count, current count {}, expected count {}", current, expected);
        if (reconnectFailedCount != 0) {
            long interval = reconnectInterval[reconnectFailedCount % reconnectInterval.length];
            try {
                log.info("backoff, create connection after {}ms", interval);
                Thread.sleep(interval);
            } catch (InterruptedException e) {
                log.error("error occurs while backoff, exception: ", e);
            }
        }

        for (int i = 0; i < expected - current; i++) {
            try {
                newConnection();
            } catch (Throwable t) {
                reconnectFailedCount++;
                log.error("failed to create connection, {}", t.getMessage());
            }
        }

        long updatedCount = allConnections().size();
        if (updatedCount == expected) {
            reconnectFailedCount = 0;
        }
        log.info("finish updating connection count, current count {}", updatedCount);
    }

    /**
     * create new connection to server
     *
     * @return connection
     * @throws ExecutionException   executionException
     * @throws InterruptedException interruptedException
     */
    public Connection newConnection() throws ExecutionException, InterruptedException {
        return connectionManager.connect(SOCKET_ADDRESS).get();
    }

    /**
     * generate header with authorization parameters
     *
     * @return Http2Header
     */
    public Http2Headers authHeader() {
        Http2Headers headers = new DefaultHttp2Headers();
        Map<String, String> authParams = authHandler.getAuthParams();
        authParams.forEach((key, value) -> headers.set("x-auth-" + key, value));
        return headers;
    }

    public void shutdown() {
        log.info("shutdown http2 client");
        allConnections().forEach(Connection::removeConnectListener);
        connectionManager.removeConnectionListener(connectionSettingListener);
        connectionManager.shutdown();
        if (executorService != null) {
            executorService.shutdown();
        }
    }

    public void addConnectionListener(ConnectionListener listener) {
        connectionManager.addConnectionListener(listener);
    }

    public void removeConnectionListener(ConnectionListener listener) {
        connectionManager.removeConnectionListener(listener);
    }

    public List<Connection> allConnections() {
        return connectionManager.getConnectionList();
    }

    public CompletableFuture<Http2Response> sendRequest(Connection connection, Http2Headers headers, byte[] data) {
        CompletableFuture<Http2Response> completableFuture = new CompletableFuture<>();
        boolean endOfStream = false;
        int contentLength = 0;
        if (data == null || data.length == 0) {
            endOfStream = true;
        } else {
            contentLength = data.length;
        }

        headers.set("content-length", String.valueOf(contentLength));

        CompletableFuture<StreamWriteOperation> writeCompletableFuture = connection.writeHeaders(headers, endOfStream,
                new AbstractHttp2StreamDataReceiver() {
                    @Override
                    public void onDataRead(Connection connection, Http2Stream stream, StreamData streamData) {
                        completableFuture.complete(new Http2Response(streamData.getHeaders(), streamData.readAllData()));
                    }

                    @Override
                    public void onStreamError(Connection connection, Http2Stream stream, IOException e) {
                        completableFuture.completeExceptionally(e);
                    }
                }).whenComplete((c, throwable) -> {
            if (throwable != null) {
                completableFuture.completeExceptionally(throwable);
            }
        });

        if (data != null) {
            writeCompletableFuture.thenAccept(writeOperation -> writeOperation.writeData(data, true).whenComplete(
                    (o, t) -> {
                        if (t != null) {
                            completableFuture.completeExceptionally(t);
                        }
                    }
            ));
        }
        return completableFuture;
    }

    public Optional<Connection> randomConnection(Predicate<Connection> predicate) {
        List<Connection> authConnection = allConnections().stream().filter(predicate).collect(
                Collectors.toList());
        if (authConnection.isEmpty()) {
            return Optional.empty();
        }
        Random r = new Random();
        return Optional.of(authConnection.get(r.nextInt(authConnection.size())));
    }
}
