package com.aliyun.openservices.iot.api.http2.callback;

import com.aliyun.openservices.iot.api.http2.connection.Connection;
import com.aliyun.openservices.iot.api.http2.entity.StreamData;
import io.netty.handler.codec.http2.Http2Connection.PropertyKey;
import io.netty.handler.codec.http2.Http2Headers;
import io.netty.handler.codec.http2.Http2Stream;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;

/**
 * callback when whole stream data is received
 * cache HTTP2 Headers and DataFrame
 * stream will be closed after callback #onDataRead()
 *
 * @author brhao
 * @date 26/03/2018
 */
@Slf4j
public abstract class AbstractHttp2StreamDataReceiver implements Http2StreamListener {
    private static final String DATA_CACHE_KEY = "data_cache_key";

    @Override
    public void onDataRead(Connection connection, Http2Stream stream, byte[] data, boolean endOfStream) {
        log.debug("receive data on connection {}, streamId {}", connection, stream.id());

        StreamData streamData = getStreamData(connection, stream);
        if (streamData == null) {
            onStreamError(connection, stream, new IOException(
                    connection.toString() + " received data frame on " + stream.id() + ", but headers hasn't received"));
            return;
        }
        streamData.addData(data);

        if (endOfStream) {
            onDataRead(connection, stream, streamData);
            stream.closeLocalSide();
        }
    }

    @Override
    public void onHeadersRead(Connection connection, Http2Stream stream, Http2Headers headers, boolean endOfStream) {
        PropertyKey key = connection.getPropertyKey(DATA_CACHE_KEY);

        StreamData streamData = getStreamData(connection, stream);
        if (streamData == null) {
            streamData = new StreamData();
            stream.setProperty(key, streamData);
        }

        streamData.addHeaders(headers);

        if (endOfStream) {
            onDataRead(connection, stream, streamData);
            stream.closeLocalSide();
        }
    }

    protected StreamData getStreamData(Connection connection, Http2Stream stream) {
        return stream.getProperty(connection.getPropertyKey(DATA_CACHE_KEY));
    }

    /**
     * callback when receive end of stream.
     *
     * @param connection http2connection
     * @param stream     http2stream
     * @param streamData cached headers and data
     */
    public abstract void onDataRead(Connection connection, Http2Stream stream, StreamData streamData);
}