package com.aliyun.openservices.iot.api.http2.connection.impl;

import com.aliyun.openservices.iot.api.http2.connection.Connection;
import com.aliyun.openservices.iot.api.http2.connection.ConnectionListener;
import com.aliyun.openservices.iot.api.http2.connection.ConnectionManager;
import com.aliyun.openservices.iot.api.http2.connection.ConnectionStatus;
import com.aliyun.openservices.iot.api.http2.netty.NettyHttp2Handler;
import com.aliyun.openservices.iot.api.http2.netty.NettyHttp2HandlerBuilder;
import com.aliyun.openservices.iot.api.http2.netty.NettyHttp2Initializer;
import io.netty.bootstrap.Bootstrap;
import io.netty.channel.Channel;
import io.netty.channel.ChannelFutureListener;
import io.netty.channel.ChannelOption;
import io.netty.channel.EventLoopGroup;
import io.netty.channel.group.ChannelGroup;
import io.netty.channel.group.DefaultChannelGroup;
import io.netty.channel.nio.NioEventLoopGroup;
import io.netty.channel.socket.nio.NioSocketChannel;
import io.netty.handler.codec.http2.Http2Settings;
import io.netty.handler.timeout.IdleStateHandler;
import io.netty.util.concurrent.GlobalEventExecutor;
import lombok.extern.slf4j.Slf4j;

import java.net.SocketAddress;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author brhao
 * @date 23/03/2018
 */
@Slf4j
public class ConnectionManagerImpl implements ConnectionManager {
    private boolean enableSsl;
    private Bootstrap bootstrap;
    private ChannelGroup channelGroup;
    private EventLoopGroup workerGroup;
    private final long heartbeatInterval;
    private final long heartbeatTimeoutThreshold;
    private List<ConnectionListener> connectionListeners;

    private ConnectionListener innerListener = new ConnectionListener() {
        @Override
        public void onSettingReceive(Connection connection, Http2Settings settings) {
            connectionListeners.forEach(l -> l.onSettingReceive(connection, settings));
        }

        @Override
        public void onStatusChange(ConnectionStatus status, Connection connection) {
            connectionListeners.forEach(l -> l.onStatusChange(status, connection));
        }
    };

    public ConnectionManagerImpl(boolean enableSsl, long heartBeatInterval, long heartbeatTimeoutThreshold) {
        this.enableSsl = enableSsl;
        this.heartbeatInterval = heartBeatInterval;
        this.heartbeatTimeoutThreshold = heartbeatTimeoutThreshold;
        this.connectionListeners = new ArrayList<>();
        initNetty();
    }

    /**
     * initialize netty
     */
    private void initNetty() {
        log.info("[ConnectionManagerImpl]initialize netty");
        NettyHttp2Initializer initializer = createNettyHttp2Initializer();
        workerGroup = new NioEventLoopGroup();
        bootstrap = new Bootstrap();
        bootstrap.group(workerGroup);
        bootstrap.channel(NioSocketChannel.class);
        bootstrap.option(ChannelOption.SO_KEEPALIVE, true);
        bootstrap.handler(initializer);
        channelGroup = new DefaultChannelGroup(GlobalEventExecutor.INSTANCE);
    }

    private NettyHttp2Initializer createNettyHttp2Initializer() {
        NettyHttp2HandlerBuilder http2HandlerBuilder = new NettyHttp2HandlerBuilder(heartbeatTimeoutThreshold);
        NettyHttp2Initializer initializer = null;
        try {
            initializer = new NettyHttp2Initializer(http2HandlerBuilder, enableSsl);
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            log.error("failed to initialize netty, {}", e);
        }
        return initializer;
    }

    @Override
    public List<Connection> getConnectionList() {
        return channels().stream()
                .map(this::getConnection)
                .collect(Collectors.toList());
    }

    private Connection getConnection(Channel channel) {
        return channel.pipeline().get(NettyHttp2Handler.class).getConnection();
    }

    @Override
    public CompletableFuture<Connection> connect(SocketAddress address) {
        log.info("connecting to {}", address);
        CompletableFuture<Connection> cf = new CompletableFuture<>();
        bootstrap.connect(address).addListeners((ChannelFutureListener) future -> {
            if (future.isSuccess()) {
                Channel channel = future.channel();
                channelGroup.add(channel);
                channel.closeFuture().addListener((ChannelFutureListener) f -> getConnection(f.channel())
                        .onConnectionClosed());

                channel.pipeline().addFirst("heartBeatHandler", new IdleStateHandler(heartbeatInterval,
                        0, 0, TimeUnit.MILLISECONDS));

                Connection connection = getConnection(channel);
                connection.setConnectionListener(innerListener);
                cf.complete(connection);
            } else {
                cf.completeExceptionally(future.cause());
            }
        });
        return cf;
    }

    private List<Channel> channels() {
        return channelGroup.stream()
                .filter(Channel::isOpen)
                .filter(Channel::isActive)
                .collect(Collectors.toList());
    }

    @Override
    public void shutdown() {
        channelGroup.close();
        workerGroup.shutdownGracefully();
    }

    @Override
    public void addConnectionListener(ConnectionListener connectionListener) {
        this.connectionListeners.add(connectionListener);
    }

    @Override
    public void removeConnectionListener(ConnectionListener connectionListener) {
        this.connectionListeners.remove(connectionListener);
    }

}
