package com.aliyun.openservices.iot.api.http2.netty;

import com.aliyun.openservices.iot.api.http2.connection.Connection;
import com.aliyun.openservices.iot.api.http2.connection.ConnectionStatus;
import com.aliyun.openservices.iot.api.http2.connection.impl.ConnectionImpl;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.channel.ChannelHandlerContext;
import io.netty.handler.codec.http2.*;
import io.netty.handler.timeout.IdleState;
import io.netty.handler.timeout.IdleStateEvent;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;

/**
 * @author brhao
 * @date 19/03/2018
 */
@Slf4j
public class NettyHttp2Handler extends Http2ConnectionHandler implements Http2FrameListener {
    private long heartbeatTimeoutThreshold;
    private long lastHeartBeatTime;

    private Connection connection;

    NettyHttp2Handler(Http2ConnectionDecoder decoder,
                      Http2ConnectionEncoder encoder,
                      Http2Settings initialSettings, long heartbeatTimeoutThreshold) {
        super(decoder, encoder, initialSettings);
        this.heartbeatTimeoutThreshold = heartbeatTimeoutThreshold;
    }

    public Connection getConnection() {
        if (connection == null) {
            log.error("failed to get connection, netty handler not initialized correctly");
        }
        return connection;
    }

    @Override
    public void channelRead(ChannelHandlerContext ctx, Object msg) throws Exception {
        super.channelRead(ctx, msg);
        resetHeartBeatTime();
    }

    @Override
    public void handlerAdded(ChannelHandlerContext ctx) throws Exception {
        super.handlerAdded(ctx);
        connection = new ConnectionImpl(this, ctx);
        connection.setStatus(ConnectionStatus.CREATING);
    }

    @Override
    public int onDataRead(ChannelHandlerContext ctx, int streamId, ByteBuf data, int padding, boolean endOfStream)
            throws Http2Exception {
        log.debug("onDataRead, streamId: {}, size: {}, ES: {}", streamId, data.readableBytes(),
                endOfStream);
        return connection.onDataRead(ctx, streamId, data, padding, endOfStream);
    }

    @Override
    public void onHeadersRead(ChannelHandlerContext ctx, int streamId, Http2Headers headers, int padding,
                              boolean endOfStream) throws Http2Exception {
        onHeadersRead(ctx, streamId, headers, connection().connectionStream().id(), (short) 16, false, padding,
                endOfStream);
    }

    @Override
    public void onHeadersRead(ChannelHandlerContext ctx, int streamId, Http2Headers headers, int streamDependency,
                              short weight, boolean exclusive, int padding, boolean endOfStream) throws Http2Exception {
        log.debug("onHeadersRead, streamId: {}, header: {}, weight: {}, dependency: {}, exclusive: {}, isEnd: {}",
                streamId, headers, streamDependency, weight, exclusive, endOfStream);
        connection.onHeadersRead(ctx, streamId, headers, streamDependency, weight, exclusive, padding,
                endOfStream);
    }

    @Override
    public void onPriorityRead(ChannelHandlerContext ctx, int streamId, int streamDependency, short weight,
                               boolean exclusive) throws Http2Exception {
        log.debug("onPriorityRead, streamId: {}, streamDependency: {}, weight: {}, exclusive: {}",
                streamId, streamDependency, weight, exclusive);
    }

    @Override
    public void onRstStreamRead(ChannelHandlerContext ctx, int streamId, long errorCode) throws Http2Exception {
        log.debug("onRstStreamRead, streamId: {}, errorCode: {}", streamId, errorCode);
        connection.onRstStreamRead(ctx, streamId, errorCode);
    }

    @Override
    public void onSettingsAckRead(ChannelHandlerContext ctx) throws Http2Exception {
        log.debug("onSettingsAckRead");
        connection.setStatus(ConnectionStatus.CREATED);
    }

    @Override
    public void onError(ChannelHandlerContext ctx, boolean outbound, Throwable cause) {
        super.onError(ctx, outbound, cause);
        log.error("error occurs, close channel. channel id: {}, outbound: {}, error:", ctx.channel(), outbound,
                cause);
        connection.onError(ctx, outbound, cause);
    }

    @Override
    public void onSettingsRead(ChannelHandlerContext ctx, Http2Settings settings) throws Http2Exception {
        log.debug("onSettingsRead, settings: {}", settings.toString());
        connection.onSettingsRead(ctx, settings);
    }

    @Override
    public void onPingRead(ChannelHandlerContext ctx, long data) throws Http2Exception {
        log.debug("onPingRead, data: {}", data);
        encoder().frameWriter().writePing(ctx, true, data, ctx.voidPromise());
    }

    @Override
    public void onPingAckRead(ChannelHandlerContext ctx, long data) throws Http2Exception {
        log.debug("onPingAckRead, data: {}", data);
    }

    @Override
    public void onPushPromiseRead(ChannelHandlerContext ctx, int streamId, int promisedStreamId, Http2Headers headers,
                                  int padding) throws Http2Exception {
        log.debug("onPushPromiseRead, streamId: {}, promisedStreamId: {}, headers size: {}",
                streamId, promisedStreamId, headers.size());
    }

    @Override
    public void onGoAwayRead(ChannelHandlerContext ctx, int lastStreamId, long errorCode, ByteBuf debugData)
            throws Http2Exception {
        log.debug("onGoAwayRead, lastStreamId: {}, errorCode: {}, {}",
                lastStreamId, errorCode, new String(ByteBufUtil.getBytes(debugData)));
        connection.onGoAwayRead(ctx, lastStreamId, errorCode, debugData);
    }

    @Override
    public void onWindowUpdateRead(ChannelHandlerContext ctx, int streamId, int windowSizeIncrement)
            throws Http2Exception {
        log.debug("onWindowUpdateRead, streamId: {}, increment size: {}", streamId,
                windowSizeIncrement);
    }

    @Override
    public void onUnknownFrame(ChannelHandlerContext ctx, byte frameType, int streamId, Http2Flags flags,
                               ByteBuf payload) throws Http2Exception {
        log.debug("onUnknownFrame, frameType: {}, streamId: {}, size: {}, flags: {}",
                frameType, streamId, payload.readableBytes(), flags.toString());
        connection.onUnknownFrame(ctx, frameType, streamId, flags, payload);
    }

    @Override
    public void exceptionCaught(ChannelHandlerContext ctx, Throwable cause) throws Exception {
        super.exceptionCaught(ctx, cause);
        log.error("exceptionCaught: ", new Exception(cause));
        connection.onError(ctx, false, cause);
        connection.close();
    }

    @Override
    public void userEventTriggered(ChannelHandlerContext ctx, Object evt) throws Exception {
        if (evt instanceof IdleStateEvent) {
            IdleStateEvent event = (IdleStateEvent) evt;
            if (event.state() == IdleState.READER_IDLE) {
                if (isTimeout()) {
                    log.error("connection heartbeat timeout, channel:[{}], remote address:[{}] ",
                            ctx.channel().id(),
                            ctx.channel().remoteAddress());
                    throw new IOException("connection heartbeat timeout");
                }

                log.debug("send heartbeat, channel:[{}], remote address:[{}] ", ctx.channel().id(),
                        ctx.channel().remoteAddress());
                encoder().frameWriter().writePing(ctx, false, System.currentTimeMillis(), ctx.voidPromise());
                ctx.pipeline().flush();
            }
        }
    }

    private boolean isTimeout() {
        return System.currentTimeMillis() - lastHeartBeatTime > heartbeatTimeoutThreshold;
    }

    private void resetHeartBeatTime() {
        lastHeartBeatTime = System.currentTimeMillis();
    }
}
