import { unlerp } from '../core/utils.js'
import { PROPERTIES } from './properties.js'
import { ErrorLog } from '../core/ErrorLog.js'

/* global PIXI */

export class Entity {
  constructor () {
    this.defaultState = {
      x: 0,
      y: 0,
      scaleX: 1,
      scaleY: 1,
      zIndex: 0,
      alpha: 1,
      visible: false,
      rotation: 0,
      mask: -1
    }

    this.states = {}
  }

  init () {
    this.properties = Object.keys(this.defaultState)
    this.initDisplay()
    this.currentState = Object.assign({}, this.defaultState)
    if (typeof this.graphics === 'object') {
      this.container.addChild(this.graphics)
    }
  }

  setHidden (hide) {
    this.hide = hide
    this.container.visible = this.currentState.visible && !this.hide
  }

  addState (t, params, frame) {
    if (!this.states[frame]) {
      this.states[frame] = []
    }
    let state = Entity.createState(t, params.values, params.curve)

    const collision = this.states[frame].find(v => v.t === t)
    if (collision && Object.keys(state.curve).length === 0) {
      state.curve = collision.curve
    }
    if (collision) {
      if (!isStateEmpty(state) && !isStateEmpty(collision)) {
        ErrorLog.push(new Error('Different updates for same t ' + t))
      }
      Object.assign(collision, state)
    } else {
      this.states[frame].push(state)
    }
  }

  set (t, params, frame) {
    this.addState(t, params, frame)
  }

  render (progress, data, globalData) {
    let subframes = this.states[data.number]
    this.container.visible = false
    if (subframes && subframes.length) {
      let index = 0
      while (index < subframes.length - 1 && subframes[index].t < progress) {
        index++
      }
      let start = subframes[index - 1]
      let end = subframes[index]
      // This t is used to animate the interpolation
      let t

      if (!start) {
        // The start frame must be at the end of the previous turn
        var prev = this.states[data.previous.number] || []
        start = prev[prev.length - 1]

        // If it didn't exist on the previous turn, don't even animate it
        if (!start && progress >= end.t) {
          start = end
          t = 1
        } else {
          // Interpolate from zero since their is always a substate at t=1 no matter what
          t = unlerp(0, end.t, progress)
        }
      } else {
        t = unlerp(start.t, end.t, progress)
      }
      if (start) {
        const changed = {}
        const state = Object.assign({}, this.currentState)

        for (let property of this.properties) {
          const opts = PROPERTIES[property] || PROPERTIES.default
          const lerpMethod = opts.lerpMethod
          const curve = end.curve[property] || (a => a)
          const newValue = lerpMethod(start[property], end[property], curve(t))
          if (newValue !== this.currentState[property]) {
            changed[property] = true
            state[property] = newValue
          }
        }
        this.updateDisplay(state, changed, globalData, data, progress)
        Object.assign(this.currentState, state)
        this.container.visible = this.container._visible
        if (changed.children) {
          globalData.mustResetTree = true
          if (typeof this.postUpdate === 'function') {
            globalData.updatedBuffers.push(this)
          }
        }
        if (changed.zIndex) {
          globalData.mustResort = true
        }
        if (changed.mask) {
          globalData.maskUpdates[this.id] = state.mask
        }
        if (Object.keys(changed).length !== 0 && this.parent) {
          this.parent.notifyChange()
        }
      }
    } else {
      Object.assign(this.currentState, this.defaultState)
    }
  }

  notifyChange () {
    if (this.parent) {
      this.parent.notifyChange()
    }
  }

  initDisplay () {
    this.container = new PIXI.Container()
    this.container.zIndex = this.defaultState.zIndex
    this.container.id = this.id
    this.container._visible = this.defaultState.visible
  }

  updateDisplay (state, changed, globalData) {
    // We don't want to set the scale to exactly zero or PIXI may crash.
    const eps = 1e-8

    this.container.zIndex = state.zIndex
    this.container.alpha = state.alpha
    this.container.position.set(state.x * globalData.coeff, state.y * globalData.coeff)
    this.container.scale.set(state.scaleX || eps, state.scaleY || eps)
    this.container.rotation = state.rotation
    this.container._visible = state.visible && !this.hide
  }

  static createState (time = 1, values = {}, curve = {}) {
    return {
      t: time,
      ...values,
      curve: curve
    }
  }
}

function isStateEmpty (state) {
  return Object.keys(state).length === Object.keys(Entity.createState()).length
}
